/*
 *  Reset handlers.
 *
 * Copyright (c) 2003-2008 Fabrice Bellard
 * Copyright (c) 2016 Red Hat, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "qemu/osdep.h"
#include "qemu/queue.h"
#include "sysemu/reset.h"

/* reset/shutdown handler */

typedef struct QEMUResetEntry {
    QTAILQ_ENTRY(QEMUResetEntry) entry;
    QEMUResetHandler *func;
    void *opaque;
    bool skip_on_snapshot_load;
} QEMUResetEntry;

static QTAILQ_HEAD(, QEMUResetEntry) reset_handlers =
    QTAILQ_HEAD_INITIALIZER(reset_handlers);

static QTAILQ_HEAD(loader_handlers, QEMUResetEntry) loader_handlers =
    QTAILQ_HEAD_INITIALIZER(loader_handlers);

#define REGISTER_RESET_LIST(head, func, opaque)                 \
    {                                                           \
        QEMUResetEntry *re = g_malloc0(sizeof(QEMUResetEntry)); \
        re->func = func;                                        \
        re->opaque = opaque;                                    \
        QTAILQ_INSERT_TAIL(head, re, entry);                    \
    }

#define UNREGISTER_RESET_LIST(head, func, opaque)               \
    QEMUResetEntry *re;                                         \
    QTAILQ_FOREACH(re, head, entry) {                           \
        if (re->func == func && re->opaque == opaque) {         \
            QTAILQ_REMOVE(head, re, entry);                     \
            g_free(re);                                         \
            return;                                             \
        }                                                       \
    }

void qemu_register_reset(QEMUResetHandler *func, void *opaque)
{
    REGISTER_RESET_LIST(&reset_handlers, func, opaque);
}

void qemu_register_reset_nosnapshotload(QEMUResetHandler *func, void *opaque)
{
    QEMUResetEntry *re = g_new0(QEMUResetEntry, 1);

    re->func = func;
    re->opaque = opaque;
    re->skip_on_snapshot_load = true;
    QTAILQ_INSERT_TAIL(&reset_handlers, re, entry);
}

void qemu_unregister_reset(QEMUResetHandler *func, void *opaque)
{
    UNREGISTER_RESET_LIST(&reset_handlers, func, opaque);
}

void qemu_register_reset_loader(QEMUResetHandler *func, void *opaque)
{
    REGISTER_RESET_LIST(&loader_handlers, func, opaque);
}

void qemu_unregister_reset_loader(QEMUResetHandler *func, void *opaque)
{
    UNREGISTER_RESET_LIST(&loader_handlers, func, opaque);
}

void qemu_devices_reset(ShutdownCause reason)
{
    QEMUResetEntry *re, *nre;

    /* reset all devices */
    QTAILQ_FOREACH_SAFE(re, &reset_handlers, entry, nre) {
        if (reason == SHUTDOWN_CAUSE_SNAPSHOT_LOAD &&
            re->skip_on_snapshot_load) {
            continue;
        }
        if (re->func) {
            re->func(re->opaque);
        }
    }

    /* reset all device loaders */
    QTAILQ_FOREACH_SAFE(re, &loader_handlers, entry, nre) {
        if (re->func) {
            re->func(re->opaque);
        }
    }
}

