/******************************************************************************
* Copyright (c) 2024 - 2025 Advanced Micro Devices, Inc.  All rights reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

#ifndef XPM_REGS_H
#define XPM_REGS_H

#ifdef __cplusplus
extern "C" {
#endif

/* TBD: Placeholders for now, will be fixed in future by creating
		a range-checker api specific to architecture
*/

/**
 * Base Addresses
 */
#define PMC_SYSMON_BASEADDR				(0xF1270000U)

/**
 * PMC IOU SLCR Module
 */
#define PMC_IOU_SLCR_SD0_DLL_DIV_MAP0_OFFSET		(0x00000458U)
#define PMC_IOU_SLCR_SD0_DLL_DIV_MAP1_OFFSET		(0x0000045CU)
#define PMC_IOU_SLCR_SIZE				(0x00010000U)
#define PMC_IOU_SLCR_WPROT0_OFFSET			(0x00000828U)

/**
 * LPD IOU SLCR Module
 */
#define LPD_IOU_SLCR_BASEADDR				(0xF1A20000U)
#define LPD_IOU_SLCR_SIZE				(0x00020000U)
#define LPD_IOU_SLCR_WPROT0_OFFSET			(0x0000073CU)

/**
 * FPD_SLCR Module
 */
#define FPD_SLCR_BASEADDR				(0xEC8C0000U)
#define FPD_SLCR_SIZE					(0x00010000U)
#define FPD_SLCR_WPROT0_OFFSET                  	(0x00000000U)

/*
 * MMI_SLCR Module
 */
#define MMI_SLCR_BASEADDR				(0xEDC20000U)
#define MMI_SLCR_SIZE					(0x00010000U)
#define MMI_SLCR_WPROTS_OFFSET				(0x00000000U)
#define MMI_SLCR_WPROTP_OFFSET				(0x00000004U)

/*
 * UDH_PLL Module
 */
#define MMI_UDH_PLL_BASEADDR				(0xEDE90000U)
#define MMI_UDH_PLL_SIZE				(0x00010000U)
#define MMI_UDH_PLL_WPROTS_OFFSET			(0x00000030U)
#define MMI_UDH_PLL_WPROTP_OFFSET			(0x00000034U)

/*
 * MMI_CRX Module
 */
#define MMI_CRX_BASEADDR				(0xEDC00000U)
#define MMI_CRX_SIZE					(0x00010000U)
#define MMI_CRX_WPROTS_OFFSET				(0x00000030U)
#define MMI_CRX_WPROTP_OFFSET				(0x00000034U)
#define MMI_CRX_RST_DC_OFFSET				(0x00000318U)
#define MMI_CRX_RST_UDH_OFFSET				(0x00000320U)

#define MMI_CRX_RST_DC_SS_CFG_POR_MASK			(0x00000002U)
#define MMI_CRX_RST_UDH_SS_CFG_POR_MASK			(0x00001000U)


/*
 * MMI_DC Module
 */
#define MMI_DC_BASEADDR					(0xEDD08000U)
#define MMI_DC_SIZE					(0x00008000U)
#define MMI_DC_WPROTS_OFFSET				(0x00004C70U)

/*
 * MMI_UDH_SLCR Module
 */
#define MMI_UDH_SLCR_BASEADDR				(0xEDEA0000U)
#define MMI_UDH_SLCR_SIZE				(0x00008000U)
#define MMI_UDH_SLCR_WPROTS_OFFSET			(0x00000000U)
#define MMI_UDH_SLCR_WPROTP_OFFSET			(0x00000004U)

/*
 * AIE2PS Module
 */
#ifndef VIVADO_ME_BASEADDR
	#define VIVADO_ME_BASEADDR				(0x20000000000UL)
#endif

#define AIE2PS_NPI_ME_PROT_REG_CTRL_OFFSET			(0x00000200U)
#define ME_PROT_REG_CTRL_PROTECTED_REG_EN_MASK			(0x00000001U)
#define ME_NPI_REG_PCSR_MASK_ME_SHIM_RESET_MASK			(0x08000000U)

#define AIE2PS_CORE_MODULE_MEM_CTRL_OFFSET			(0x00036070U)
#define AIE2PS_CORE_MODULE_MEM_CTRL_MEM_ZEROISATION_MASK	(0x00000001U)
#define AIE2PS_MEM_MODULE_MEM_CTRL_OFFSET			(0x00016010U)
#define AIE2PS_MEM_MODULE_MEM_CTRL_MEM_ZEROISATION_MASK		(0x00000001U)
#define AIE2PS_MEM_TILE_MODULE_MEM_CTRL_OFFSET			(0x00096048U)
#define AIE2PS_MEM_TILE_MODULE_MEM_CTRL_MEM_ZEROISATION_MASK	(0x00000001U)

#define AIE2PS_UC_MODULE_MEM_ZERO_OFFSET			(0x000C0030U)

#define AIE2PS_PL_MODULE_COLUMN_CLK_CTRL_OFFSET			(0x0007FF20U)
#define AIE2PS_PL_MODULE_COLUMN_RST_CTRL_OFFSET			(0x0007FF28U)

#define AIE2PS_NOC_MODULE_ME_AXIMM_CONFIG_OFFSET		(0x00002100U)
#define ME_AXIMM_CONFIG_DECERR_BLOCK_EN_MASK			(0x00000008U)
#define ME_AXIMM_CONFIG_SLVERR_BLOCK_EN_MASK			(0x00000004U)

#define AIE2PS_PL_MODULE_COLUMN_CLK_CTRL_OFFSET			(0x0007FF20U)

#define AIE2PS_PL_MODULE_TILE_CTRL_AXI_MM_OFFSET		(0x0007FF44U)
#define TILE_CTRL_AXI_MM_ISOLATE_FROM_EAST_MASK			(0x00000008U)
#define TILE_CTRL_AXI_MM_ISOLATE_FROM_WEST_MASK			(0x00000002U)

#define AIE2PS_NOC_MODULE_INTR_CTRL_L2_INTR_OFFSET		(0x00001010U)

#define AIE2PS_PL_MODULE_INT_CTRL_HW_ERR_INT_OFFSET		(0x0007FF58U)

#define AIE2PS_PL_MODULE_INT_CTRL_HW_ERR_MASK_OFFSET		(0x0007FF50U)

#define AIE2PS_UC_MODULE_MEM_DM_ECC_SCRUB_PERIOD_OFFSET		(0x000C0038U)
#define MEM_DM_ECC_SCRUB_PERIOD_SCRUBBING_PERIOD_MASK		(0x0000001FU)

#define AIE2PS_UC_MODULE_MEMORY_PRIVILEGED_OFFSET		(0x000C0034U)
#define AIE2PS_UC_MODULE_MEM_PRIV_MEMORY_PRIVILEGED_MASK	(0x00000001U)

#define AIE2PS_NOC_MODULE_DMA_PAUSE_OFFSET			(0x00009348U)
#define AIE2PS_NOC_MODULE_DMA_PAUSE_MASK			(0x0000000FU)

#define AIE2PS_UC_MODULE_DMA_PAUSE_OFFSET			(0x000C0120U)
#define AIE2PS_UC_MODULE_DMA_PAUSE_MASK				(0x00000003U)

#define AIE2PS_PL_MODULE_INT_CTRL_HW_ERR_STATUS_OFFSET		(0x0007FF50U)

#define AIE2PS_MEM_TILE_MODULE_MEM_CTRL_OFFSET			(0x00096048U)
#define AIE2PS_MEM_TILE_MODULE_MEM_CTRL_MEM_INTERLEAVING_MASK	(0x00000002U)

#define AIE2PS_PL_MODULE_NMU_SWITCHES_CONFIG_OFFSET		(0x0007FF48U)

#define AIE2PS_UC_MODULE_CORE_PRIVATE_DATA_MEM			(0x00088000U)
#define AIE2PS_NPI_ME_PROT_REG_CTRL_WE_COL_ID_FIRST_MASK	(0x000000FEU)
#define AIE2PS_NPI_ME_PROT_REG_CTRL_WE_COL_ID_LAST_MASK		(0x00007F00U)
#define AIE2PS_NPI_ME_PROT_REG_CTRL_WE_COL_ID_FIRST_SHIFT	(1U)
#define AIE2PS_NPI_ME_PROT_REG_CTRL_WE_COL_ID_LAST_SHIFT	(8U)

#define AIE2PS_CORE_MODULE_MODULE_CLOCK_CONTROL_OFFSET		(0x00060000U)
#define AIE2PS_CORE_MODULE_CLK_CTRL_PKT_TLAST_ERR_EN_MASK	(0x00000010U)
#define AIE2PS_MEM_TILE_MODULE_MODULE_CLK_CTRL_OFFSET		(0x000FFF00U)
#define AIE2PS_MEM_TILE_MODULE_CLK_CTRL_PKT_TLAST_ERR_EN_MASK	(0x00000010U)
#define AIE2PS_PL_MODULE_MODULE_CLK_CTRL_OFFSET			(0x0007FF00U)
#define AIE2PS_PL_MODULE_CLK_CTRL_PKT_TLAST_ERR_EN_MASK		(0x00000010U)
#define AIE2PS_CTRL_PKT_TLAST_ERR_EN_SHIFT			(4U)

/**
 * Definitions required from pmc_tap.h
 */
#define PMC_TAP_VERSION_PLATFORM_SHIFT				(24U)
#define PMC_TAP_VERSION_PLATFORM_MASK				(0x0F000000U)
#define PMC_TAP_VERSION_PLATFORM_VERSION_SHIFT			(28U)
#define PMC_TAP_VERSION_PLATFORM_VERSION_MASK			(0xF0000000U)
#define PMC_TAP_IDCODE_SI_REV_MASK				(0xF0000000U)
#define PMC_TAP_IDCODE_SI_REV_SHIFT				(28U)

/**
 * SLR TYPES
 */
#define PMC_TAP_SLR_TYPE_OFFSET					(0x00000024U)
#define PMC_TAP_SLR_TYPE_WIDTH					(0x00000002U)
#define PMC_TAP_SLR_TYPE_MASK					(0x00000007U)

/**
 * LPD SLCR Module
 */
#define LPD_SLCR_WPROT0_OFFSET                  		(0x00000000U)

/**
 * Register: LPD_SLCR_SECURE
 */
#define LPD_SLCR_SECURE_WPROT0_OFFSET				(0x00000000U)

/**
 * Register: FPD_SLCR_SECURE
 */
#define FPD_SLCR_SECURE_WPROT0_OFFSET				(0x00000000U)

/**
 * Register: PMXC_GLOBAL
 */
#define PMXC_GLOBAL_BASEADDR					(0xF1110000U)

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU0_CORE0_MASK		(0x00010000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU0_CORE1_MASK		(0x00020000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU1_CORE0_MASK		(0x00100000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU1_CORE1_MASK		(0x00200000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU2_CORE0_MASK		(0x01000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU2_CORE1_MASK		(0x02000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU3_CORE0_MASK		(0x10000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_APU3_CORE1_MASK		(0x20000000U)

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUA_CORE0_MASK		(0x00000100U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUA_CORE1_MASK		(0x00000200U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUB_CORE0_MASK		(0x00000040U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUB_CORE1_MASK		(0x00000080U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUC_CORE0_MASK		(0x00000010U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUC_CORE1_MASK		(0x00000020U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUD_CORE0_MASK		(0x00000004U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUD_CORE1_MASK		(0x00000008U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUE_CORE0_MASK		(0x00000001U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE0_RPUE_CORE1_MASK		(0x00000002U)

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B0_I0_MASK		(0x00010000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B0_I1_MASK		(0x00020000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B0_I2_MASK		(0x00040000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B0_I3_MASK		(0x00080000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B1_I0_MASK		(0x00100000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B1_I1_MASK		(0x00200000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B1_I2_MASK		(0x00400000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B1_I3_MASK		(0x00800000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B2_I0_MASK		(0x01000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B2_I1_MASK		(0x02000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B2_I2_MASK		(0x04000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B2_I3_MASK		(0x08000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B3_I0_MASK		(0x10000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B3_I1_MASK		(0x20000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B3_I2_MASK		(0x40000000U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_OCM_B3_I3_MASK		(0x80000000U)

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_A0_MASK		(0x00000100U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_A1_MASK		(0x00000200U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_B0_MASK		(0x00000040U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_B1_MASK		(0x00000080U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_C0_MASK		(0x00000010U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_C1_MASK		(0x00000020U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_D0_MASK		(0x00000004U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_D1_MASK		(0x00000008U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_E0_MASK		(0x00000001U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE1_RPU_TCM_E1_MASK		(0x00000002U)

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE2_GEM0_MASK		(0x00000001U)
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE2_GEM1_MASK		(0x00000002U)

#define PMXC_GLOBAL_PMC_AUX_PWR_STATE_0				( (PMXC_GLOBAL_BASEADDR) + (u32)0x00006100U )
#define PMXC_GLOBAL_PMC_AUX_PWR_STATE_1				( (PMXC_GLOBAL_BASEADDR) + (u32)0x00006104U )
#define PMXC_GLOBAL_PMC_AUX_PWR_STATE_2				( (PMXC_GLOBAL_BASEADDR) + (u32)0x00006108U )

#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE_0			( (PMXC_GLOBAL_BASEADDR) + (u32)0x00012100U )
#define PMXC_GLOBAL_PMC_MSTR_PWR_STATE_1			( (PMXC_GLOBAL_BASEADDR) + (u32)0x00012104U )


#define PMC_GLOBAL_GGS4_OFFSET					(0x00000040U)
#define PMC_GLOBAL_PGGS3_OFFSET					(0x0000005CU)
#define PMC_GLOBAL_PERS_GLOB_GEN_STORAGE0			( ( PMC_GLOBAL_BASEADDR ) + 0x00000050U )

#define PWR_SUPPLY_STATUS_OFFSET				(0x10CU)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_RAM_MASK		(0x00000080U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_PL_MASK		(0x00000040U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCAUX_MASK		(0x00000020U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_SOC_MASK		(0x00000010U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_LPD_MASK		(0x00000008U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_FPD_MASK		(0x00000004U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCINT_PMC_MASK		(0x00000002U)
#define PMC_GLOBAL_PWR_SUPPLY_STATUS_VCCAUX_PMC_MASK		(0x00000001U)

/**
 * PMC Global GIC Proxy Module
 */
#define PMC_GLOBAL_GIC_PROXY_BASE_OFFSET			(0x30000U)
#define GIC_PROXY_GROUP_OFFSET(g)				(0x14U * (g))

#define PMC_GLOBAL_GICP_IRQ_ENABLE_OFFSET			(0x300A8U)
#define PMC_GLOBAL_GICP_IRQ_DISABLE_OFFSET			(0x300ACU)

/**
 * GIC Proxy register offsets
 */
#define GIC_PROXY_IRQ_STATUS_OFFSET				(0x0U)
#define GIC_PROXY_IRQ_MASK_OFFSET				(0x4U)
#define GIC_PROXY_IRQ_ENABLE_OFFSET				(0x8U)
#define GIC_PROXY_IRQ_DISABLE_OFFSET				(0xCU)
#define GIC_PROXY_ALL_MASK					(0xFFFFFFFFU)

/**
 * IPI register masks
 */
#define ASU_IPI_MASK						(0x00000001U)
#define PMC_IPI_MASK						(0x00000002U)
#define IPI_0_MASK						(0x00000004U)
#define IPI_1_MASK						(0x00000008U)
#define IPI_2_MASK						(0x00000010U)
#define IPI_3_MASK						(0x00000020U)
#define IPI_4_MASK						(0x00000040U)
#define IPI_5_MASK						(0x00000080U)
#define IPI_6_MASK						(0x00000200U)

/**
 * IPI registers
 */
#define IPI_PMC_IER						(IPI_BASEADDR + 0x20018U)
#define IPI_PMC_IDR						(IPI_BASEADDR + 0x2001CU)
#define IPI_ISR_OFFSET						(0x10U)

#define RPU_0_CFG_OFFSET					(0x00000000U)
#define RPU_1_CFG_OFFSET					(0x00000000U)

#define XPM_RPU_CLUSTER_LOCKSTEP_DISABLE 			(0x1U)
#define XPM_RPU_CLUSTER_LOCKSTEP_ENABLE 			(0x0U)
#define XPM_A78_CLUSTER_CONFIGURED 				(1U)
#define XPM_R52_CLUSTER_CONFIGURED 				(1U)

#define XPM_APU_CLUSTER_LOCKSTEP_DISABLE			(0U)
#define XPM_APU_CLUSTER_LOCKSTEP_ENABLE				(1U)

#define RPU_CLUSTER_CORE_CFG0_OFFSET				(0x0U)
#define RPU_CLUSTER_CORE_CFG0_REMAP_MASK			(0x00000020U)
#define RPU_CLUSTER_CORE_CFG0_CPU_HALT_MASK			(0x00000001U)

#define RPU_CLUSTER_CORE_VECTABLE_OFFSET			(0x10U)
#define RPU_CLUSTER_CORE_VECTABLE_MASK				(0xFFFFFFE0U)
#define XPM_RPU_CLUSTER_CFG_OFFSET				(0x00000000U)

/* ASU GLOBAL registers */
#define ASU_GLOBAL_BASEADDR					(0xEBF80000U)
#define ASU_GLOBAL_ASU_MB_SOFT_RST				(ASU_GLOBAL_BASEADDR + 0x10054U)
#define ASU_GLOBAL_ASU_MB_SOFT_RST_VAL_MASK			(0x1U)

/* PSX CRL registers*/
#define PSX_CRL_BASEADDR					(0xEB5E0000U)
#define PSX_CRL_SIZE						(0x00300000U)
#define PSX_CRL_WPROT_OFFSET					(0x0000001CU)

#define PSX_CRL_RST_RPU_A					( (PSX_CRL_BASEADDR) + (u32)0x00000310U )
#define PSX_CRL_RST_RPU_B					( (PSX_CRL_BASEADDR) + (u32)0x00000314U )
#define PSX_CRL_RST_RPU_C					( (PSX_CRL_BASEADDR) + (u32)0x00000318U )
#define PSX_CRL_RST_RPU_D					( (PSX_CRL_BASEADDR) + (u32)0x0000031CU )
#define PSX_CRL_RST_RPU_E					( (PSX_CRL_BASEADDR) + (u32)0x00000320U )
#define CRL_RST_RPU_ADDR					(PSX_CRL_BASEADDR + 0x310U)
#define CRL_ASU_MB_RST_MODE_OFFSET				(0x000003C4U)
#define CRL_ASU_MB_RST_WAKEUP_MASK				(0x4U)

#define PSXC_CRL_RPU_CLK_CTRL					( (PSX_CRL_BASEADDR) + (u32)0x0000010CU )
#define PSXC_CRL_RPU_CLK_CTRL_CLKACT_MASK			(0x02000000U)

#define PSXC_CRL_RST_RPU_TOP_RESET_MASK				(0x00010000U)
#define PSXC_CRL_RST_RPU_A_CORE0_RESET_MASK			(0x00000101U)
#define PSXC_CRL_RST_RPU_A_CORE1_RESET_MASK			(0x00000202U)
#define PSXC_CRL_RST_RPU_B_CORE0_RESET_MASK			(0x00000101U)
#define PSXC_CRL_RST_RPU_B_CORE1_RESET_MASK			(0x00000202U)
#define PSXC_CRL_RST_RPU_C_CORE0_RESET_MASK			(0x00000101U)
#define PSXC_CRL_RST_RPU_C_CORE1_RESET_MASK			(0x00000202U)
#define PSXC_CRL_RST_RPU_D_CORE0_RESET_MASK			(0x00000101U)
#define PSXC_CRL_RST_RPU_D_CORE1_RESET_MASK			(0x00000202U)
#define PSXC_CRL_RST_RPU_E_CORE0_RESET_MASK			(0x00000101U)
#define PSXC_CRL_RST_RPU_E_CORE1_RESET_MASK			(0x00000202U)

#define PSXC_CRL_GEM0_REF_CTRL					( (PSX_CRL_BASEADDR) + (u32)0x00000128U )
#define PSXC_CRL_GEM0_REF_CTRL_CLKACT_MASK			(0x02000000U)
#define PSXC_CRL_GEM1_REF_CTRL					( (PSX_CRL_BASEADDR) + (u32)0x0000012CU )
#define PSXC_CRL_GEM1_REF_CTRL_CLKACT_MASK			(0x02000000U)

#define PSXC_CRL_RST_GEM0					( (PSX_CRL_BASEADDR) + (u32)0x0000033CU )
#define PSXC_CRL_RST_GEM0_RESET_MASK				(0x00000001U)
#define PSXC_CRL_RST_GEM1					( (PSX_CRL_BASEADDR) + (u32)0x00000340U )
#define PSXC_CRL_RST_GEM1_RESET_MASK				(0x00000001U)

/*
 * CRF
 */
#define PSXC_CRF_BASEADDR					(0xEC200000U)
#define PSXC_CRF_SIZE						(0x100000)
#define PSXC_CRF_WPROT_OFFSET					(0x0000001CU)

#define PSXC_CRF_ACPU0_CLK_CTRL					( (PSXC_CRF_BASEADDR) + (u32)0x0000010CU )
#define PSXC_CRF_ACPU1_CLK_CTRL					( (PSXC_CRF_BASEADDR) + (u32)0x00000110U )
#define PSXC_CRF_ACPU2_CLK_CTRL					( (PSXC_CRF_BASEADDR) + (u32)0x00000114U )
#define PSXC_CRF_ACPU3_CLK_CTRL					( (PSXC_CRF_BASEADDR) + (u32)0x00000118U )

#define PSXC_CRF_ACPU_CTRL_CLKACT_MASK				(0x02000000U)

#define PSXC_CRF_RST_APU0					( (PSXC_CRF_BASEADDR) + (u32)0x00000300U )
#define PSXC_CRF_RST_APU1					( (PSXC_CRF_BASEADDR) + (u32)0x00000304U )
#define PSXC_CRF_RST_APU2					( (PSXC_CRF_BASEADDR) + (u32)0x00000308U )
#define PSXC_CRF_RST_APU3					( (PSXC_CRF_BASEADDR) + (u32)0x0000030CU )

#define PSXC_CRF_RST_APU_CORE0_WARM_RST_MASK			(0x00000011U)
#define PSXC_CRF_RST_APU_CORE1_WARM_RST_MASK			(0x00000022U)
#define PSX_CRF_RST_APU_WARM_RST_MASK				(0x00000030U)
#define ACPU_CLUSTER_COLD_WARM_RST_MASK				((u32)0x00000300U)

/* TODO: Check these values are correct */
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_DIS_RPU_SHIFT		(4U)
#define PSXC_GLOBAL_REG_PWR_CTRL1_IRQ_RPU_X_COREX_SHIFT		(4U)


#define RPU_A_TOPRESET_MASK					(0x00010000U)
#define RPU_A_DBGRST_MASK					(0x00100000U)
#define RPU_A_DCLS_TOPRESET_MASK				(0x00040000U)
#define RPU_CORE0A_POR_MASK					(0x00000100U)
#define RPU_CORE0A_RESET_MASK					(0x00000001U)

/* EFUSE_CACHE registers */
/* putting def guard because , xplmi_hw.h already defined this */
#ifndef EFUSE_CACHE_BASEADDR
#define EFUSE_CACHE_BASEADDR					(0xF1250000U)
#endif
#define EFUSE_CACHE_TRIM_BRAM_OFFSET				(0x00000098U)
#define EFUSE_CACHE_TRIM_URAM_OFFSET				(0x0000009CU)
#define EFUSE_CACHE_VID						(0x000000D4U)
#define EFUSE_CACHE_TRIM_CFRM_VGG_0_OFFSET			(0x000001B4U)
#define EFUSE_CACHE_TRIM_CFRM_VGG_1_OFFSET			(0x000001B8U)
#define EFUSE_CACHE_TRIM_CFRM_VGG_2_OFFSET			(0x000001BCU)
#define EFUSE_CACHE_TRIM_CRAM_OFFSET				(0x000001C0U)
#define EFUSE_CACHE_TBITS1_BISR_RSVD_OFFSET			(0x00000500U)
#define EFUSE_CACHE_BISR_RSVD_0_OFFSET				(0x00000504U)
#define EFUSE_CACHE_TBITS2_BISR_RSVD_OFFSET			(0x00000BFCU)
#define EFUSE_CACHE_TBITS1_BISR_RSVD    		( ( EFUSE_CACHE_BASEADDR ) + EFUSE_CACHE_TBITS1_BISR_RSVD_OFFSET)
#define EFUSE_CACHE_TBITS2_BISR_RSVD    		( ( EFUSE_CACHE_BASEADDR ) + EFUSE_CACHE_TBITS2_BISR_RSVD_OFFSET)

#define EFUSE_CACHE_TRIM_AMS_3_OFFSET				(0X00000200U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_SLOPE_5_0_SHIFT		(9U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_SLOPE_5_0_MASK		(0X00007E00U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_INT_OFFSET_5_0_SHIFT	(3U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_INT_OFFSET_5_0_MASK	(0X000001F8U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_DELTA_16_0_SHIFT		(15U)
#define EFUSE_CACHE_TRIM_AMS_3_TSENS_DELTA_16_0_MASK		(0Xffff8000U)
#define EFUSE_CACHE_TRIM_AMS_4_OFFSET				(0x00000190U)
#define EFUSE_CACHE_TRIM_AMS_11_OFFSET				(0X000001ACU)
#define EFUSE_CACHE_TRIM_AMS_11_IXPCM_PROCESS_15_0_SHIFT	(15U)
#define EFUSE_CACHE_TRIM_AMS_11_IXPCM_PROCESS_15_0_MASK		(0X7FFF8000U)
#define EFUSE_CACHE_TRIM_AMS_11_RES_PROCESS_0_SHIFT		(31U)
#define EFUSE_CACHE_TRIM_AMS_11_RES_PROCESS_0_MASK		(0X80000000U)
#define EFUSE_CACHE_TRIM_AMS_12_OFFSET				(0X000001B0U)
#define EFUSE_CACHE_TRIM_AMS_12_BJT_PROCESS_3_0_SHIFT		(6U)
#define EFUSE_CACHE_TRIM_AMS_12_BJT_PROCESS_3_0_MASK		(0X000003C0U)
#define EFUSE_CACHE_TRIM_AMS_12_RES_PROCESS_6_1_SHIFT		(0U)
#define EFUSE_CACHE_TRIM_AMS_12_RES_PROCESS_6_1_MASK		(0X0000003FU)
#define EFUSE_CACHE_TRIM_AMS_12_TSENS_EXT_OFFSET_5_0_SHIFT	(10U)
#define EFUSE_CACHE_TRIM_AMS_12_TSENS_EXT_OFFSET_5_0_MASK	(0X0000FC00U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_1_0_SHIFT		(16U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_1_0_MASK		(0X00030000U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_14_2_SHIFT		(18U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_14_2_MASK		(0X7ffc0000U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_2_SHIFT		(18U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_2_MASK		(0X00040000U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_6_SHIFT		(22U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_6_MASK		(0X00400000U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_4_3_SHIFT		(19U)
#define EFUSE_CACHE_TRIM_AMS_12_SHARED_SPARE_4_3_MASK		(0X00180000U)

#define EFUSE_CONFIG0_OFFSET					(0x180U)
#define EFUSE_CONFIG0_OFFSET_MASK				(0x3fU)
#define EFUSE_CONFIG0_OFFSET_SHIFT				(0U)
#define EFUSE_CONFIG0_SLOPE_MASK				(0xfc0U)
#define EFUSE_CONFIG0_SLOPE_SHIFT				(6U)
#define EFUSE_CONFIG0_PROCESS_MASK				(0xffff0000U)
#define EFUSE_CONFIG0_DELTA_SHIFT				(12U)
#define EFUSE_CONFIG0_DELTA_MASK				(0x0000f000U)
#define EFUSE_CONFIG0_PROCESS_SHIFT				(16U)
#define EFUSE_CONFIG1_OFFSET					(0x184U)
#define EFUSE_CONFIG1_RESISTOR_MASK				(0x7fU)
#define EFUSE_CONFIG1_RESISTOR_SHIFT				(0U)
#define EFUSE_CONFIG1_BJT_OFFSET_MASK				(0x780U)
#define EFUSE_CONFIG1_BJT_OFFSET_SHIFT				(7U)
#define EFUSE_CONFIG1_EXT_OFFSET_MASK				(0x1f800U)
#define EFUSE_CONFIG1_EXT_OFFSET_SHIFT				(11U)
#define EFUSE_CONFIG1_ANA_SPARE_MASK				(0x00060000U)
#define EFUSE_CONFIG1_ANA_SPARE_SHIFT				(17U)
#define EFUSE_CONFIG1_DIG_SPARE_MASK				(0xfff80000U)
#define EFUSE_CONFIG1_DIG_SPARE_SHIFT				(19U)
#define CAL_SM_BIP_TSENS_OFFSET					(0x144U)
#define CAL_SM_BIP_TSENS_BIP_MASK				(0x00000001U)
#define CAL_SM_BIP_TSENS_BIP_SHIFT				(0U)
#define CAL_SM_BIP_TSENS_TSENS_MASK				(0x00000002U)
#define CAL_SM_BIP_TSENS_TSENS_SHIFT				(1U)
#define TSENS_BIAS_CTRL_OFFSET					(0x17CU)
#define TSENS_BIAS_VAL_MASK					(0x00000003U)
#define TSENS_BIAS_VAL_SHIFT					(0U)

/* VID fields */
#define VID_MP_MAP_OFFSET			(0xFU)
#define VID_MP_MAP_MASK				(0x7FFFU)
#define VID_CNTRL_OFFSET			(0x1EU)
#define VID_CNTRL_MASK				(0x3U)

#define GetRpuRstMask(Mask, ClusterNum, CoreNum)  (Mask << ((2U * ClusterNum)\
						+ CoreNum))

#define XPM_RPU_CORE0	(0U)
#define XPM_RPU_CORE1	(1U)

/*Registers*/
/*
 * Definitions required for RPU_CLUSTER
 */
#define PSX_RPU_CLUSTER_A_CLUSTER_CFG			(0xEB580000U)
#define PSX_RPU_CLUSTER_B_CLUSTER_CFG			(0xEB590000U)
#define PSX_RPU_CLUSTER_C_CLUSTER_CFG			(0xEB5A0000U)
#define PSX_RPU_CLUSTER_D_CLUSTER_CFG			(0xEB5B0000U)
#define PSX_RPU_CLUSTER_E_CLUSTER_CFG			(0xEB5C0000U)

#define PSX_RPU_CLUSTER_A0_CORE0_CFG0			(0xEB588000U)
#define PSX_RPU_CLUSTER_A1_CORE1_CFG0			(0xEB58C000U)
#define PSX_RPU_CLUSTER_B0_CORE0_CFG0			(0xEB598000U)
#define PSX_RPU_CLUSTER_B1_CORE1_CFG0			(0xEB59C000U)
#define PSX_RPU_CLUSTER_C0_CORE0_CFG0			(0xEB5A8000U)
#define PSX_RPU_CLUSTER_C1_CORE1_CFG0			(0xEB5AC000U)
#define PSX_RPU_CLUSTER_D0_CORE0_CFG0			(0xEB5B8000U)
#define PSX_RPU_CLUSTER_D1_CORE1_CFG0			(0xEB5BC000U)
#define PSX_RPU_CLUSTER_E0_CORE0_CFG0			(0xEB5C8000U)
#define PSX_RPU_CLUSTER_E1_CORE1_CFG0			(0xEB588000U)

#define PSX_RPU_CLUSTER_A0_CORE_0_VECTABLE		(0xEB588008U)
#define PSX_RPU_CLUSTER_A1_CORE_1_VECTABLE		(0xEB58C008U)
#define PSX_RPU_CLUSTER_B0_CORE_0_VECTABLE		(0xEB598008U)
#define PSX_RPU_CLUSTER_B1_CORE_1_VECTABLE		(0xEB59C008U)
#define PSX_RPU_CLUSTER_C0_CORE_0_VECTABLE		(0xEB5A8008U)
#define PSX_RPU_CLUSTER_C1_CORE_1_VECTABLE		(0xEB5AC008U)
#define PSX_RPU_CLUSTER_D0_CORE_0_VECTABLE		(0xEB5B8008U)
#define PSX_RPU_CLUSTER_D1_CORE_1_VECTABLE		(0xEB5BC008U)
#define PSX_RPU_CLUSTER_E0_CORE_0_VECTABLE		(0xEB5C8008U)
#define PSX_RPU_CLUSTER_E1_CORE_1_VECTABLE		(0xEB5CC008U)

#define PSXC_RPU_CLUSTER_A0_CORE_0_PWRDWN		(0xEB588200U)
#define PSXC_RPU_CLUSTER_A1_CORE_1_PWRDWN		(0xEB58C200U)
#define PSXC_RPU_CLUSTER_B0_CORE_0_PWRDWN		(0xEB598200U)
#define PSXC_RPU_CLUSTER_B1_CORE_1_PWRDWN		(0xEB59C200U)
#define PSXC_RPU_CLUSTER_C0_CORE_0_PWRDWN		(0xEB5A8200U)
#define PSXC_RPU_CLUSTER_C1_CORE_1_PWRDWN		(0xEB5AC200U)
#define PSXC_RPU_CLUSTER_D0_CORE_0_PWRDWN		(0xEB5B8200U)
#define PSXC_RPU_CLUSTER_D1_CORE_1_PWRDWN		(0xEB5BC200U)
#define PSXC_RPU_CLUSTER_E0_CORE_0_PWRDWN		(0xEB5C8200U)
#define PSXC_RPU_CLUSTER_E1_CORE_1_PWRDWN		(0xEB5CC200U)

#define PSXC_RPU_CLUSTER_CORE_PWRDWN_MASK		(0x00000001U)

#define PSX_RPU_CLUSTER_CFG_SLSPLIT_MASK		(0x00000001U)

#define PSX_RPU_CLUSTER_CORE_VECTABLE_MASK		(0xFFFFFFE0U)

#define RPU_CLUSTER_OFFSET				(0x10000U)
#define XPM_RPU_CLUSTER_CORE_OFFSET			(0x4000U)

#define XPM_GET_RPU_CLUSTER_CORE_REG(ClusterNum, CoreNum, Offset) \
		GET_REGISTER_ADDR(RPU_CLUSTER_A0_ADDR, \
		(ClusterNum) * RPU_CLUSTER_OFFSET, Offset, \
		(CoreNum) * XPM_RPU_CLUSTER_CORE_OFFSET)

#define GET_RPU_CLUSTER_REG(ClusterNum, Offset) \
		GET_REGISTER_ADDR(RPU_CLUSTER_BASEADDR, \
		(ClusterNum) * RPU_CLUSTER_OFFSET, Offset, 0U)

/*
 * Definitions required for FPX_SLCR
 */
#define FPX_SLCR_APU_CTRL_OFFSET			(0x1000U)

/**
 * OSPI Mux select related macros
 */
#define XPM_OSPI_MUX_SEL_OFFSET				(0x00000504U)
#define XPM_OSPI_MUX_SEL_MASK				(0x2U)
#define XPM_OSPI_MUX_SEL_SHIFT				(0x1U)

/**
 * CRP_RESET_REASON
 */
#define LAST_RESET_REASON_REG				(CRP_BASEADDR + 0x224)
#define CRP_RESET_REASON_MASK				(0x0000FFFFU)
#define CRP_RESET_REASON_SLR_SYS_MASK			(0x00000400U)
#define CRP_RESET_REASON_SW_SYS_MASK			(0x00000200U)
#define CRP_RESET_REASON_ERR_SYS_MASK			(0x00000100U)
#define CRP_RESET_REASON_DAP_SYS_MASK			(0x00000080U)
#define CRP_RESET_REASON_ERR_POR_MASK			(0x00000008U)
#define CRP_RESET_REASON_SLR_POR_MASK			(0x00000004U)
#define CRP_RESET_REASON_SW_POR_MASK			(0x00000002U)
#define CRP_RESET_REASON_EXTERNAL_POR_MASK		(0x00000001U)

/*
* CRP
*/
#define CRP_SIZE					(0x00010000U)
#define CRP_WPROT_OFFSET				(0x0000001CU)

/**
 * CRP CFU Registers
 */
#define CRP_CFU_REF_CTRL_OFFSET				(0x00000108U)
#define CRP_CFU_REF_CTRL_DIVISOR0_SHIFT			(8U)
#define CRP_CFU_REF_CTRL_DIVISOR0_WIDTH			(10U)
#define CRP_CFU_REF_CTRL_DIVISOR0_MASK			(BITNMASK( \
								CRP_CFU_REF_CTRL_DIVISOR0_SHIFT, \
								CRP_CFU_REF_CTRL_DIVISOR0_WIDTH \
								) \
							)
/**
 * USB registers
 */
#define XPM_USB0_CUR_PWR_OFFSET				(0x00000600U)
#define XPM_USB0_PWR_REQ_OFFSET				(0x00000608U)
#define XPM_USB1_CUR_PWR_OFFSET				(0x00000650U)
#define XPM_USB1_PWR_REQ_OFFSET				(0x00000658U)

/*
 * Definitions required for APU_CLUSTER
 */
#define APU_CLUSTER0_RVBARADDR0L			((u32)0xECC00040U)
#define APU_CLUSTER0_RVBARADDR1L			((u32)0xECC00048U)
#define APU_CLUSTER1_RVBARADDR0L			((u32)0xECD00040U)
#define APU_CLUSTER1_RVBARADDR1L			((u32)0xECD00048U)
#define APU_CLUSTER2_RVBARADDR0L			((u32)0xECE00040U)
#define APU_CLUSTER2_RVBARADDR1L			((u32)0xECE00048U)
#define APU_CLUSTER3_RVBARADDR0L			((u32)0xECF00040U)
#define APU_CLUSTER3_RVBARADDR1L			((u32)0xECF00048U)

#define APU_CLUSTER_BASEADDR				(0xECC00000U)
#define APU_CLUSTER_OFFSET				(0x00100000U)
#define APU_CLUSTER2_OFFSET				(0x00E00000U)

#define APU_CLUSTER_RVBARADDR0L_OFFSET			(0x00000040U)
#define APU_CLUSTER_RVBARADDR0H_OFFSET			(0x00000044U)

/**
 * Register: APU_PCIL
 */
#define APU_PCIL_BASEADDR				((u32)0xECB10000U)

#define APU_PCIL_CORE_0_BASEADDR			( APU_PCIL_BASEADDR + 0x00000000U)
#define APU_PCIL_CORE_1_BASEADDR			( APU_PCIL_BASEADDR + 0x00000030U)
#define APU_PCIL_CORE_4_BASEADDR			( APU_PCIL_BASEADDR + 0x000000C0U)
#define APU_PCIL_CORE_5_BASEADDR			( APU_PCIL_BASEADDR + 0x000000F0U)
#define APU_PCIL_CORE_8_BASEADDR			( APU_PCIL_BASEADDR + 0x00000180U)
#define APU_PCIL_CORE_9_BASEADDR			( APU_PCIL_BASEADDR + 0x000001B0U)
#define APU_PCIL_CORE_12_BASEADDR			( APU_PCIL_BASEADDR + 0x00000240U)
#define APU_PCIL_CORE_13_BASEADDR			( APU_PCIL_BASEADDR + 0x00000270U)

#define APU_PCIL_CORE_0_PWRDWN				( APU_PCIL_BASEADDR + 0x00000000U)
#define APU_PCIL_CORE_1_PWRDWN				( APU_PCIL_BASEADDR + 0x00000030U)
#define APU_PCIL_CORE_4_PWRDWN				( APU_PCIL_BASEADDR + 0x000000C0U)
#define APU_PCIL_CORE_5_PWRDWN				( APU_PCIL_BASEADDR + 0x000000F0U)
#define APU_PCIL_CORE_8_PWRDWN				( APU_PCIL_BASEADDR + 0x00000180U)
#define APU_PCIL_CORE_9_PWRDWN				( APU_PCIL_BASEADDR + 0x000001B0U)
#define APU_PCIL_CORE_12_PWRDWN				( APU_PCIL_BASEADDR + 0x00000240U)
#define APU_PCIL_CORE_13_PWRDWN				( APU_PCIL_BASEADDR + 0x00000270U)

#define APU_PCIL_CORE_PWRDWN_MASK			((u32)0x00000001U)
#define APU_PCIL_CORE_PSTATE_OFFSET			((u32)0x00000008U)
#define APU_PCIL_CORE_PSTATE_MASK			((u32)0x0000003FU)
#define APU_PCIL_CORE_PSTATE_VAL			((u32)0x00000008U)
#define APU_PCIL_CORE_PREQ_OFFSET			((u32)0x00000004U)
#define APU_PCIL_CORE_PREQ_MASK				((u32)0x00000001U)
#define APU_PCIL_CORE_PACTIVE_OFFSET			((u32)0x0000000CU)
#define APU_PCIL_CORE_PACCEPT_MASK			((u32)0x01000000U)
#define APU_PCIL_CORE_PDENY_MASK			((u32)0x02000000U)
#define APU_PCIL_CORE_IDS_POWER_MASK			((u32)0x00000001U)
#define APU_PCIL_CORE_IDS_POWER_OFFSET			((u32)0x0000001CU)
#define APU_PCIL_CORE_ISR_POWER_OFFSET			((u32)0x00000010U)
#define APU_PCIL_CORE_IEN_POWER_OFFSET			((u32)0x00000018U)
#define APU_PCIL_CORE_ISR_WAKE_OFFSET			((u32)0x00000020U)
#define APU_PCIL_CORE_IEN_WAKE_MASK			((u32)0x00000001U)
#define APU_PCIL_CORE_IEN_WAKE_OFFSET			((u32)0x00000028U)
#define APU_PCIL_CORE_IDS_WAKE_OFFSET			((u32)0x0000002CU)

#define APU_PCIL_CLUSTER_0_BASEADDR			( APU_PCIL_BASEADDR + 0x00008004U)
#define APU_PCIL_CLUSTER_1_BASEADDR			( APU_PCIL_BASEADDR + 0x00009004U)
#define APU_PCIL_CLUSTER_2_BASEADDR			( APU_PCIL_BASEADDR + 0x0000A004U)
#define APU_PCIL_CLUSTER_3_BASEADDR			( APU_PCIL_BASEADDR + 0x0000B004U)

#define APU_PCIL_CLUSTER_0_PACTIVE			( APU_PCIL_BASEADDR + 0x0000800CU)
#define APU_PCIL_CLUSTER_1_PACTIVE			( APU_PCIL_BASEADDR + 0x0000900CU)
#define APU_PCIL_CLUSTER_2_PACTIVE			( APU_PCIL_BASEADDR + 0x0000A00CU)
#define APU_PCIL_CLUSTER_3_PACTIVE			( APU_PCIL_BASEADDR + 0x0000B00CU )

#define APU_PCIL_CLUSTER_0_PSTATE			( APU_PCIL_BASEADDR + 0x00008008U )
#define APU_PCIL_CLUSTER_1_PSTATE			( APU_PCIL_BASEADDR + 0x00009008U )
#define APU_PCIL_CLUSTER_2_PSTATE			( APU_PCIL_BASEADDR + 0x0000A008U )
#define APU_PCIL_CLUSTER_3_PSTATE			( APU_PCIL_BASEADDR + 0x0000B008U )

#define APU_PCIL_CLUSTER_PREQ_OFFSET			(0x00000000U)
#define APU_PCIL_CLUSTER_PREQ_MASK			(0x00000001U)
#define APU_PCIL_CLUSTER_PSTATE_OFFSET			(0x00000004U)
#define APU_PCIL_CLUSTER_PSTATE_MASK			(0x0000007FU)
#define APU_PCIL_CLUSTER_PSTATE_VAL			(0x00000048U)
#define APU_PCIL_CLUSTER_PACTIVE_OFFSET			(0x00000008U)
#define APU_PCIL_CLUSTER_PACCEPT_MASK			(0x01000000U)
#define APU_PCIL_CLUSTER_ISR_POWER_OFFSET		((u32)0x0000000CU)
#define APU_PCIL_CLUSTER_ISR_WAKE_OFFSET		((u32)0x0000001CU)

#define PSX_APU_CLUSTER_RVBARADDR0L_MASK		(0xFFFFFFFCU)

#define XPM_R52_0A_TCMA_BASE_ADDR			(0xEBA00000U)
#define XPM_R52_1A_TCMA_BASE_ADDR			(0xEBA40000U)
#define XPM_R52_0B_TCMA_BASE_ADDR			(0xEBA80000U)
#define XPM_R52_1B_TCMA_BASE_ADDR			(0xEBAC0000U)
#define XPM_R52_TCM_CLUSTER_OFFSET			(0x00080000U)

/*
 * TCM address for R52
 */
#define XPM_R52_TCMA_LOAD_ADDRESS			(0x0U)
#define XPM_R52_TCM_TOTAL_LENGTH			(0x30000U)


#define XPM_R52_0_TCMA_ECC_DONE				(0x00000001U)
#define XPM_R52_1_TCMA_ECC_DONE				(0x00000002U)

/**
 * AFI port control registers
 */
#define AFI_FM_PORT_EN_MASK				(0x80000000U)
#define AFI_FM_WR_CTRL_OFFSET				(0x00000020U)
#define AFI_FS_PORT_EN_MASK				(0x00000001U)
#define CHI_FPD_PORT_EN_MASK				(0x00000001U)
#define ACP_APU0_PORT_EN_MASK				(0x00000020U)
#define PS_DTI0_PORT_EN_MASK				(0x00000004U)

/**
 * CRP_RST_NONPS
 */
#define CRP_RST_NONPS_NOC_RESET_MASK			(0x00000040U)
#define CRP_RST_NONPS_NOC_POR_MASK			(0x00000020U)

#define CRP_RST_NONPS_SYS_RST_1_MASK			(0x00000004U)
#define CRP_RST_NONPS_SYS_RST_2_MASK			(0x00000002U)
#define CRP_RST_NONPS_SYS_RST_3_MASK			(0x00000001U)

#define CRP_RST_PS_PL_POR_MASK				(0x00000020U)
#define CRP_RST_PS_PS_POR_MASK				(0x00000040U)
#define CRP_RST_PS_PL_SRST_MASK				(0x00000002U)
/*
 * Definitions required for NPI
 */
#define NPI_PCSR_UNLOCK_VAL				(0xF9E8D7C6U)
#define NPI_PCSR_LOCK_OFFSET				(0x0000000CU)
#define NPI_PCSR_MASK_OFFSET				(0x00000000U)
#define NPI_PCSR_CONTROL_OFFSET				(0x00000004U)
#define NPI_PCSR_STATUS_OFFSET				(0x00000008U)

/*
 * GTY PCSR Module
 */
#define GTY_PCSR_MASK_OFFSET				(0x00000000U)
#define GTY_PCSR_CONTROL_OFFSET				(0x00000004U)
#define GTY_PCSR_STATUS_OFFSET				(0x00000008U)
#define GTY_PCSR_BISR_TRIGGER_MASK			(0x20000000U)
#define GTY_PCSR_STATUS_BISR_DONE_MASK			(0x00080000U)
#define GTY_PCSR_STATUS_BISR_PASS_MASK			(0x00100000U)

/**
 * PMC Analog Registers
 */
#define PMC_ANLG_CFG_POR_CNT_SKIP_OFFSET		(0x00000030U)
#define PMC_ANLG_CFG_POR_CNT_SKIP_OFFSET_VAL_MASK	(0x00000001U)

/**
 * SysMon Registers
 */
#define PMC_SYSMON_SUPPLY0				PMC_SYSMON_BASEADDR + (0x00001040U)
#define PMC_SYSMON_SUPPLY0_TH_LOWER			PMC_SYSMON_BASEADDR + (0x00001980U)
#define PMC_SYSMON_NEW_DATA_FLAG0			PMC_SYSMON_BASEADDR + (0x00001000U)

/**
 * Register: PSXC_LPX_SLCR
 */
#define PSXC_LPX_SLCR_BASEADDR					(0xEB410000U)
#define PSXC_LPX_SLCR_SIZE					(0x00100000U)

/* WAKEUP0_IRQ registers for APU wakeup */
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050600U )
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_MASK				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050604U )
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050608U )
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_DIS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005060CU )
/* WAKEUP1_IRQ registers for RPU wakeup */
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050614U )
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_MASK				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050618U )
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005061CU )
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_DIS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050620U )

#define PSXC_LPX_SLCR_REQ_SWRST_INT_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060008U )
#define PSXC_LPX_SLCR_REQ_SWRST_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060000U )
#define PSXC_LPX_SLCR_REQ_PWRUP0_INT_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060088U )
#define PSXC_LPX_SLCR_REQ_PWRUP0_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0006008CU )
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060080U )
#define PSXC_LPX_SLCR_REQ_PWRUP1_INT_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600A8U )
#define PSXC_LPX_SLCR_REQ_PWRUP1_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600ACU )
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600A0U )
#define PSXC_LPX_SLCR_REQ_PWRUP2_INT_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600C8U )

/* REQ_PWRUP0 registers for APU  power up request */
#define PSXC_LPX_SLCR_REQ_PWRUP0_INT_EN    			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060088U )
#define PSXC_LPX_SLCR_REQ_PWRUP0_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0006008CU )
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060080U )
#define PSXC_LPX_SLCR_REQ_PWRUP0_INT_MASK			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060084U )
/* REQ_PWRUP1 registers for RPU power up request */
#define PSXC_LPX_SLCR_REQ_PWRUP1_INT_EN    			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600A8U )
#define PSXC_LPX_SLCR_REQ_PWRUP1_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600ACU )
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600A0U )
#define PSXC_LPX_SLCR_REQ_PWRUP1_INT_MASK			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600A4U )
/* REQ_PWRUP2 registers for OCM power up request */
#define PSXC_LPX_SLCR_REQ_PWRUP2_INT_EN    			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600C8U )
#define PSXC_LPX_SLCR_REQ_PWRUP2_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600CCU )
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000600C0U )

/* REQ_PWRDWN0 registers for APU power down request */
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060020U )
#define PSXC_LPX_SLCR_REQ_PWRDWN0_INT_MASK			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060024U )
#define PSXC_LPX_SLCR_REQ_PWRDWN0_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0006002CU )
/* REQ_PWRDWN1 registers for RPU & TCM power down request */
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060040U )
#define PSXC_LPX_SLCR_REQ_PWRDWN1_INT_MASK			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060044U )
#define PSXC_LPX_SLCR_REQ_PWRDWN1_INT_EN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060048U )
#define PSXC_LPX_SLCR_REQ_PWRDWN1_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0006004CU )
/* REQ_PWRDWN2 registers for OCM power down request */
#define PSXC_LPX_SLCR_REQ_PWRDWN2_INT_EN    			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060068U )
#define PSXC_LPX_SLCR_REQ_PWRDWN2_INT_DIS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0006006CU )
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00060060U )

#define PSXC_LPX_SLCR_POWER_DWN_IRQ_EN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050638U )
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_DIS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005063CU )
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050630U )
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_MASK			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050634U )

#define PSXC_LPX_SLCR_PWR_RST_IRQ_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050650U )
#define PSXC_LPX_SLCR_PWR_RST_IRQ_MASK				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050654U )
#define PSXC_LPX_SLCR_PWR_RST_IRQ_DIS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005065CU )

#define PSXC_LPX_SLCR_APU0_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050000U )
#define PSXC_LPX_SLCR_APU0_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050020U )
#define PSXC_LPX_SLCR_APU1_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050040U )
#define PSXC_LPX_SLCR_APU1_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050060U )
#define PSXC_LPX_SLCR_APU2_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050080U )
#define PSXC_LPX_SLCR_APU2_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500A0U )
#define PSXC_LPX_SLCR_APU3_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500C0U )
#define PSXC_LPX_SLCR_APU3_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500E0U )

#define PSXC_LPX_SLCR_APU_CORE_PWR_CNTRL_WPROT_OFFSET		((u32)0x00000010U)

#define PSXC_LPX_SLCR_APU0_CORE0_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005000CU )
#define PSXC_LPX_SLCR_APU0_CORE1_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005002CU )
#define PSXC_LPX_SLCR_APU1_CORE0_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005004CU )
#define PSXC_LPX_SLCR_APU1_CORE1_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005006CU )
#define PSXC_LPX_SLCR_APU2_CORE0_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005008CU )
#define PSXC_LPX_SLCR_APU2_CORE1_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500ACU )
#define PSXC_LPX_SLCR_APU3_CORE0_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500CCU )
#define PSXC_LPX_SLCR_APU3_CORE1_PWR_CNTRL_STS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000500ECU )

#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU0_CORE0_MASK		(0x000000001U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU0_CORE1_MASK		(0x000000002U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU1_CORE0_MASK		(0x000000004U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU1_CORE1_MASK		(0x000000008U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU2_CORE0_MASK		(0x000000010U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU2_CORE1_MASK		(0x000000020U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU3_CORE0_MASK		(0x000000040U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_APU3_CORE1_MASK		(0x000000080U)

#define PSXC_LPX_SLCR_RPU_PCIL_A0_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010000U )
#define PSXC_LPX_SLCR_RPU_PCIL_A1_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010100U )
#define PSXC_LPX_SLCR_RPU_PCIL_B0_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011000U )
#define PSXC_LPX_SLCR_RPU_PCIL_B1_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011100U )
#define PSXC_LPX_SLCR_RPU_PCIL_C0_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111C4U )
#define PSXC_LPX_SLCR_RPU_PCIL_C1_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111E4U )
#define PSXC_LPX_SLCR_RPU_PCIL_D0_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011204U )
#define PSXC_LPX_SLCR_RPU_PCIL_D1_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011224U )
#define PSXC_LPX_SLCR_RPU_PCIL_E0_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011244U )
#define PSXC_LPX_SLCR_RPU_PCIL_E1_BASEADDR			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011264U )

#define PSXC_LPX_SLCR_RPU_PCIL_A0_ISR_OFFSET			((u32)0x00010000U)
#define PSXC_LPX_SLCR_RPU_PCIL_A1_ISR_OFFSET			((u32)0x00010100U)
#define PSXC_LPX_SLCR_RPU_PCIL_B0_ISR_OFFSET			((u32)0x00011000U)
#define PSXC_LPX_SLCR_RPU_PCIL_B1_ISR_OFFSET			((u32)0x00011100U)
#define PSXC_LPX_SLCR_RPU_PCIL_C0_ISR_OFFSET			((u32)0x000111C4U)
#define PSXC_LPX_SLCR_RPU_PCIL_C1_ISR_OFFSET			((u32)0x000111E4U)
#define PSXC_LPX_SLCR_RPU_PCIL_D0_ISR_OFFSET			((u32)0x00011204U)
#define PSXC_LPX_SLCR_RPU_PCIL_D1_ISR_OFFSET			((u32)0x00011224U)
#define PSXC_LPX_SLCR_RPU_PCIL_E0_ISR_OFFSET			((u32)0x00011244U)
#define PSXC_LPX_SLCR_RPU_PCIL_E1_ISR_OFFSET			((u32)0x00011264U)

#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUA_CORE0_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUA_CORE1_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUB_CORE0_MASK		(0x00000004U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUB_CORE1_MASK		(0x00000008U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUC_CORE0_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUC_CORE1_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUD_CORE0_MASK		(0x00000040U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUD_CORE1_MASK		(0x00000080U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUE_CORE0_MASK		(0x00000100U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_RPUE_CORE1_MASK		(0x00000200U)

#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_A0_MASK			(0x00000001U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_A1_MASK			(0x00000002U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_B0_MASK			(0x00000004U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_B1_MASK			(0x00000008U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_C0_MASK			(0x00000010U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_C1_MASK			(0x00000020U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_D0_MASK			(0x00000040U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_D1_MASK			(0x00000080U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_E0_MASK			(0x00000100U)
#define PSXC_LPX_SLCR_RPU_CACHE_CNTRL_E1_MASK			(0x00000200U)

#define PSXC_LPX_SLCR_RPU_PCIL_ISR_OFFSET			(0x00000000U)
#define PSXC_LPX_SLCR_RPU_PCIL_IDS_OFFSET			(0x0000000CU)
#define PSXC_LPX_SLCR_RPU_PCIL_IEN_OFFSET			(0x00000008U)
#define PSXC_LPX_SLCR_RPU_PCIL_PR_OFFSET			(0x00000080U)
#define PSXC_LPX_SLCR_RPU_PCIL_PS_OFFSET			(0x00000084U)
#define PSXC_LPX_SLCR_RPU_PCIL_PA_OFFSET			(0x00000088U)
#define PSXC_LPX_SLCR_RPU_PCIL_PWRDWN_OFFSET			(0x000000C0U)
#define PSXC_LPX_SLCR_RPU_PCIL_PWRDWN_EN_MASK			(0x00000001U)

#define LPD_SLCR_RPU_PCIL_A0_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010084U )
#define LPD_SLCR_RPU_PCIL_A1_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010184U )
#define LPD_SLCR_RPU_PCIL_B0_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011084U )
#define LPD_SLCR_RPU_PCIL_B1_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011184U )
#define LPD_SLCR_RPU_PCIL_C0_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111D8U )
#define LPD_SLCR_RPU_PCIL_C1_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111F8U )
#define LPD_SLCR_RPU_PCIL_D0_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011218U )
#define LPD_SLCR_RPU_PCIL_D1_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011238U )
#define LPD_SLCR_RPU_PCIL_E0_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011258U )
#define LPD_SLCR_RPU_PCIL_E1_PS					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011278U )

#define LPD_SLCR_RPU_PCIL_A0_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010080U )
#define LPD_SLCR_RPU_PCIL_A1_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010180U )
#define LPD_SLCR_RPU_PCIL_B0_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011080U )
#define LPD_SLCR_RPU_PCIL_B1_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011180U )
#define LPD_SLCR_RPU_PCIL_C0_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111D4U )
#define LPD_SLCR_RPU_PCIL_C1_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111F4U )
#define LPD_SLCR_RPU_PCIL_D0_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011214U )
#define LPD_SLCR_RPU_PCIL_D1_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011234U )
#define LPD_SLCR_RPU_PCIL_E0_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011254U )
#define LPD_SLCR_RPU_PCIL_E1_PR					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011274U )

#define LPD_SLCR_RPU_PCIL_A0_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010088U )
#define LPD_SLCR_RPU_PCIL_A1_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010188U )
#define LPD_SLCR_RPU_PCIL_B0_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011088U )
#define LPD_SLCR_RPU_PCIL_B1_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011188U )
#define LPD_SLCR_RPU_PCIL_C0_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111DCU )
#define LPD_SLCR_RPU_PCIL_C1_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111FCU )
#define LPD_SLCR_RPU_PCIL_D0_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001121CU )
#define LPD_SLCR_RPU_PCIL_D1_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001123CU )
#define LPD_SLCR_RPU_PCIL_E0_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001125CU )
#define LPD_SLCR_RPU_PCIL_E1_PA					( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001127CU )

#define LPD_SLCR_RPU_PCIL_A0_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010000U )
#define LPD_SLCR_RPU_PCIL_A1_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010100U )
#define LPD_SLCR_RPU_PCIL_B0_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011000U )
#define LPD_SLCR_RPU_PCIL_B1_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011100U )
#define LPD_SLCR_RPU_PCIL_C0_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111C4U )
#define LPD_SLCR_RPU_PCIL_C1_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111E4U )
#define LPD_SLCR_RPU_PCIL_D0_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011204U )
#define LPD_SLCR_RPU_PCIL_D1_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011224U )
#define LPD_SLCR_RPU_PCIL_E0_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011244U )
#define LPD_SLCR_RPU_PCIL_E1_ISR				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011264U )

#define PSXC_LPX_SLCR_RPU_PCIL_A0_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010008U )
#define PSXC_LPX_SLCR_RPU_PCIL_A1_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00010108U )
#define PSXC_LPX_SLCR_RPU_PCIL_B0_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011008U )
#define PSXC_LPX_SLCR_RPU_PCIL_B1_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011108U )
#define PSXC_LPX_SLCR_RPU_PCIL_C0_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111CCU )
#define PSXC_LPX_SLCR_RPU_PCIL_C1_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111ECU )
#define PSXC_LPX_SLCR_RPU_PCIL_D0_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001120CU )
#define PSXC_LPX_SLCR_RPU_PCIL_D1_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001122CU )
#define PSXC_LPX_SLCR_RPU_PCIL_E0_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001124CU )
#define PSXC_LPX_SLCR_RPU_PCIL_E1_IEN				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001126CU )

#define PSXC_LPX_SLCR_RPU_PCIL_A0_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001000CU )
#define PSXC_LPX_SLCR_RPU_PCIL_A1_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001010CU )
#define PSXC_LPX_SLCR_RPU_PCIL_B0_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001100CU )
#define PSXC_LPX_SLCR_RPU_PCIL_B1_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0001110CU )
#define PSXC_LPX_SLCR_RPU_PCIL_C0_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111D0U )
#define PSXC_LPX_SLCR_RPU_PCIL_C1_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111F0U )
#define PSXC_LPX_SLCR_RPU_PCIL_D0_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011210U )
#define PSXC_LPX_SLCR_RPU_PCIL_D1_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011230U )
#define PSXC_LPX_SLCR_RPU_PCIL_E0_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011250U )
#define PSXC_LPX_SLCR_RPU_PCIL_E1_IDS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011270U )

#define PSXC_LPX_SLCR_RPU_PCIL_A0_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000100C0U )
#define PSXC_LPX_SLCR_RPU_PCIL_A1_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000101C0U )
#define PSXC_LPX_SLCR_RPU_PCIL_B0_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000110C0U )
#define PSXC_LPX_SLCR_RPU_PCIL_B1_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111C0U )
#define PSXC_LPX_SLCR_RPU_PCIL_C0_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000111E0U )
#define PSXC_LPX_SLCR_RPU_PCIL_C1_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011200U )
#define PSXC_LPX_SLCR_RPU_PCIL_D0_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011220U )
#define PSXC_LPX_SLCR_RPU_PCIL_D1_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011240U )
#define PSXC_LPX_SLCR_RPU_PCIL_E0_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011260U )
#define PSXC_LPX_SLCR_RPU_PCIL_E1_PWRDWN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00011280U )

#define PSXC_LPX_SLCR_RPU_PCIL_PS_PSTATE_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_RPU_PCIL_PR_PREQ_MASK			((u32)0x00000001U)
#define LPD_SLCR_RPU_PCIL_PA_PACTIVE_MASK			((u32)0x00000003U)
#define PSXC_LPX_SLCR_RPU_PCIL_PA_PACCEPT_MASK			((u32)0x00000100U)
#define PSXC_LPX_SLCR_RPU_PCIL_ISR_PACTIVE1_MASK		((u32)0x00000001U)
#define PSXC_LPX_SLCR_APU_CORE_PWR_CNTRL_STS_OFFSET		(0x0000000CU)
#define PSXC_LPX_SLCR_APU_CORE_PWR_CNTRL_PWRUP_GATES_MASK	((u32)0x00000001U)
#define PSXC_LPX_SLCR_APU_CORE_PWR_CNTRL_STS_ISOLATION_MASK	((u32)0x00000100U)
#define PSXC_LPX_SLCR_APU_CORE_PWR_CNTRL_STS_VALUE		((u32)0x000009FFU)

#define PSXC_LPX_SLCR_RPU_A0_POWER_DWN_MASK			(0x00010000U)
#define PSXC_LPX_SLCR_RPU_A1_POWER_DWN_MASK			(0x00020000U)
#define PSXC_LPX_SLCR_RPU_B0_POWER_DWN_MASK			(0x00040000U)
#define PSXC_LPX_SLCR_RPU_B1_POWER_DWN_MASK			(0x00080000U)
#define PSXC_LPX_SLCR_RPU_C0_POWER_DWN_MASK			(0x00100000U)
#define PSXC_LPX_SLCR_RPU_C1_POWER_DWN_MASK			(0x00200000U)
#define PSXC_LPX_SLCR_RPU_D0_POWER_DWN_MASK			(0x00400000U)
#define PSXC_LPX_SLCR_RPU_D1_POWER_DWN_MASK			(0x00800000U)
#define PSXC_LPX_SLCR_RPU_E0_POWER_DWN_MASK			(0x01000000U)
#define PSXC_LPX_SLCR_RPU_E1_POWER_DWN_MASK			(0x02000000U)

#define PSXC_LPX_SLCR_APU0_CORE0_PWRDWN_MASK			(0x00000001U)
#define PSXC_LPX_SLCR_APU0_CORE1_PWRDWN_MASK			(0x00000002U)
#define PSXC_LPX_SLCR_APU1_CORE0_PWRDWN_MASK			(0x00000004U)
#define PSXC_LPX_SLCR_APU1_CORE1_PWRDWN_MASK			(0x00000008U)
#define PSXC_LPX_SLCR_APU2_CORE0_PWRDWN_MASK			(0x00000010U)
#define PSXC_LPX_SLCR_APU2_CORE1_PWRDWN_MASK			(0x00000020U)
#define PSXC_LPX_SLCR_APU3_CORE0_PWRDWN_MASK			(0x00000040U)
#define PSXC_LPX_SLCR_APU3_CORE1_PWRDWN_MASK			(0x00000080U)

#define PSXC_LPX_SLCR_RPU0_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050100U )
#define PSXC_LPX_SLCR_RPU0_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050120U )
#define PSXC_LPX_SLCR_RPU1_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050140U )
#define PSXC_LPX_SLCR_RPU1_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050160U )
#define PSXC_LPX_SLCR_RPU2_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050180U )
#define PSXC_LPX_SLCR_RPU2_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000501A0U )
#define PSXC_LPX_SLCR_RPU3_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000501C0U )
#define PSXC_LPX_SLCR_RPU3_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000501E0U )
#define PSXC_LPX_SLCR_RPU4_CORE0_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050200U )
#define PSXC_LPX_SLCR_RPU4_CORE1_PWR_CNTRL_REG0			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050220U )

#define PSXC_LPX_SLCR_RPU_CORE_PWR_CNTRL_WPROT_OFFSET		((u32)0x00000010U)
#define PSXC_LPX_SLCR_RPU_CORE_PWR_CNTRL_STS_OFFSET		((u32)0x0000000CU)
#define PSXC_LPX_SLCR_RPU_CORE_PWR_CNTRL_STS_ISOLATION_MASK	((u32)0x00000100U)
#define PSXC_LPX_SLCR_RPU_CORE_PWR_CNTRL_PWRUP_GATES_MASK	((u32)0x00000001U)
#define PSXC_LPX_SLCR_RPU_CACHE_PWR_CNTRL			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050400U )
#define PSXC_LPX_SLCR_RPU_CACHE_CE_CNTRL			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050404U )
#define PSXC_LPX_SLCR_RPU_CORE_PWR_CNTRL_STS_VALUE		((u32)0x000009FFU)

#define PSXC_LPX_SLCR_OCM_CE_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050304U )
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B0_I0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B0_I1_MASK			((u32)0x00000002U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B0_I2_MASK			((u32)0x00000004U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B0_I3_MASK			((u32)0x00000008U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B1_I0_MASK			((u32)0x00000100U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B1_I1_MASK			((u32)0x00000200U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B1_I2_MASK			((u32)0x00000400U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B1_I3_MASK			((u32)0x00000800U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B2_I0_MASK			((u32)0x00010000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B2_I1_MASK			((u32)0x00020000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B2_I2_MASK			((u32)0x00040000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B2_I3_MASK			((u32)0x00080000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B3_I0_MASK			((u32)0x01000000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B3_I1_MASK			((u32)0x02000000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B3_I2_MASK			((u32)0x04000000U)
#define PSXC_LPX_SLCR_OCM_CE_CNTRL_B3_I3_MASK			((u32)0x08000000U)

#define PSXC_LPX_SLCR_OCM_PWR_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050300U )
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B0_I0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B0_I1_MASK			((u32)0x00000002U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B0_I2_MASK			((u32)0x00000004U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B0_I3_MASK			((u32)0x00000008U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B1_I0_MASK			((u32)0x00000100U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B1_I1_MASK			((u32)0x00000200U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B1_I2_MASK			((u32)0x00000400U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B1_I3_MASK			((u32)0x00000800U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B2_I0_MASK			((u32)0x00010000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B2_I1_MASK			((u32)0x00020000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B2_I2_MASK			((u32)0x00040000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B2_I3_MASK			((u32)0x00080000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B3_I0_MASK			((u32)0x01000000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B3_I1_MASK			((u32)0x02000000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B3_I2_MASK			((u32)0x04000000U)
#define PSXC_LPX_SLCR_OCM_PWR_CNTRL_B3_I3_MASK			((u32)0x08000000U)

#define PSXC_LPX_SLCR_OCM_PWR_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005030CU )
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B0_I0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B0_I1_MASK			((u32)0x00000002U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B0_I2_MASK			((u32)0x00000004U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B0_I3_MASK			((u32)0x00000008U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B1_I0_MASK			((u32)0x00000100U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B1_I1_MASK			((u32)0x00000200U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B1_I2_MASK			((u32)0x00000400U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B1_I3_MASK			((u32)0x00000800U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B2_I0_MASK			((u32)0x00010000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B2_I1_MASK			((u32)0x00020000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B2_I2_MASK			((u32)0x00040000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B2_I3_MASK			((u32)0x00080000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B3_I0_MASK			((u32)0x01000000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B3_I1_MASK			((u32)0x02000000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B3_I2_MASK			((u32)0x04000000U)
#define PSXC_LPX_SLCR_OCM_PWR_STATUS_B3_I3_MASK			((u32)0x08000000U)

#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM0A_MASK		((u32)0x00010000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM1A_MASK		((u32)0x00020000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM0B_MASK		((u32)0x00040000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM1B_MASK		((u32)0x00080000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM0C_MASK		((u32)0x00100000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM1C_MASK		((u32)0x00200000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM0D_MASK		((u32)0x00400000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM1D_MASK		((u32)0x00800000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM0E_MASK		((u32)0x01000000U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_TCM1E_MASK		((u32)0x02000000U)

#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_GEM0_MASK		BITNMASK(28,1)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_GEM1_MASK		BITNMASK(29,1)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_GEM0_MASK		BITNMASK(30,1)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_GEM1_MASK		BITNMASK(31,1)


#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0A_RET_MASK		((u32)0x00100000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1A_RET_MASK		((u32)0x00200000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0B_RET_MASK		((u32)0x00400000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1B_RET_MASK		((u32)0x00800000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0C_RET_MASK		((u32)0x01000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1C_RET_MASK		((u32)0x02000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0D_RET_MASK		((u32)0x04000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1D_RET_MASK		((u32)0x08000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0E_RET_MASK		((u32)0x10000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1E_RET_MASK		((u32)0x20000000U)

#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND0_MASK	((u32)0x00000001U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND1_MASK	((u32)0x00000002U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND2_MASK	((u32)0x00000004U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND3_MASK	((u32)0x00000008U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND4_MASK	((u32)0x00000010U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND5_MASK	((u32)0x00000020U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND6_MASK	((u32)0x00000040U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND7_MASK	((u32)0x00000080U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND8_MASK	((u32)0x00000100U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND9_MASK	((u32)0x00000200U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND10_MASK	((u32)0x00000400U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND11_MASK	((u32)0x00000800U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND12_MASK	((u32)0x00001000U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND13_MASK	((u32)0x00002000U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND14_MASK	((u32)0x00004000U)
#define PSXC_LPX_SLCR_REQ_PWRUP2_STATUS_OCM_ISLAND15_MASK	((u32)0x00008000U)

#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND0_RET_MASK	((u32)0x00010000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND1_RET_MASK	((u32)0x00020000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND2_RET_MASK	((u32)0x00040000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND3_RET_MASK	((u32)0x00080000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND4_RET_MASK	((u32)0x00100000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND5_RET_MASK	((u32)0x00200000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND6_RET_MASK	((u32)0x00400000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND7_RET_MASK	((u32)0x00800000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND8_RET_MASK	((u32)0x01000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND9_RET_MASK	((u32)0x02000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND10_RET_MASK	((u32)0x04000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND11_RET_MASK	((u32)0x08000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND12_RET_MASK	((u32)0x10000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND13_RET_MASK	((u32)0x20000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND14_RET_MASK	((u32)0x40000000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN2_STATUS_OCM_ISLAND15_RET_MASK	((u32)0x80000000U)

#define PSXC_LPX_SLCR_OCM_RET_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050308U )
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B0_I0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B0_I1_MASK			((u32)0x00000002U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B0_I2_MASK			((u32)0x00000004U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B0_I3_MASK			((u32)0x00000008U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B1_I0_MASK			((u32)0x00000100U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B1_I1_MASK			((u32)0x00000200U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B1_I2_MASK			((u32)0x00000400U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B1_I3_MASK			((u32)0x00000800U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B2_I0_MASK			((u32)0x00010000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B2_I1_MASK			((u32)0x00020000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B2_I2_MASK			((u32)0x00040000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B2_I3_MASK			((u32)0x00080000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B3_I0_MASK			((u32)0x01000000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B3_I1_MASK			((u32)0x02000000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B3_I2_MASK			((u32)0x04000000U)
#define PSXC_LPX_SLCR_OCM_RET_CNTRL_B3_I3_MASK			((u32)0x08000000U)

#define PSXC_LPX_SLCR_RPU_TCM_RET_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050418U )
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050414U )
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMA0_MASK		((u32)0x00000001U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMA1_MASK		((u32)0x00000002U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMB0_MASK		((u32)0x00000004U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMB1_MASK		((u32)0x00000008U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMC0_MASK		((u32)0x00000010U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMC1_MASK		((u32)0x00000020U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMD0_MASK		((u32)0x00000040U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCMD1_MASK		((u32)0x00000080U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCME0_MASK		((u32)0x00000100U)
#define PSXC_LPX_SLCR_RPU_TCM_CE_CNTRL_TCME1_MASK		((u32)0x00000200U)

#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050410U )
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMA0_MASK		((u32)0x00000001U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMA1_MASK		((u32)0x00000002U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMB0_MASK		((u32)0x00000004U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMB1_MASK		((u32)0x00000008U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMC0_MASK		((u32)0x00000010U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMC1_MASK		((u32)0x00000020U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMD0_MASK		((u32)0x00000040U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCMD1_MASK		((u32)0x00000080U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCME0_MASK		((u32)0x00000100U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_CNTRL_TCME1_MASK		((u32)0x00000200U)

#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x0005041CU )
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMA0_MASK		((u32)0x00000001U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMA1_MASK		((u32)0x00000002U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMB0_MASK		((u32)0x00000004U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMB1_MASK		((u32)0x00000008U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMC0_MASK		((u32)0x00000010U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMC1_MASK		((u32)0x00000020U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMD0_MASK		((u32)0x00000040U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCMD1_MASK		((u32)0x00000080U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCME0_MASK		((u32)0x00000100U)
#define PSXC_LPX_SLCR_RPU_TCM_PWR_STATUS_TCME1_MASK		((u32)0x00000200U)

#define PSXC_LPX_SLCR_GEM_CE_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050424U )
#define PSXC_LPX_SLCR_GEM_CE_CNTRL_GEM0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_GEM_CE_CNTRL_GEM1_MASK			((u32)0x00000002U)

#define PSXC_LPX_SLCR_GEM_PWR_CNTRL				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050420U )
#define PSXC_LPX_SLCR_GEM_PWR_CNTRL_GEM0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_GEM_PWR_CNTRL_GEM1_MASK			((u32)0x00000100U)

#define PSXC_LPX_SLCR_GEM_PWR_STATUS				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050428U )
#define PSXC_LPX_SLCR_GEM_PWR_STATUS_GEM0_MASK			((u32)0x00000001U)
#define PSXC_LPX_SLCR_GEM_PWR_STATUS_GEM1_MASK			((u32)0x00000100U)

#define PSXC_LPX_SLCR_PWR_CTRL_STS_PRDY_SHIFT			(0U)
#define PSXC_LPX_SLCR_PWR_CTRL_STS_PRDY_WIDTH			((u32)4U)
#define PSXC_LPX_SLCR_PWR_CTRL_MAX_PWRUP_STAGES			PSXC_LPX_SLCR_PWR_CTRL_STS_PRDY_WIDTH

#define PSXC_LPX_SLCR_SCAN_CLEAR_TRIGGER			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050910U )
#define PSXC_LPX_SLCR_MEM_CLEAR_TRIGGER				( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x00050900U )

#define LPX_SLCR_RPU_PCIL_AB_PWRDWN_OFFSET			(0x000000C0U)
#define LPX_SLCR_RPU_PCIL_CDE_PWRDWN_OFFSET			(0x0000001CU)

#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU0_CORE0_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU0_CORE1_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU1_CORE0_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU1_CORE1_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU2_CORE0_MASK		(0x00000100U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU2_CORE1_MASK		(0x00000200U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU3_CORE0_MASK		(0x00001000U)
#define PSXC_LPX_SLCR_REQ_PWRUP0_STATUS_APU3_CORE1_MASK		(0x00002000U)

#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU0_CORE0_MASK	(0x00000001U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU0_CORE1_MASK	(0x00000002U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU1_CORE0_MASK	(0x00000010U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU1_CORE1_MASK	(0x00000020U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU2_CORE0_MASK	(0x00000100U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU2_CORE1_MASK	(0x00000200U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU3_CORE0_MASK	(0x00001000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN0_STATUS_APU3_CORE1_MASK	(0x00002000U)

#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUA_CORE0_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUA_CORE1_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUB_CORE0_MASK		(0x00000004U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUB_CORE1_MASK		(0x00000008U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUC_CORE0_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUC_CORE1_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUD_CORE0_MASK		(0x00000040U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUD_CORE1_MASK		(0x00000080U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUE_CORE0_MASK		(0x00000100U)
#define PSXC_LPX_SLCR_REQ_PWRUP1_STATUS_RPUE_CORE1_MASK		(0x00000200U)

#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUA_CORE0_MASK	(0x00000001U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUA_CORE1_MASK	(0x00000002U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUB_CORE0_MASK	(0x00000004U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUB_CORE1_MASK	(0x00000008U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUC_CORE0_MASK	(0x00000010U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUC_CORE1_MASK 	(0x00000020U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUD_CORE0_MASK	(0x00000040U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUD_CORE1_MASK	(0x00000080U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUE_CORE0_MASK	(0x00000100U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_RPUE_CORE1_MASK	(0x00000200U)

#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU0_CORE0_MASK	(0x00000001U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU0_CORE1_MASK	(0x00000002U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU1_CORE0_MASK	(0x00000004U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU1_CORE1_MASK	(0x00000008U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU2_CORE0_MASK	(0x00000010U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU2_CORE1_MASK	(0x00000020U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU3_CORE0_MASK	(0x00000040U)
#define PSXC_LPX_SLCR_WAKEUP0_IRQ_STATUS_APU3_CORE1_MASK	(0x00000080U)

#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUA_CORE0_MASK	(0x00000001U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUA_CORE1_MASK	(0x00000002U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUB_CORE0_MASK	(0x00000004U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUB_CORE1_MASK	(0x00000008U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUC_CORE0_MASK	(0x00000010U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUC_CORE1_MASK	(0x00000020U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUD_CORE0_MASK	(0x00000040U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUD_CORE1_MASK	(0x00000080U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUE_CORE0_MASK	(0x00000100U)
#define PSXC_LPX_SLCR_WAKEUP1_IRQ_STATUS_RPUE_CORE1_MASK	(0x00000200U)

#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU0_CORE0_MASK	(0x00000001U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU0_CORE1_MASK	(0x00000002U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU1_CORE0_MASK	(0x00000004U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU1_CORE1_MASK	(0x00000008U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU2_CORE0_MASK	(0x00000010U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU2_CORE1_MASK	(0x00000020U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU3_CORE0_MASK	(0x00000040U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_APU3_CORE1_MASK	(0x00000080U)

#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUA_CORE0_MASK	(0x00010000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUA_CORE1_MASK	(0x00020000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUB_CORE0_MASK	(0x00040000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUB_CORE1_MASK	(0x00080000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUC_CORE0_MASK	(0x00100000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUC_CORE1_MASK	(0x00200000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUD_CORE0_MASK	(0x00400000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUD_CORE1_MASK	(0x00800000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUE_CORE0_MASK	(0x01000000U)
#define PSXC_LPX_SLCR_POWER_DWN_IRQ_STATUS_RPUE_CORE1_MASK	(0x02000000U)

#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0A_MASK		(0x00000400U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1A_MASK		(0x00000800U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0B_MASK		(0x00001000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1B_MASK		(0x00002000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0C_MASK		(0x00004000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1C_MASK		(0x00008000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0D_MASK		(0x00010000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1D_MASK		(0x00020000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM0E_MASK		(0x00040000U)
#define PSXC_LPX_SLCR_REQ_PWRDWN1_STATUS_TCM1E_MASK		(0x00080000U)

#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU0_CORE0_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU0_CORE1_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU1_CORE0_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU1_CORE1_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU2_CORE0_MASK		(0x00000100U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU2_CORE1_MASK		(0x00000200U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU3_CORE0_MASK		(0x00001000U)
#define PSXC_LPX_SLCR_SCAN_MEM_CLEAR_APU3_CORE1_MASK		(0x00002000U)

/* Mask used for APUx WAKEUP/POWER_DWN IRQ regs */
#define PSXC_LPX_SLCR_ACPU0_CORE0_POWER_CTRL_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_ACPU0_CORE1_POWER_CTRL_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_ACPU1_CORE0_POWER_CTRL_MASK		(0x00000004U)
#define PSXC_LPX_SLCR_ACPU1_CORE1_POWER_CTRL_MASK		(0x00000008U)
#define PSXC_LPX_SLCR_ACPU2_CORE0_POWER_CTRL_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_ACPU2_CORE1_POWER_CTRL_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_ACPU3_CORE0_POWER_CTRL_MASK		(0x00000040U)
#define PSXC_LPX_SLCR_ACPU3_CORE1_POWER_CTRL_MASK		(0x00000080U)

/* Masks used for APUx REQ PWRUP/PWRDWN INT regs */
#define PSXC_LPX_SLCR_ACPU0_CORE0_PWR_UPDWN_MASK		(0x00000001U)
#define PSXC_LPX_SLCR_ACPU0_CORE1_PWR_UPDWN_MASK		(0x00000002U)
#define PSXC_LPX_SLCR_ACPU1_CORE0_PWR_UPDWN_MASK		(0x00000010U)
#define PSXC_LPX_SLCR_ACPU1_CORE1_PWR_UPDWN_MASK		(0x00000020U)
#define PSXC_LPX_SLCR_ACPU2_CORE0_PWR_UPDWN_MASK		(0x00000100U)
#define PSXC_LPX_SLCR_ACPU2_CORE1_PWR_UPDWN_MASK		(0x00000200U)
#define PSXC_LPX_SLCR_ACPU3_CORE0_PWR_UPDWN_MASK		(0x00001000U)
#define PSXC_LPX_SLCR_ACPU3_CORE1_PWR_UPDWN_MASK		(0x00002000U)

#define PSXC_LPX_SLCR_PMC_IRQ_PWR_MB_IRQ_EN			( (PSXC_LPX_SLCR_BASEADDR) + (u32)0x000506E8U )
/* Mask used to enable all power related interrupts to PMC */
#define PSXC_LPX_SLCR_PMC_IRQ_PWR_MB_IRQ_EN_MASK		(0x0000D800U)

/*
 *R52 PCIL offsets
 */
//#define LPX_SLCR_RPU_PCIL_A0_ISR_OFFSET	(0x10000U)
//#define LPX_SLCR_RPU_PCIL_A1_ISR_OFFSET	(0x10100U)
//#define LPX_SLCR_RPU_PCIL_B0_ISR_OFFSET	(0x11000U)
//#define LPX_SLCR_RPU_PCIL_B1_ISR_OFFSET	(0x11100U)

/*Note: offset is from PCIL_ISR*/
//#define LPX_SLCR_RPU_PCIL_X_IEN_OFFSET	(0x8U)
#define LPX_SLCR_RPU_PCIL_X_IDS_OFFSET				(0xCU)
/*PR=PREQ of PChannel*/
//#define LPX_SLCR_RPU_PCIL_X_PR_OFFSET	(0x80U)
/*PS=PSTATE of PChannel*/
//#define LPX_SLCR_RPU_PCIL_X_PS_OFFSET	(0x84U)
//#define LPX_SLCR_RPU_PCIL_X_PWRDWN_OFFSET	(0xC0U)

/**
 * HNICX_NPI_0 Module
 */
#define HNICX_NPI_0_BASEADDR					(0xF6AF0000U)

/**
 * HNICX_NPI_0 registers
 */
#define HNICX_NPI_0_NPI_CSR_INST_NPI_CSR_CMD_WRITE		BIT32(30)
#define HNICX_NPI_0_NPI_CSR_WR_STATUS_VALID_RESP		(0x1U)
#define HNICX_NPI_0_NPI_CSR_INST_OFFSET				(0x200U)
#define HNICX_NPI_0_NPI_CSR_WDATA_OFFSET			(0x204U)
#define HNICX_NPI_0_NPI_CSR_WR_STATUS_OFFSET			(0x208U)
#define HNICX_NPI_0_NPI_CSR_INST_NPI_CSR_ADDR_MASK		(0x3FFFFFFFU)
#define HNICX_NPI_0_NPI_CSR_WR_STATUS_MASK			(0x00000007U)

/*
 * PPU RAM Memory
*/
#define PPU_RAM_INSTR_CNTLR_BASEADDR			(0xF0200000U)
#define PPU_RAM_INSTR_CNTLR_HIGHADDR			(0xF027FFAFU)
#define PPU_RAM_DATA_CNTLR_BASEADDR			(0xF0280000U)
#define PPU_RAM_DATA_CNTLR_HIGHADDR			(0xF029FFFFU)


#ifdef __cplusplus
}
#endif

#endif /* XPM_REGS_H */
