#define _CFFI_

/* We try to define Py_LIMITED_API before including Python.h.

   Mess: we can only define it if Py_DEBUG, Py_TRACE_REFS and
   Py_REF_DEBUG are not defined.  This is a best-effort approximation:
   we can learn about Py_DEBUG from pyconfig.h, but it is unclear if
   the same works for the other two macros.  Py_DEBUG implies them,
   but not the other way around.

   The implementation is messy (issue #350): on Windows, with _MSC_VER,
   we have to define Py_LIMITED_API even before including pyconfig.h.
   In that case, we guess what pyconfig.h will do to the macros above,
   and check our guess after the #include.

   Note that on Windows, with CPython 3.x, you need >= 3.5 and virtualenv
   version >= 16.0.0.  With older versions of either, you don't get a
   copy of PYTHON3.DLL in the virtualenv.  We can't check the version of
   CPython *before* we even include pyconfig.h.  ffi.set_source() puts
   a ``#define _CFFI_NO_LIMITED_API'' at the start of this file if it is
   running on Windows < 3.5, as an attempt at fixing it, but that's
   arguably wrong because it may not be the target version of Python.
   Still better than nothing I guess.  As another workaround, you can
   remove the definition of Py_LIMITED_API here.

   See also 'py_limited_api' in cffi/setuptools_ext.py.
*/
#if !defined(_CFFI_USE_EMBEDDING) && !defined(Py_LIMITED_API)
#  ifdef _MSC_VER
#    if !defined(_DEBUG) && !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#    include <pyconfig.h>
     /* sanity-check: Py_LIMITED_API will cause crashes if any of these
        are also defined.  Normally, the Python file PC/pyconfig.h does not
        cause any of these to be defined, with the exception that _DEBUG
        causes Py_DEBUG.  Double-check that. */
#    ifdef Py_LIMITED_API
#      if defined(Py_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_DEBUG, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_TRACE_REFS)
#        error "pyconfig.h unexpectedly defines Py_TRACE_REFS, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_REF_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_REF_DEBUG, but Py_LIMITED_API is set"
#      endif
#    endif
#  else
#    include <pyconfig.h>
#    if !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#  endif
#endif

#include <Python.h>
#ifdef __cplusplus
extern "C" {
#endif
#include <stddef.h>

/* This part is from file 'cffi/parse_c_type.h'.  It is copied at the
   beginning of C sources generated by CFFI's ffi.set_source(). */

typedef void *_cffi_opcode_t;

#define _CFFI_OP(opcode, arg)   (_cffi_opcode_t)(opcode | (((uintptr_t)(arg)) << 8))
#define _CFFI_GETOP(cffi_opcode)    ((unsigned char)(uintptr_t)cffi_opcode)
#define _CFFI_GETARG(cffi_opcode)   (((intptr_t)cffi_opcode) >> 8)

#define _CFFI_OP_PRIMITIVE       1
#define _CFFI_OP_POINTER         3
#define _CFFI_OP_ARRAY           5
#define _CFFI_OP_OPEN_ARRAY      7
#define _CFFI_OP_STRUCT_UNION    9
#define _CFFI_OP_ENUM           11
#define _CFFI_OP_FUNCTION       13
#define _CFFI_OP_FUNCTION_END   15
#define _CFFI_OP_NOOP           17
#define _CFFI_OP_BITFIELD       19
#define _CFFI_OP_TYPENAME       21
#define _CFFI_OP_CPYTHON_BLTN_V 23   // varargs
#define _CFFI_OP_CPYTHON_BLTN_N 25   // noargs
#define _CFFI_OP_CPYTHON_BLTN_O 27   // O  (i.e. a single arg)
#define _CFFI_OP_CONSTANT       29
#define _CFFI_OP_CONSTANT_INT   31
#define _CFFI_OP_GLOBAL_VAR     33
#define _CFFI_OP_DLOPEN_FUNC    35
#define _CFFI_OP_DLOPEN_CONST   37
#define _CFFI_OP_GLOBAL_VAR_F   39
#define _CFFI_OP_EXTERN_PYTHON  41

#define _CFFI_PRIM_VOID          0
#define _CFFI_PRIM_BOOL          1
#define _CFFI_PRIM_CHAR          2
#define _CFFI_PRIM_SCHAR         3
#define _CFFI_PRIM_UCHAR         4
#define _CFFI_PRIM_SHORT         5
#define _CFFI_PRIM_USHORT        6
#define _CFFI_PRIM_INT           7
#define _CFFI_PRIM_UINT          8
#define _CFFI_PRIM_LONG          9
#define _CFFI_PRIM_ULONG        10
#define _CFFI_PRIM_LONGLONG     11
#define _CFFI_PRIM_ULONGLONG    12
#define _CFFI_PRIM_FLOAT        13
#define _CFFI_PRIM_DOUBLE       14
#define _CFFI_PRIM_LONGDOUBLE   15

#define _CFFI_PRIM_WCHAR        16
#define _CFFI_PRIM_INT8         17
#define _CFFI_PRIM_UINT8        18
#define _CFFI_PRIM_INT16        19
#define _CFFI_PRIM_UINT16       20
#define _CFFI_PRIM_INT32        21
#define _CFFI_PRIM_UINT32       22
#define _CFFI_PRIM_INT64        23
#define _CFFI_PRIM_UINT64       24
#define _CFFI_PRIM_INTPTR       25
#define _CFFI_PRIM_UINTPTR      26
#define _CFFI_PRIM_PTRDIFF      27
#define _CFFI_PRIM_SIZE         28
#define _CFFI_PRIM_SSIZE        29
#define _CFFI_PRIM_INT_LEAST8   30
#define _CFFI_PRIM_UINT_LEAST8  31
#define _CFFI_PRIM_INT_LEAST16  32
#define _CFFI_PRIM_UINT_LEAST16 33
#define _CFFI_PRIM_INT_LEAST32  34
#define _CFFI_PRIM_UINT_LEAST32 35
#define _CFFI_PRIM_INT_LEAST64  36
#define _CFFI_PRIM_UINT_LEAST64 37
#define _CFFI_PRIM_INT_FAST8    38
#define _CFFI_PRIM_UINT_FAST8   39
#define _CFFI_PRIM_INT_FAST16   40
#define _CFFI_PRIM_UINT_FAST16  41
#define _CFFI_PRIM_INT_FAST32   42
#define _CFFI_PRIM_UINT_FAST32  43
#define _CFFI_PRIM_INT_FAST64   44
#define _CFFI_PRIM_UINT_FAST64  45
#define _CFFI_PRIM_INTMAX       46
#define _CFFI_PRIM_UINTMAX      47
#define _CFFI_PRIM_FLOATCOMPLEX 48
#define _CFFI_PRIM_DOUBLECOMPLEX 49
#define _CFFI_PRIM_CHAR16       50
#define _CFFI_PRIM_CHAR32       51

#define _CFFI__NUM_PRIM         52
#define _CFFI__UNKNOWN_PRIM           (-1)
#define _CFFI__UNKNOWN_FLOAT_PRIM     (-2)
#define _CFFI__UNKNOWN_LONG_DOUBLE    (-3)

#define _CFFI__IO_FILE_STRUCT         (-1)


struct _cffi_global_s {
    const char *name;
    void *address;
    _cffi_opcode_t type_op;
    void *size_or_direct_fn;  // OP_GLOBAL_VAR: size, or 0 if unknown
                              // OP_CPYTHON_BLTN_*: addr of direct function
};

struct _cffi_getconst_s {
    unsigned long long value;
    const struct _cffi_type_context_s *ctx;
    int gindex;
};

struct _cffi_struct_union_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_STRUCT_UNION
    int flags;               // _CFFI_F_* flags below
    size_t size;
    int alignment;
    int first_field_index;   // -> _cffi_fields array
    int num_fields;
};
#define _CFFI_F_UNION         0x01   // is a union, not a struct
#define _CFFI_F_CHECK_FIELDS  0x02   // complain if fields are not in the
                                     // "standard layout" or if some are missing
#define _CFFI_F_PACKED        0x04   // for CHECK_FIELDS, assume a packed struct
#define _CFFI_F_EXTERNAL      0x08   // in some other ffi.include()
#define _CFFI_F_OPAQUE        0x10   // opaque

struct _cffi_field_s {
    const char *name;
    size_t field_offset;
    size_t field_size;
    _cffi_opcode_t field_type_op;
};

struct _cffi_enum_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_ENUM
    int type_prim;           // _CFFI_PRIM_xxx
    const char *enumerators; // comma-delimited string
};

struct _cffi_typename_s {
    const char *name;
    int type_index;   /* if opaque, points to a possibly artificial
                         OP_STRUCT which is itself opaque */
};

struct _cffi_type_context_s {
    _cffi_opcode_t *types;
    const struct _cffi_global_s *globals;
    const struct _cffi_field_s *fields;
    const struct _cffi_struct_union_s *struct_unions;
    const struct _cffi_enum_s *enums;
    const struct _cffi_typename_s *typenames;
    int num_globals;
    int num_struct_unions;
    int num_enums;
    int num_typenames;
    const char *const *includes;
    int num_types;
    int flags;      /* future extension */
};

struct _cffi_parse_info_s {
    const struct _cffi_type_context_s *ctx;
    _cffi_opcode_t *output;
    unsigned int output_size;
    size_t error_location;
    const char *error_message;
};

struct _cffi_externpy_s {
    const char *name;
    size_t size_of_result;
    void *reserved1, *reserved2;
};

#ifdef _CFFI_INTERNAL
static int parse_c_type(struct _cffi_parse_info_s *info, const char *input);
static int search_in_globals(const struct _cffi_type_context_s *ctx,
                             const char *search, size_t search_len);
static int search_in_struct_unions(const struct _cffi_type_context_s *ctx,
                                   const char *search, size_t search_len);
#endif

/* this block of #ifs should be kept exactly identical between
   c/_cffi_backend.c, cffi/vengine_cpy.py, cffi/vengine_gen.py
   and cffi/_cffi_include.h */
#if defined(_MSC_VER)
# include <malloc.h>   /* for alloca() */
# if _MSC_VER < 1600   /* MSVC < 2010 */
   typedef __int8 int8_t;
   typedef __int16 int16_t;
   typedef __int32 int32_t;
   typedef __int64 int64_t;
   typedef unsigned __int8 uint8_t;
   typedef unsigned __int16 uint16_t;
   typedef unsigned __int32 uint32_t;
   typedef unsigned __int64 uint64_t;
   typedef __int8 int_least8_t;
   typedef __int16 int_least16_t;
   typedef __int32 int_least32_t;
   typedef __int64 int_least64_t;
   typedef unsigned __int8 uint_least8_t;
   typedef unsigned __int16 uint_least16_t;
   typedef unsigned __int32 uint_least32_t;
   typedef unsigned __int64 uint_least64_t;
   typedef __int8 int_fast8_t;
   typedef __int16 int_fast16_t;
   typedef __int32 int_fast32_t;
   typedef __int64 int_fast64_t;
   typedef unsigned __int8 uint_fast8_t;
   typedef unsigned __int16 uint_fast16_t;
   typedef unsigned __int32 uint_fast32_t;
   typedef unsigned __int64 uint_fast64_t;
   typedef __int64 intmax_t;
   typedef unsigned __int64 uintmax_t;
# else
#  include <stdint.h>
# endif
# if _MSC_VER < 1800   /* MSVC < 2013 */
#  ifndef __cplusplus
    typedef unsigned char _Bool;
#  endif
# endif
#else
# include <stdint.h>
# if (defined (__SVR4) && defined (__sun)) || defined(_AIX) || defined(__hpux)
#  include <alloca.h>
# endif
#endif

#ifdef __GNUC__
# define _CFFI_UNUSED_FN  __attribute__((unused))
#else
# define _CFFI_UNUSED_FN  /* nothing */
#endif

#ifdef __cplusplus
# ifndef _Bool
   typedef bool _Bool;   /* semi-hackish: C++ has no _Bool; bool is builtin */
# endif
#endif

/**********  CPython-specific section  **********/
#ifndef PYPY_VERSION


#if PY_MAJOR_VERSION >= 3
# define PyInt_FromLong PyLong_FromLong
#endif

#define _cffi_from_c_double PyFloat_FromDouble
#define _cffi_from_c_float PyFloat_FromDouble
#define _cffi_from_c_long PyInt_FromLong
#define _cffi_from_c_ulong PyLong_FromUnsignedLong
#define _cffi_from_c_longlong PyLong_FromLongLong
#define _cffi_from_c_ulonglong PyLong_FromUnsignedLongLong
#define _cffi_from_c__Bool PyBool_FromLong

#define _cffi_to_c_double PyFloat_AsDouble
#define _cffi_to_c_float PyFloat_AsDouble

#define _cffi_from_c_int(x, type)                                        \
    (((type)-1) > 0 ? /* unsigned */                                     \
        (sizeof(type) < sizeof(long) ?                                   \
            PyInt_FromLong((long)x) :                                    \
         sizeof(type) == sizeof(long) ?                                  \
            PyLong_FromUnsignedLong((unsigned long)x) :                  \
            PyLong_FromUnsignedLongLong((unsigned long long)x)) :        \
        (sizeof(type) <= sizeof(long) ?                                  \
            PyInt_FromLong((long)x) :                                    \
            PyLong_FromLongLong((long long)x)))

#define _cffi_to_c_int(o, type)                                          \
    ((type)(                                                             \
     sizeof(type) == 1 ? (((type)-1) > 0 ? (type)_cffi_to_c_u8(o)        \
                                         : (type)_cffi_to_c_i8(o)) :     \
     sizeof(type) == 2 ? (((type)-1) > 0 ? (type)_cffi_to_c_u16(o)       \
                                         : (type)_cffi_to_c_i16(o)) :    \
     sizeof(type) == 4 ? (((type)-1) > 0 ? (type)_cffi_to_c_u32(o)       \
                                         : (type)_cffi_to_c_i32(o)) :    \
     sizeof(type) == 8 ? (((type)-1) > 0 ? (type)_cffi_to_c_u64(o)       \
                                         : (type)_cffi_to_c_i64(o)) :    \
     (Py_FatalError("unsupported size for type " #type), (type)0)))

#define _cffi_to_c_i8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[1])
#define _cffi_to_c_u8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[2])
#define _cffi_to_c_i16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[3])
#define _cffi_to_c_u16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[4])
#define _cffi_to_c_i32                                                   \
                 ((int(*)(PyObject *))_cffi_exports[5])
#define _cffi_to_c_u32                                                   \
                 ((unsigned int(*)(PyObject *))_cffi_exports[6])
#define _cffi_to_c_i64                                                   \
                 ((long long(*)(PyObject *))_cffi_exports[7])
#define _cffi_to_c_u64                                                   \
                 ((unsigned long long(*)(PyObject *))_cffi_exports[8])
#define _cffi_to_c_char                                                  \
                 ((int(*)(PyObject *))_cffi_exports[9])
#define _cffi_from_c_pointer                                             \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[10])
#define _cffi_to_c_pointer                                               \
    ((char *(*)(PyObject *, struct _cffi_ctypedescr *))_cffi_exports[11])
#define _cffi_get_struct_layout                                          \
    not used any more
#define _cffi_restore_errno                                              \
    ((void(*)(void))_cffi_exports[13])
#define _cffi_save_errno                                                 \
    ((void(*)(void))_cffi_exports[14])
#define _cffi_from_c_char                                                \
    ((PyObject *(*)(char))_cffi_exports[15])
#define _cffi_from_c_deref                                               \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[16])
#define _cffi_to_c                                                       \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[17])
#define _cffi_from_c_struct                                              \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[18])
#define _cffi_to_c_wchar_t                                               \
    ((_cffi_wchar_t(*)(PyObject *))_cffi_exports[19])
#define _cffi_from_c_wchar_t                                             \
    ((PyObject *(*)(_cffi_wchar_t))_cffi_exports[20])
#define _cffi_to_c_long_double                                           \
    ((long double(*)(PyObject *))_cffi_exports[21])
#define _cffi_to_c__Bool                                                 \
    ((_Bool(*)(PyObject *))_cffi_exports[22])
#define _cffi_prepare_pointer_call_argument                              \
    ((Py_ssize_t(*)(struct _cffi_ctypedescr *,                           \
                    PyObject *, char **))_cffi_exports[23])
#define _cffi_convert_array_from_object                                  \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[24])
#define _CFFI_CPIDX  25
#define _cffi_call_python                                                \
    ((void(*)(struct _cffi_externpy_s *, char *))_cffi_exports[_CFFI_CPIDX])
#define _cffi_to_c_wchar3216_t                                           \
    ((int(*)(PyObject *))_cffi_exports[26])
#define _cffi_from_c_wchar3216_t                                         \
    ((PyObject *(*)(int))_cffi_exports[27])
#define _CFFI_NUM_EXPORTS 28

struct _cffi_ctypedescr;

static void *_cffi_exports[_CFFI_NUM_EXPORTS];

#define _cffi_type(index)   (                           \
    assert((((uintptr_t)_cffi_types[index]) & 1) == 0), \
    (struct _cffi_ctypedescr *)_cffi_types[index])

static PyObject *_cffi_init(const char *module_name, Py_ssize_t version,
                            const struct _cffi_type_context_s *ctx)
{
    PyObject *module, *o_arg, *new_module;
    void *raw[] = {
        (void *)module_name,
        (void *)version,
        (void *)_cffi_exports,
        (void *)ctx,
    };

    module = PyImport_ImportModule("_cffi_backend");
    if (module == NULL)
        goto failure;

    o_arg = PyLong_FromVoidPtr((void *)raw);
    if (o_arg == NULL)
        goto failure;

    new_module = PyObject_CallMethod(
        module, (char *)"_init_cffi_1_0_external_module", (char *)"O", o_arg);

    Py_DECREF(o_arg);
    Py_DECREF(module);
    return new_module;

  failure:
    Py_XDECREF(module);
    return NULL;
}


#ifdef HAVE_WCHAR_H
typedef wchar_t _cffi_wchar_t;
#else
typedef uint16_t _cffi_wchar_t;   /* same random pick as _cffi_backend.c */
#endif

_CFFI_UNUSED_FN static uint16_t _cffi_to_c_char16_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return (uint16_t)_cffi_to_c_wchar_t(o);
    else
        return (uint16_t)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char16_t(uint16_t x)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

_CFFI_UNUSED_FN static int _cffi_to_c_char32_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return (int)_cffi_to_c_wchar_t(o);
    else
        return (int)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char32_t(unsigned int x)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

union _cffi_union_alignment_u {
    unsigned char m_char;
    unsigned short m_short;
    unsigned int m_int;
    unsigned long m_long;
    unsigned long long m_longlong;
    float m_float;
    double m_double;
    long double m_longdouble;
};

struct _cffi_freeme_s {
    struct _cffi_freeme_s *next;
    union _cffi_union_alignment_u alignment;
};

_CFFI_UNUSED_FN static int
_cffi_convert_array_argument(struct _cffi_ctypedescr *ctptr, PyObject *arg,
                             char **output_data, Py_ssize_t datasize,
                             struct _cffi_freeme_s **freeme)
{
    char *p;
    if (datasize < 0)
        return -1;

    p = *output_data;
    if (p == NULL) {
        struct _cffi_freeme_s *fp = (struct _cffi_freeme_s *)PyObject_Malloc(
            offsetof(struct _cffi_freeme_s, alignment) + (size_t)datasize);
        if (fp == NULL)
            return -1;
        fp->next = *freeme;
        *freeme = fp;
        p = *output_data = (char *)&fp->alignment;
    }
    memset((void *)p, 0, (size_t)datasize);
    return _cffi_convert_array_from_object(p, ctptr, arg);
}

_CFFI_UNUSED_FN static void
_cffi_free_array_arguments(struct _cffi_freeme_s *freeme)
{
    do {
        void *p = (void *)freeme;
        freeme = freeme->next;
        PyObject_Free(p);
    } while (freeme != NULL);
}

/**********  end CPython-specific section  **********/
#else
_CFFI_UNUSED_FN
static void (*_cffi_call_python_org)(struct _cffi_externpy_s *, char *);
# define _cffi_call_python  _cffi_call_python_org
#endif


#define _cffi_array_len(array)   (sizeof(array) / sizeof((array)[0]))

#define _cffi_prim_int(size, sign)                                      \
    ((size) == 1 ? ((sign) ? _CFFI_PRIM_INT8  : _CFFI_PRIM_UINT8)  :    \
     (size) == 2 ? ((sign) ? _CFFI_PRIM_INT16 : _CFFI_PRIM_UINT16) :    \
     (size) == 4 ? ((sign) ? _CFFI_PRIM_INT32 : _CFFI_PRIM_UINT32) :    \
     (size) == 8 ? ((sign) ? _CFFI_PRIM_INT64 : _CFFI_PRIM_UINT64) :    \
     _CFFI__UNKNOWN_PRIM)

#define _cffi_prim_float(size)                                          \
    ((size) == sizeof(float) ? _CFFI_PRIM_FLOAT :                       \
     (size) == sizeof(double) ? _CFFI_PRIM_DOUBLE :                     \
     (size) == sizeof(long double) ? _CFFI__UNKNOWN_LONG_DOUBLE :       \
     _CFFI__UNKNOWN_FLOAT_PRIM)

#define _cffi_check_int(got, got_nonpos, expected)      \
    ((got_nonpos) == (expected <= 0) &&                 \
     (got) == (unsigned long long)expected)

#ifdef MS_WIN32
# define _cffi_stdcall  __stdcall
#else
# define _cffi_stdcall  /* nothing */
#endif

#ifdef __cplusplus
}
#endif

/************************************************************/


/* define our OpenSSL API compatibility level to 1.0.1. Any symbols older than
   that will raise an error during compilation. We can raise this number again
   after we drop 1.0.2 support in the distant future.  */
#define OPENSSL_API_COMPAT 0x10001000L

#include <openssl/opensslv.h>


#if defined(LIBRESSL_VERSION_NUMBER)
#define CRYPTOGRAPHY_IS_LIBRESSL 1
#else
#define CRYPTOGRAPHY_IS_LIBRESSL 0
#endif

#if defined(OPENSSL_IS_BORINGSSL)
#define CRYPTOGRAPHY_IS_BORINGSSL 1
#else
#define CRYPTOGRAPHY_IS_BORINGSSL 0
#endif

/*
    LibreSSL removed e_os2.h from the public headers so we'll only include it
    if we're using vanilla OpenSSL.
*/
#if !CRYPTOGRAPHY_IS_LIBRESSL
#include <openssl/e_os2.h>
#endif
#if defined(_WIN32)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <Wincrypt.h>
#include <Winsock2.h>
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_322     (LIBRESSL_VERSION_NUMBER < 0x3020200f)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_332     (LIBRESSL_VERSION_NUMBER < 0x3030200f)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340     (LIBRESSL_VERSION_NUMBER < 0x3040000f)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350     (LIBRESSL_VERSION_NUMBER < 0x3050000f)

#else
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_322 (0)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_332 (0)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 (0)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350 (0)
#endif

#if OPENSSL_VERSION_NUMBER < 0x10100000
    #error "pyca/cryptography MUST be linked with Openssl 1.1.0 or later"
#endif

#define CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x10101040 && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_300_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x30000000 && !CRYPTOGRAPHY_IS_LIBRESSL)

#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111     (OPENSSL_VERSION_NUMBER < 0x10101000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B     (OPENSSL_VERSION_NUMBER < 0x10101020 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D     (OPENSSL_VERSION_NUMBER < 0x10101040 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111E     (OPENSSL_VERSION_NUMBER < 0x10101050 || CRYPTOGRAPHY_IS_LIBRESSL)
#if (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D && !CRYPTOGRAPHY_IS_LIBRESSL &&     !defined(OPENSSL_NO_ENGINE)) || defined(USE_OSRANDOM_RNG_FOR_TESTING)
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 1
#else
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 0
#endif


#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
#include <openssl/provider.h>
#include <openssl/proverr.h>
#endif


#include <openssl/asn1.h>


#include <openssl/bn.h>


#include <openssl/bio.h>


#if !defined(OPENSSL_NO_CMAC)
#include <openssl/cmac.h>
#endif


#include <openssl/conf.h>


#include <openssl/crypto.h>


#include <openssl/dh.h>


#include <openssl/dsa.h>


#include <openssl/ec.h>
#include <openssl/obj_mac.h>


#include <openssl/ecdh.h>


#include <openssl/ecdsa.h>


#include <openssl/engine.h>


#include <openssl/err.h>


#include <openssl/evp.h>


#include <openssl/crypto.h>


#include <openssl/hmac.h>


#include <openssl/obj_mac.h>


#include <openssl/objects.h>


#include <openssl/opensslv.h>

#ifndef OPENSSL_NO_ENGINE
/* OpenSSL has ENGINE support so include all of this. */
#ifdef _WIN32
  #include <Wincrypt.h>
#else
  #include <fcntl.h>
  #include <unistd.h>
   /* for defined(BSD) */
  #ifndef __MVS__
    #include <sys/param.h>
  #endif

  #ifdef BSD
    /* for SYS_getentropy */
    #include <sys/syscall.h>
  #endif

  #ifdef __APPLE__
    #include <sys/random.h>
    /* To support weak linking we need to declare this as a weak import even if
     * it's not present in sys/random (e.g. macOS < 10.12). */
    extern int getentropy(void *buffer, size_t size) __attribute((weak_import));
  #endif

  #ifdef __linux__
    /* for SYS_getrandom */
    #include <sys/syscall.h>
    #ifndef GRND_NONBLOCK
      #define GRND_NONBLOCK 0x0001
    #endif /* GRND_NONBLOCK */

    #ifndef SYS_getrandom
      /* We only bother to define the constants for platforms where we ship
       * wheels, since that's the predominant way you get a situation where
       * you don't have SYS_getrandom at compile time but do have the syscall
       * at runtime */
      #if defined(__x86_64__)
        #define SYS_getrandom 318
      #elif defined(__i386__)
        #define SYS_getrandom 355
      #elif defined(__aarch64__)
        #define SYS_getrandom 278
      #endif
    #endif
  #endif /* __linux__ */
#endif /* _WIN32 */

#define CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM 1
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY 2
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM 3
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM 4

#ifndef CRYPTOGRAPHY_OSRANDOM_ENGINE
  #if defined(_WIN32)
    /* Windows */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
  #elif defined(BSD) && defined(SYS_getentropy)
    /* OpenBSD 5.6+ & macOS with SYS_getentropy defined, although < 10.12 will fallback
     * to urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
  #elif defined(__linux__) && defined(SYS_getrandom)
    /* Linux 3.17+ */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
  #else
    /* Keep this as last entry, fall back to /dev/urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
  #endif
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE */

/* Fallbacks need /dev/urandom helper functions. */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM || \
     CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM || \
     (CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY && \
     defined(__APPLE__))
  #define CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM 1
#endif

enum {
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED = -2,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS
};

enum {
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS
};

/* engine ctrl */
#define CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION ENGINE_CMD_BASE

/* error reporting */
static void ERR_load_Cryptography_OSRandom_strings(void);
static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line);

#define CRYPTOGRAPHY_OSRANDOM_F_INIT 100
#define CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES 101
#define CRYPTOGRAPHY_OSRANDOM_F_FINISH 102
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD 300
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ 301

#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT 100
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM 101
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT 102

#define CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED 200

#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED 300
#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED 301

#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED 400
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED 402
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED 403
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT 404
#endif


#include <openssl/pem.h>


#include <openssl/pkcs12.h>


#include <openssl/rand.h>


#include <openssl/rsa.h>


#include <openssl/ssl.h>

typedef STACK_OF(SSL_CIPHER) Cryptography_STACK_OF_SSL_CIPHER;


#include <openssl/ssl.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(X509) Cryptography_STACK_OF_X509;
typedef STACK_OF(X509_CRL) Cryptography_STACK_OF_X509_CRL;
typedef STACK_OF(X509_REVOKED) Cryptography_STACK_OF_X509_REVOKED;


#include <openssl/x509.h>

/*
 * See the comment above Cryptography_STACK_OF_X509 in x509.py
 */
typedef STACK_OF(X509_NAME) Cryptography_STACK_OF_X509_NAME;
typedef STACK_OF(X509_NAME_ENTRY) Cryptography_STACK_OF_X509_NAME_ENTRY;


#include <openssl/x509v3.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */


#include <openssl/x509_vfy.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(ASN1_OBJECT) Cryptography_STACK_OF_ASN1_OBJECT;
typedef STACK_OF(X509_OBJECT) Cryptography_STACK_OF_X509_OBJECT;


#include <openssl/pkcs7.h>


#include <string.h>




#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
static const long Cryptography_HAS_PROVIDERS = 1;
#else
static const long Cryptography_HAS_PROVIDERS = 0;
typedef void OSSL_PROVIDER;
typedef void OSSL_LIB_CTX;
static const long PROV_R_BAD_DECRYPT = 0;
static const long PROV_R_XTS_DUPLICATED_KEYS = 0;
static const long PROV_R_WRONG_FINAL_BLOCK_LENGTH = 0;
OSSL_PROVIDER *(*OSSL_PROVIDER_load)(OSSL_LIB_CTX *, const char *) = NULL;
int (*OSSL_PROVIDER_unload)(OSSL_PROVIDER *) = NULL;
#endif




#if CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_BN_FLAGS = 0;

static const int BN_FLG_CONSTTIME = 0;
void (*BN_set_flags)(BIGNUM *, int) = NULL;
int (*BN_prime_checks_for_size)(int) = NULL;
#else
static const long Cryptography_HAS_BN_FLAGS = 1;
#endif


#if CRYPTOGRAPHY_IS_LIBRESSL || CRYPTOGRAPHY_IS_BORINGSSL
#include <sys/socket.h>
#include <stdlib.h>
typedef struct sockaddr BIO_ADDR;

BIO_ADDR *BIO_ADDR_new(void) {
    return malloc(sizeof(struct sockaddr_storage));
}

void BIO_ADDR_free(BIO_ADDR *ptr) {
    free(ptr);
}
#endif






/* In 1.1.0 SSLeay has finally been retired. We bidirectionally define the
   values so you can use either one. This is so we can use the new function
   names no matter what OpenSSL we're running on, but users on older pyOpenSSL
   releases won't see issues if they're running OpenSSL 1.1.0 */
#if !defined(SSLEAY_VERSION)
# define SSLeay                  OpenSSL_version_num
# define SSLeay_version          OpenSSL_version
# define SSLEAY_VERSION_NUMBER   OPENSSL_VERSION_NUMBER
# define SSLEAY_VERSION          OPENSSL_VERSION
# define SSLEAY_CFLAGS           OPENSSL_CFLAGS
# define SSLEAY_BUILT_ON         OPENSSL_BUILT_ON
# define SSLEAY_PLATFORM         OPENSSL_PLATFORM
# define SSLEAY_DIR              OPENSSL_DIR
#endif
#if !defined(OPENSSL_VERSION)
# define OpenSSL_version_num     SSLeay
# define OpenSSL_version         SSLeay_version
# define OPENSSL_VERSION         SSLEAY_VERSION
# define OPENSSL_CFLAGS          SSLEAY_CFLAGS
# define OPENSSL_BUILT_ON        SSLEAY_BUILT_ON
# define OPENSSL_PLATFORM        SSLEAY_PLATFORM
# define OPENSSL_DIR             SSLEAY_DIR
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_OPENSSL_CLEANUP = 0;
void (*OPENSSL_cleanup)(void) = NULL;
#else
static const long Cryptography_HAS_OPENSSL_CLEANUP = 1;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL || CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_MEM_FUNCTIONS = 0;
int (*Cryptography_CRYPTO_set_mem_functions)(
    void *(*)(size_t, const char *, int),
    void *(*)(void *, size_t, const char *, int),
    void (*)(void *, const char *, int)) = NULL;

#else
static const long Cryptography_HAS_MEM_FUNCTIONS = 1;

int Cryptography_CRYPTO_set_mem_functions(
    void *(*m)(size_t, const char *, int),
    void *(*r)(void *, size_t, const char *, int),
    void (*f)(void *, const char *, int)
) {
    return CRYPTO_set_mem_functions(m, r, f);
}
#endif

void *Cryptography_malloc_wrapper(size_t size, const char *path, int line) {
    return malloc(size);
}

void *Cryptography_realloc_wrapper(void *ptr, size_t size, const char *path,
                                   int line) {
    return realloc(ptr, size);
}

void Cryptography_free_wrapper(void *ptr, const char *path, int line) {
    free(ptr);
}


#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350
#ifndef DH_CHECK_Q_NOT_PRIME
#define DH_CHECK_Q_NOT_PRIME            0x10
#endif

#ifndef DH_CHECK_INVALID_Q_VALUE
#define DH_CHECK_INVALID_Q_VALUE        0x20
#endif

#ifndef DH_CHECK_INVALID_J_VALUE
#define DH_CHECK_INVALID_J_VALUE        0x40
#endif

/* DH_check implementation taken from OpenSSL 1.1.0pre6 */

/*-
 * Check that p is a safe prime and
 * if g is 2, 3 or 5, check that it is a suitable generator
 * where
 * for 2, p mod 24 == 11
 * for 3, p mod 12 == 5
 * for 5, p mod 10 == 3 or 7
 * should hold.
 */

int Cryptography_DH_check(const DH *dh, int *ret)
{
    int ok = 0, r;
    BN_CTX *ctx = NULL;
    BN_ULONG l;
    BIGNUM *t1 = NULL, *t2 = NULL;

    *ret = 0;
    ctx = BN_CTX_new();
    if (ctx == NULL)
        goto err;
    BN_CTX_start(ctx);
    t1 = BN_CTX_get(ctx);
    if (t1 == NULL)
        goto err;
    t2 = BN_CTX_get(ctx);
    if (t2 == NULL)
        goto err;

    if (dh->q) {
        if (BN_cmp(dh->g, BN_value_one()) <= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else if (BN_cmp(dh->g, dh->p) >= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else {
            /* Check g^q == 1 mod p */
            if (!BN_mod_exp(t1, dh->g, dh->q, dh->p, ctx))
                goto err;
            if (!BN_is_one(t1))
                *ret |= DH_NOT_SUITABLE_GENERATOR;
        }
        r = BN_is_prime_ex(dh->q, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_Q_NOT_PRIME;
        /* Check p == 1 mod q  i.e. q divides p - 1 */
        if (!BN_div(t1, t2, dh->p, dh->q, ctx))
            goto err;
        if (!BN_is_one(t2))
            *ret |= DH_CHECK_INVALID_Q_VALUE;
        if (dh->j && BN_cmp(dh->j, t1))
            *ret |= DH_CHECK_INVALID_J_VALUE;

    } else if (BN_is_word(dh->g, DH_GENERATOR_2)) {
        l = BN_mod_word(dh->p, 24);
        if (l == (BN_ULONG)-1)
            goto err;
        if (l != 11)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else if (BN_is_word(dh->g, DH_GENERATOR_5)) {
        l = BN_mod_word(dh->p, 10);
        if (l == (BN_ULONG)-1)
            goto err;
        if ((l != 3) && (l != 7))
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else
        *ret |= DH_UNABLE_TO_CHECK_GENERATOR;

    r = BN_is_prime_ex(dh->p, BN_prime_checks, ctx, NULL);
    if (r < 0)
        goto err;
    if (!r)
        *ret |= DH_CHECK_P_NOT_PRIME;
    else if (!dh->q) {
        if (!BN_rshift1(t1, dh->p))
            goto err;
        r = BN_is_prime_ex(t1, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_P_NOT_SAFE_PRIME;
    }
    ok = 1;
 err:
    if (ctx != NULL) {
        BN_CTX_end(ctx);
        BN_CTX_free(ctx);
    }
    return (ok);
}
#else
int Cryptography_DH_check(const DH *dh, int *ret) {
    return DH_check(dh, ret);
}
#endif

/* These functions were added in OpenSSL 1.1.0f commit d0c50e80a8 */
/* Define our own to simplify support across all versions. */
#if defined(EVP_PKEY_DHX) && EVP_PKEY_DHX != -1
DH *Cryptography_d2i_DHxparams_bio(BIO *bp, DH **x) {
    return ASN1_d2i_bio_of(DH, DH_new, d2i_DHxparams, bp, x);
}
int Cryptography_i2d_DHxparams_bio(BIO *bp, DH *x) {
    return ASN1_i2d_bio_of_const(DH, i2d_DHxparams, bp, x);
}
#else
DH *(*Cryptography_d2i_DHxparams_bio)(BIO *bp, DH **x) = NULL;
int (*Cryptography_i2d_DHxparams_bio)(BIO *bp, DH *x) = NULL;
#endif




#if defined(OPENSSL_NO_EC2M)
static const long Cryptography_HAS_EC2M = 0;

int (*EC_POINT_get_affine_coordinates_GF2m)(const EC_GROUP *,
    const EC_POINT *, BIGNUM *, BIGNUM *, BN_CTX *) = NULL;

#else
static const long Cryptography_HAS_EC2M = 1;
#endif






#ifdef OPENSSL_NO_ENGINE
static const long Cryptography_HAS_ENGINE = 0;

#if CRYPTOGRAPHY_IS_BORINGSSL
typedef void UI_METHOD;
#endif

/* Despite being OPENSSL_NO_ENGINE, BoringSSL defines these symbols. */
#if !CRYPTOGRAPHY_IS_BORINGSSL
int (*ENGINE_free)(ENGINE *) = NULL;
void (*ENGINE_load_builtin_engines)(void) = NULL;
#endif

ENGINE *(*ENGINE_by_id)(const char *) = NULL;
int (*ENGINE_init)(ENGINE *) = NULL;
int (*ENGINE_finish)(ENGINE *) = NULL;
ENGINE *(*ENGINE_get_default_RAND)(void) = NULL;
int (*ENGINE_set_default_RAND)(ENGINE *) = NULL;
void (*ENGINE_unregister_RAND)(ENGINE *) = NULL;
int (*ENGINE_ctrl_cmd)(ENGINE *, const char *, long, void *,
                       void (*)(void), int) = NULL;

const char *(*ENGINE_get_id)(const ENGINE *) = NULL;
const char *(*ENGINE_get_name)(const ENGINE *) = NULL;

int (*ENGINE_ctrl_cmd_string)(ENGINE *, const char *, const char *,
                              int) = NULL;
EVP_PKEY *(*ENGINE_load_private_key)(ENGINE *, const char *, UI_METHOD *,
                                     void *) = NULL;
EVP_PKEY *(*ENGINE_load_public_key)(ENGINE *, const char *,
                                    UI_METHOD *, void *) = NULL;

#else
static const long Cryptography_HAS_ENGINE = 1;
#endif


/* This define is tied to provider support and is conditionally
   removed if Cryptography_HAS_PROVIDERS is false */
#ifndef ERR_LIB_PROV
#define ERR_LIB_PROV 0
#endif

#if !CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER || CRYPTOGRAPHY_IS_BORINGSSL
static const int EVP_R_XTS_DUPLICATED_KEYS = 0;
#endif

#if CRYPTOGRAPHY_IS_BORINGSSL
static const int ERR_LIB_PKCS12 = 0;
static const int EVP_F_EVP_ENCRYPTFINAL_EX = 0;
static const int EVP_R_BAD_DECRYPT = 0;
static const int EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH = 0;
static const int EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM = 0;
static const int PKCS12_R_PKCS12_CIPHERFINAL_ERROR = 0;
#else
static const int CIPHER_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH = 0;
#endif

/* SSL_R_UNEXPECTED_EOF_WHILE_READING is needed for pyOpenSSL
   with OpenSSL 3+ */
#if defined(SSL_R_UNEXPECTED_EOF_WHILE_READING)
#define Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING 1
#else
#define Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING 0
#define SSL_R_UNEXPECTED_EOF_WHILE_READING 0
#endif


#ifdef EVP_PKEY_DHX
const long Cryptography_HAS_EVP_PKEY_DHX = 1;
#else
const long Cryptography_HAS_EVP_PKEY_DHX = 0;
const long EVP_PKEY_DHX = -1;
#endif

EVP_MD_CTX *Cryptography_EVP_MD_CTX_new(void) {
    return EVP_MD_CTX_new();
}
void Cryptography_EVP_MD_CTX_free(EVP_MD_CTX *md) {
    EVP_MD_CTX_free(md);
}

#if CRYPTOGRAPHY_IS_LIBRESSL || defined(OPENSSL_NO_SCRYPT)
static const long Cryptography_HAS_SCRYPT = 0;
int (*EVP_PBE_scrypt)(const char *, size_t, const unsigned char *, size_t,
                      uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *,
                      size_t) = NULL;
#else
static const long Cryptography_HAS_SCRYPT = 1;
#endif

#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 1;
#else
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 0;
size_t (*EVP_PKEY_get1_tls_encodedpoint)(EVP_PKEY *, unsigned char **) = NULL;
int (*EVP_PKEY_set1_tls_encodedpoint)(EVP_PKEY *, const unsigned char *,
                                      size_t) = NULL;
#endif

#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 ||     (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL)
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 0;
int (*EVP_DigestSign)(EVP_MD_CTX *, unsigned char *, size_t *,
                      const unsigned char *tbs, size_t) = NULL;
int (*EVP_DigestVerify)(EVP_MD_CTX *, const unsigned char *, size_t,
                        const unsigned char *, size_t) = NULL;
#else
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_RAW_KEY = 0;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 0;
int (*EVP_DigestFinalXOF)(EVP_MD_CTX *, unsigned char *, size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_private_key)(int, ENGINE *, const unsigned char *,
                                       size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_public_key)(int, ENGINE *, const unsigned char *,
                                      size_t) = NULL;
int (*EVP_PKEY_get_raw_private_key)(const EVP_PKEY *, unsigned char *,
                                    size_t *) = NULL;
int (*EVP_PKEY_get_raw_public_key)(const EVP_PKEY *, unsigned char *,
                                   size_t *) = NULL;
#else
static const long Cryptography_HAS_RAW_KEY = 1;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 1;
#endif

/* OpenSSL 1.1.0+ does this define for us, but if not present we'll do it */
#if !defined(EVP_CTRL_AEAD_SET_IVLEN)
# define EVP_CTRL_AEAD_SET_IVLEN EVP_CTRL_GCM_SET_IVLEN
#endif
#if !defined(EVP_CTRL_AEAD_GET_TAG)
# define EVP_CTRL_AEAD_GET_TAG EVP_CTRL_GCM_GET_TAG
#endif
#if !defined(EVP_CTRL_AEAD_SET_TAG)
# define EVP_CTRL_AEAD_SET_TAG EVP_CTRL_GCM_SET_TAG
#endif

/* This is tied to X25519 support so we reuse the Cryptography_HAS_X25519
   conditional to remove it. OpenSSL 1.1.0 didn't have this define, but
   1.1.1 will when it is released. We can remove this in the distant
   future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X25519
#define EVP_PKEY_X25519 NID_X25519
#endif

/* This is tied to X448 support so we reuse the Cryptography_HAS_X448
   conditional to remove it. OpenSSL 1.1.1 adds this define.  We can remove
   this in the distant future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X448
#define EVP_PKEY_X448 NID_X448
#endif

/* This is tied to ED25519 support so we reuse the Cryptography_HAS_ED25519
   conditional to remove it. */
#ifndef EVP_PKEY_ED25519
#define EVP_PKEY_ED25519 NID_ED25519
#endif

/* This is tied to ED448 support so we reuse the Cryptography_HAS_ED448
   conditional to remove it. */
#ifndef EVP_PKEY_ED448
#define EVP_PKEY_ED448 NID_ED448
#endif

/* This is tied to poly1305 support so we reuse the Cryptography_HAS_POLY1305
   conditional to remove it. */
#ifndef EVP_PKEY_POLY1305
#define EVP_PKEY_POLY1305 NID_poly1305
#endif

#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
static const long Cryptography_HAS_300_FIPS = 1;
static const long Cryptography_HAS_300_EVP_CIPHER = 1;
#else
static const long Cryptography_HAS_300_FIPS = 0;
static const long Cryptography_HAS_300_EVP_CIPHER = 0;
int (*EVP_default_properties_is_fips_enabled)(OSSL_LIB_CTX *) = NULL;
int (*EVP_default_properties_enable_fips)(OSSL_LIB_CTX *, int) = NULL;
EVP_CIPHER * (*EVP_CIPHER_fetch)(OSSL_LIB_CTX *, const char *,
                                 const char *) = NULL;
void (*EVP_CIPHER_free)(EVP_CIPHER *) = NULL;
#endif

#if CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_EVP_PKEY_DH = 0;
int (*EVP_PKEY_set1_DH)(EVP_PKEY *, DH *) = NULL;
#else
static const long Cryptography_HAS_EVP_PKEY_DH = 1;
#endif

// This can be removed when we drop OpenSSL 1.1.0 support
// OPENSSL_LESS_THAN_111
#if !defined(EVP_PKEY_RSA_PSS)
#define EVP_PKEY_RSA_PSS 912
#endif


#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350 || CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
static const long Cryptography_HAS_FIPS = 0;
int (*FIPS_mode_set)(int) = NULL;
int (*FIPS_mode)(void) = NULL;
#else
static const long Cryptography_HAS_FIPS = 1;
#endif




#ifndef NID_ED25519
static const long Cryptography_HAS_ED25519 = 0;
static const int NID_ED25519 = 0;
#else
static const long Cryptography_HAS_ED25519 = 1;
#endif
#ifndef NID_ED448
static const long Cryptography_HAS_ED448 = 0;
static const int NID_ED448 = 0;
#else
static const long Cryptography_HAS_ED448 = 1;
#endif
#ifndef NID_poly1305
static const long Cryptography_HAS_POLY1305 = 0;
static const int NID_poly1305 = 0;
#else
static const long Cryptography_HAS_POLY1305 = 1;
#endif





/* osurandom engine
 *
 * Windows         CryptGenRandom()
 * macOS >= 10.12  getentropy()
 * OpenBSD 5.6+    getentropy()
 * other BSD       getentropy() if SYS_getentropy is defined
 * Linux 3.17+     getrandom() with fallback to /dev/urandom
 * other           /dev/urandom with cached fd
 *
 * The /dev/urandom, getrandom and getentropy code is derived from Python's
 * Python/random.c, written by Antoine Pitrou and Victor Stinner.
 *
 * Copyright 2001-2016 Python Software Foundation; All Rights Reserved.
 */

#ifdef __linux__
#include <poll.h>
#endif

#if CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE
/* OpenSSL has ENGINE support and is older than 1.1.1d (the first version that
 * properly implements fork safety in its RNG) so build the engine. */
static const char *Cryptography_osrandom_engine_id = "osrandom";

/****************************************************************************
 * Windows
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine CryptGenRandom()";
static HCRYPTPROV hCryptProv = 0;

static int osrandom_init(ENGINE *e) {
    if (hCryptProv != 0) {
        return 1;
    }
    if (CryptAcquireContext(&hCryptProv, NULL, NULL,
                            PROV_RSA_FULL, CRYPT_VERIFYCONTEXT)) {
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_INIT,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    if (hCryptProv == 0) {
        return 0;
    }

    if (!CryptGenRandom(hCryptProv, (DWORD)size, buffer)) {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM,
            __FILE__, __LINE__
        );
        return 0;
    }
    return 1;
}

static int osrandom_finish(ENGINE *e) {
    if (CryptReleaseContext(hCryptProv, 0)) {
        hCryptProv = 0;
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_FINISH,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_status(void) {
    return hCryptProv != 0;
}

static const char *osurandom_get_implementation(void) {
    return "CryptGenRandom";
}

#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM */

/****************************************************************************
 * /dev/urandom helpers for all non-BSD Unix platforms
 */
#ifdef CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM

static struct {
    int fd;
    dev_t st_dev;
    ino_t st_ino;
} urandom_cache = { -1 };

static int open_cloexec(const char *path) {
    int open_flags = O_RDONLY;
#ifdef O_CLOEXEC
    open_flags |= O_CLOEXEC;
#endif

    int fd = open(path, open_flags);
    if (fd == -1) {
        return -1;
    }

#ifndef O_CLOEXEC
    int flags = fcntl(fd, F_GETFD);
    if (flags == -1) {
        return -1;
    }
    if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1) {
        return -1;
    }
#endif
    return fd;
}

#ifdef __linux__
/* On Linux, we open("/dev/random") and use poll() to wait until it's readable
 * before we read from /dev/urandom, this ensures that we don't read from
 * /dev/urandom before the kernel CSPRNG is initialized. This isn't necessary on
 * other platforms because they don't have the same _bug_ as Linux does with
 * /dev/urandom and early boot. */
static int wait_on_devrandom(void) {
    struct pollfd pfd = {};
    int ret = 0;
    int random_fd = open_cloexec("/dev/random");
    if (random_fd < 0) {
        return -1;
    }
    pfd.fd = random_fd;
    pfd.events = POLLIN;
    pfd.revents = 0;
    do {
        ret = poll(&pfd, 1, -1);
    } while (ret < 0 && (errno == EINTR || errno == EAGAIN));
    close(random_fd);
    return ret;
}
#endif

/* return -1 on error */
static int dev_urandom_fd(void) {
    int fd = -1;
    struct stat st;

    /* Check that fd still points to the correct device */
    if (urandom_cache.fd >= 0) {
        if (fstat(urandom_cache.fd, &st)
                || st.st_dev != urandom_cache.st_dev
                || st.st_ino != urandom_cache.st_ino) {
            /* Somebody replaced our FD. Invalidate our cache but don't
             * close the fd. */
            urandom_cache.fd = -1;
        }
    }
    if (urandom_cache.fd < 0) {
#ifdef __linux__
        if (wait_on_devrandom() < 0) {
            goto error;
        }
#endif

        fd = open_cloexec("/dev/urandom");
        if (fd < 0) {
            goto error;
        }
        if (fstat(fd, &st)) {
            goto error;
        }
        /* Another thread initialized the fd */
        if (urandom_cache.fd >= 0) {
            close(fd);
            return urandom_cache.fd;
        }
        urandom_cache.st_dev = st.st_dev;
        urandom_cache.st_ino = st.st_ino;
        urandom_cache.fd = fd;
    }
    return urandom_cache.fd;

  error:
    if (fd != -1) {
        close(fd);
    }
    ERR_Cryptography_OSRandom_error(
        CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD,
        CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED,
        __FILE__, __LINE__
    );
    return -1;
}

static int dev_urandom_read(unsigned char *buffer, int size) {
    int fd;
    int n;

    fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }

    while (size > 0) {
        do {
            n = (int)read(fd, buffer, (size_t)size);
        } while (n < 0 && errno == EINTR);

        if (n <= 0) {
            ERR_Cryptography_OSRandom_error(
                CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ,
                CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED,
                __FILE__, __LINE__
            );
            return 0;
        }
        buffer += n;
        size -= n;
    }
    return 1;
}

static void dev_urandom_close(void) {
    if (urandom_cache.fd >= 0) {
        int fd;
        struct stat st;

        if (fstat(urandom_cache.fd, &st)
                && st.st_dev == urandom_cache.st_dev
                && st.st_ino == urandom_cache.st_ino) {
            fd = urandom_cache.fd;
            urandom_cache.fd = -1;
            close(fd);
        }
    }
}
#endif /* CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM */

/****************************************************************************
 * BSD getentropy
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getentropy()";

static int getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT;

static int osrandom_init(ENGINE *e) {
#if !defined(__APPLE__)
    getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
#else
    if (__builtin_available(macOS 10.12, *)) {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
    } else {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK;
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
#endif
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    int len;
    int res;

    switch(getentropy_works) {
#if defined(__APPLE__)
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return dev_urandom_read(buffer, size);
#endif
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        while (size > 0) {
            /* OpenBSD and macOS restrict maximum buffer size to 256. */
            len = size > 256 ? 256 : size;
/* on mac, availability is already checked using `__builtin_available` above */
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunguarded-availability"
            res = getentropy(buffer, (size_t)len);
#pragma clang diagnostic pop
            if (res < 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += len;
            size -= len;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    return 1;
}

static int osrandom_rand_status(void) {
    return 1;
}

static const char *osurandom_get_implementation(void) {
    switch(getentropy_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        return "getentropy";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY */

/****************************************************************************
 * Linux getrandom engine with fallback to dev_urandom
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getrandom()";

static int getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT;

static int osrandom_init(ENGINE *e) {
    /* We try to detect working getrandom until we succeed. */
    if (getrandom_works != CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS) {
        long n;
        char dest[1];
        /* if the kernel CSPRNG is not initialized this will block */
        n = syscall(SYS_getrandom, dest, sizeof(dest), 0);
        if (n == sizeof(dest)) {
            getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS;
        } else {
            int e = errno;
            switch(e) {
            case ENOSYS:
                /* Fallback: Kernel does not support the syscall. */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            case EPERM:
                /* Fallback: seccomp prevents syscall */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            default:
                /* EINTR cannot occur for buflen < 256. */
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_INIT,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED,
                    "errno", e
                );
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED;
                break;
            }
        }
    }

    /* fallback to dev urandom */
    if (getrandom_works == CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK) {
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    long n;

    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return dev_urandom_read(buffer, size);
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        while (size > 0) {
            do {
                n = syscall(SYS_getrandom, buffer, size, 0);
            } while (n < 0 && errno == EINTR);

            if (n <= 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += n;
            size -= (int)n;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return urandom_cache.fd >= 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return 1;
    }
    __builtin_unreachable();
}

static const char *osurandom_get_implementation(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return "<failed>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return "<not initialized>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return "getrandom";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM */

/****************************************************************************
 * dev_urandom engine for all remaining platforms
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine /dev/urandom";

static int osrandom_init(ENGINE *e) {
    int fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    return dev_urandom_read(buffer, size);
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    return urandom_cache.fd >= 0;
}

static const char *osurandom_get_implementation(void) {
    return "/dev/urandom";
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM */

/****************************************************************************
 * ENGINE boiler plate
 */

/* This replicates the behavior of the OpenSSL FIPS RNG, which returns a
   -1 in the event that there is an error when calling RAND_pseudo_bytes. */
static int osrandom_pseudo_rand_bytes(unsigned char *buffer, int size) {
    int res = osrandom_rand_bytes(buffer, size);
    if (res == 0) {
        return -1;
    } else {
        return res;
    }
}

static RAND_METHOD osrandom_rand = {
    NULL,
    osrandom_rand_bytes,
    NULL,
    NULL,
    osrandom_pseudo_rand_bytes,
    osrandom_rand_status,
};

static const ENGINE_CMD_DEFN osrandom_cmd_defns[] = {
    {CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION,
     "get_implementation",
     "Get CPRNG implementation.",
     ENGINE_CMD_FLAG_NO_INPUT},
     {0, NULL, NULL, 0}
};

static int osrandom_ctrl(ENGINE *e, int cmd, long i, void *p, void (*f) (void)) {
    const char *name;
    size_t len;

    switch (cmd) {
    case CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION:
        /* i: buffer size, p: char* buffer */
        name = osurandom_get_implementation();
        len = strlen(name);
        if ((p == NULL) && (i == 0)) {
            /* return required buffer len */
            return (int)len;
        }
        if ((p == NULL) || i < 0 || ((size_t)i <= len)) {
            /* no buffer or buffer too small */
            ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_INVALID_ARGUMENT);
            return 0;
        }
        strcpy((char *)p, name);
        return (int)len;
    default:
        ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_CTRL_COMMAND_NOT_IMPLEMENTED);
        return 0;
    }
}

/* error reporting */
#define ERR_FUNC(func) ERR_PACK(0, func, 0)
#define ERR_REASON(reason) ERR_PACK(0, 0, reason)

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_lib_name[] = {
    {0, "osrandom_engine"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_funcs[] = {
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_INIT),
     "osrandom_init"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES),
     "osrandom_rand_bytes"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_FINISH),
     "osrandom_finish"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD),
     "dev_urandom_fd"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ),
     "dev_urandom_read"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_reasons[] = {
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT),
     "CryptAcquireContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM),
     "CryptGenRandom() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT),
     "CryptReleaseContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED),
     "getentropy() failed"},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED),
     "open('/dev/urandom') failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED),
     "Reading from /dev/urandom fd failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED),
     "getrandom() initialization failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED),
     "getrandom() initialization failed with unexpected errno."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED),
     "getrandom() syscall failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT),
     "getrandom() engine was not properly initialized."},
    {0, NULL}
};

static int Cryptography_OSRandom_lib_error_code = 0;

static void ERR_load_Cryptography_OSRandom_strings(void)
{
    if (Cryptography_OSRandom_lib_error_code == 0) {
        Cryptography_OSRandom_lib_error_code = ERR_get_next_error_library();
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_lib_name);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_funcs);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_reasons);
    }
}

static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line)
{
    ERR_PUT_error(Cryptography_OSRandom_lib_error_code, function, reason,
                  file, line);
}

/* Returns 1 if successfully added, 2 if engine has previously been added,
   and 0 for error. */
int Cryptography_add_osrandom_engine(void) {
    ENGINE *e;

    ERR_load_Cryptography_OSRandom_strings();

    e = ENGINE_by_id(Cryptography_osrandom_engine_id);
    if (e != NULL) {
        ENGINE_free(e);
        return 2;
    } else {
        ERR_clear_error();
    }

    e = ENGINE_new();
    if (e == NULL) {
        return 0;
    }
    if (!ENGINE_set_id(e, Cryptography_osrandom_engine_id) ||
            !ENGINE_set_name(e, Cryptography_osrandom_engine_name) ||
            !ENGINE_set_RAND(e, &osrandom_rand) ||
            !ENGINE_set_init_function(e, osrandom_init) ||
            !ENGINE_set_finish_function(e, osrandom_finish) ||
            !ENGINE_set_cmd_defns(e, osrandom_cmd_defns) ||
            !ENGINE_set_ctrl_function(e, osrandom_ctrl)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_add(e)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_free(e)) {
        return 0;
    }

    return 1;
}

#else
/* If OpenSSL has no ENGINE support then we don't want
 * to compile the osrandom engine, but we do need some
 * placeholders */
static const char *Cryptography_osrandom_engine_id = "no-engine-support";
static const char *Cryptography_osrandom_engine_name = "osrandom_engine disabled";

int Cryptography_add_osrandom_engine(void) {
    return 0;
}

#endif


#if !defined(EVP_PKEY_DHX) || EVP_PKEY_DHX == -1
int (*PEM_write_bio_DHxparams)(BIO *, DH *) = NULL;
#endif






// BoringSSL doesn't define this constant, but the value is used for
// automatic salt length computation as in OpenSSL and LibreSSL
#if !defined(RSA_PSS_SALTLEN_AUTO)
#define RSA_PSS_SALTLEN_AUTO -2
#endif


// This symbol is being preserved because removing it will break users with
// pyOpenSSL < 19.1 and pip < 20.x. We need to leave this in place until those
// users have upgraded. PersistentlyDeprecated2020
static const long Cryptography_HAS_TLSEXT_HOSTNAME = 1;

#ifdef OPENSSL_NO_ENGINE
int (*SSL_CTX_set_client_cert_engine)(SSL_CTX *, ENGINE *) = NULL;
#endif

#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350 || CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_VERIFIED_CHAIN = 0;
Cryptography_STACK_OF_X509 *(*SSL_get0_verified_chain)(const SSL *) = NULL;
#else
static const long Cryptography_HAS_VERIFIED_CHAIN = 1;
#endif

#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350 ||     (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL)
static const long Cryptography_HAS_KEYLOG = 0;
void (*SSL_CTX_set_keylog_callback)(SSL_CTX *,
                                    void (*) (const SSL *, const char *)
                                    ) = NULL;
void (*(*SSL_CTX_get_keylog_callback)(SSL_CTX *))(
                                                  const SSL *,
                                                  const char *
                                                  ) = NULL;
#else
static const long Cryptography_HAS_KEYLOG = 1;
#endif

static const long Cryptography_HAS_SECURE_RENEGOTIATION = 1;

#ifdef OPENSSL_NO_SSL3_METHOD
static const long Cryptography_HAS_SSL3_METHOD = 0;
SSL_METHOD* (*SSLv3_method)(void) = NULL;
SSL_METHOD* (*SSLv3_client_method)(void) = NULL;
SSL_METHOD* (*SSLv3_server_method)(void) = NULL;
#else
static const long Cryptography_HAS_SSL3_METHOD = 1;
#endif

static const long Cryptography_HAS_RELEASE_BUFFERS = 1;
static const long Cryptography_HAS_OP_NO_COMPRESSION = 1;
static const long Cryptography_HAS_TLSv1_1 = 1;
static const long Cryptography_HAS_TLSv1_2 = 1;
static const long Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING = 1;
static const long Cryptography_HAS_SSL_OP_NO_TICKET = 1;
static const long Cryptography_HAS_SSL_SET_SSL_CTX = 1;
static const long Cryptography_HAS_NEXTPROTONEG = 0;
static const long Cryptography_HAS_ALPN = 1;

#ifdef SSL_OP_NO_RENEGOTIATION
static const long Cryptography_HAS_OP_NO_RENEGOTIATION = 1;
#else
static const long Cryptography_HAS_OP_NO_RENEGOTIATION = 0;
static const long SSL_OP_NO_RENEGOTIATION = 0;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
void (*SSL_CTX_set_cert_cb)(SSL_CTX *, int (*)(SSL *, void *), void *) = NULL;
void (*SSL_set_cert_cb)(SSL *, int (*)(SSL *, void *), void *) = NULL;
static const long Cryptography_HAS_SET_CERT_CB = 0;
#else
static const long Cryptography_HAS_SET_CERT_CB = 1;
#endif

static const long Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS = 1;

/* in OpenSSL 1.1.0 the SSL_ST values were renamed to TLS_ST and several were
   removed */
#if CRYPTOGRAPHY_IS_LIBRESSL || CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_SSL_ST = 1;
#else
static const long Cryptography_HAS_SSL_ST = 0;
static const long SSL_ST_BEFORE = 0;
static const long SSL_ST_OK = 0;
static const long SSL_ST_INIT = 0;
static const long SSL_ST_RENEGOTIATE = 0;
#endif
#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_TLS_ST = 1;
#else
static const long Cryptography_HAS_TLS_ST = 0;
static const long TLS_ST_BEFORE = 0;
static const long TLS_ST_OK = 0;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL || CRYPTOGRAPHY_IS_BORINGSSL
#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_332
static const long SSL_OP_NO_DTLSv1 = 0;
static const long SSL_OP_NO_DTLSv1_2 = 0;
#endif
long (*DTLS_set_link_mtu)(SSL *, long) = NULL;
long (*DTLS_get_link_min_mtu)(SSL *) = NULL;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 || CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_DTLS_GET_DATA_MTU = 0;
size_t (*DTLS_get_data_mtu)(SSL *) = NULL;
#else
static const long Cryptography_HAS_DTLS_GET_DATA_MTU = 1;
#endif

static const long Cryptography_HAS_DTLS = 1;
/* Wrap DTLSv1_get_timeout to avoid cffi to handle a 'struct timeval'. */
long Cryptography_DTLSv1_get_timeout(SSL *ssl, time_t *ptv_sec,
                                     long *ptv_usec) {
    struct timeval tv = { 0 };
    long r = DTLSv1_get_timeout(ssl, &tv);

    if (r == 1) {
        if (ptv_sec) {
            *ptv_sec = tv.tv_sec;
        }

        if (ptv_usec) {
            *ptv_usec = tv.tv_usec;
        }
    }

    return r;
}

#if CRYPTOGRAPHY_IS_LIBRESSL || defined(OPENSSL_NO_PSK)
static const long Cryptography_HAS_PSK = 0;
int (*SSL_CTX_use_psk_identity_hint)(SSL_CTX *, const char *) = NULL;
void (*SSL_CTX_set_psk_server_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
void (*SSL_CTX_set_psk_client_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            char *,
                                            unsigned int,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
#else
static const long Cryptography_HAS_PSK = 1;
#endif

#if !CRYPTOGRAPHY_IS_LIBRESSL && !CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_CUSTOM_EXT = 1;
#else
static const long Cryptography_HAS_CUSTOM_EXT = 0;
typedef int (*custom_ext_add_cb)(SSL *, unsigned int,
                                 const unsigned char **,
                                 size_t *, int *,
                                 void *);
typedef void (*custom_ext_free_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   void *);
typedef int (*custom_ext_parse_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   size_t, int *,
                                   void *);
int (*SSL_CTX_add_client_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;
int (*SSL_CTX_add_server_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;
int (*SSL_extension_supported)(unsigned int) = NULL;
#endif

#ifndef OPENSSL_NO_SRTP
static const long Cryptography_HAS_SRTP = 1;
#else
static const long Cryptography_HAS_SRTP = 0;
int (*SSL_CTX_set_tlsext_use_srtp)(SSL_CTX *, const char *) = NULL;
int (*SSL_set_tlsext_use_srtp)(SSL *, const char *) = NULL;
SRTP_PROTECTION_PROFILE * (*SSL_get_selected_srtp_profile)(SSL *) = NULL;
#endif

#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 ||     (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL)
static const long Cryptography_HAS_TLSv1_3 = 0;
static const long TLS1_3_VERSION = 0;
static const long SSL_OP_NO_TLSv1_3 = 0;
#else
static const long Cryptography_HAS_TLSv1_3 = 1;
#endif

#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 ||     (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL) ||     CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_TLSv1_3_FUNCTIONS = 0;

static const long SSL_VERIFY_POST_HANDSHAKE = 0;
int (*SSL_CTX_set_ciphersuites)(SSL_CTX *, const char *) = NULL;
int (*SSL_verify_client_post_handshake)(SSL *) = NULL;
void (*SSL_CTX_set_post_handshake_auth)(SSL_CTX *, int) = NULL;
void (*SSL_set_post_handshake_auth)(SSL *, int) = NULL;
uint32_t (*SSL_SESSION_get_max_early_data)(const SSL_SESSION *) = NULL;
int (*SSL_write_early_data)(SSL *, const void *, size_t, size_t *) = NULL;
int (*SSL_read_early_data)(SSL *, void *, size_t, size_t *) = NULL;
int (*SSL_CTX_set_max_early_data)(SSL_CTX *, uint32_t) = NULL;
#else
static const long Cryptography_HAS_TLSv1_3_FUNCTIONS = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_GET_PROTO_VERSION = 0;

long (*SSL_CTX_get_min_proto_version)(SSL_CTX *) = NULL;
long (*SSL_CTX_get_max_proto_version)(SSL_CTX *) = NULL;
long (*SSL_get_min_proto_version)(SSL *) = NULL;
long (*SSL_get_max_proto_version)(SSL *) = NULL;
#else
static const long Cryptography_HAS_GET_PROTO_VERSION = 1;
#endif

#if CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_SSL_COOKIE = 0;

static const long SSL_OP_COOKIE_EXCHANGE = 0;
int (*DTLSv1_listen)(SSL *, BIO_ADDR *) = NULL;
void (*SSL_CTX_set_cookie_generate_cb)(SSL_CTX *,
                                       int (*)(
                                           SSL *,
                                           unsigned char *,
                                           unsigned int *
                                       )) = NULL;
void (*SSL_CTX_set_cookie_verify_cb)(SSL_CTX *,
                                       int (*)(
                                           SSL *,
                                           const unsigned char *,
                                           unsigned int
                                       )) = NULL;
#else
static const long Cryptography_HAS_SSL_COOKIE = 1;
#endif
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 ||     CRYPTOGRAPHY_IS_LIBRESSL || CRYPTOGRAPHY_IS_BORINGSSL ||     defined(OPENSSL_NO_PSK)
static const long Cryptography_HAS_PSK_TLSv1_3 = 0;
void (*SSL_CTX_set_psk_find_session_callback)(SSL_CTX *,
                                           int (*)(
                                               SSL *,
                                               const unsigned char *,
                                               size_t,
                                               SSL_SESSION **
                                           )) = NULL;
void (*SSL_CTX_set_psk_use_session_callback)(SSL_CTX *,
                                          int (*)(
                                              SSL *,
                                              const EVP_MD *,
                                              const unsigned char **,
                                              size_t *,
                                              SSL_SESSION **
                                          )) = NULL;
#if CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 || CRYPTOGRAPHY_IS_BORINGSSL
    const SSL_CIPHER *(*SSL_CIPHER_find)(SSL *, const unsigned char *) = NULL;
#endif
int (*SSL_SESSION_set1_master_key)(SSL_SESSION *, const unsigned char *,
                                   size_t) = NULL;
int (*SSL_SESSION_set_cipher)(SSL_SESSION *, const SSL_CIPHER *) = NULL;
#if !CRYPTOGRAPHY_IS_BORINGSSL
    int (*SSL_SESSION_set_protocol_version)(SSL_SESSION *, int) = NULL;
#endif
SSL_SESSION *(*Cryptography_SSL_SESSION_new)(void) = NULL;
#else
static const long Cryptography_HAS_PSK_TLSv1_3 = 1;
SSL_SESSION *Cryptography_SSL_SESSION_new(void) {
    return SSL_SESSION_new();
}
#endif


/* This function is no longer used by pyOpenSSL >= 22.0 */
X509_REVOKED *Cryptography_X509_REVOKED_dup(X509_REVOKED *rev) {
    return X509_REVOKED_dup(rev);
}






#if CRYPTOGRAPHY_IS_LIBRESSL && CRYPTOGRAPHY_LIBRESSL_LESS_THAN_322
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 0;
#ifndef X509_CHECK_FLAG_NEVER_CHECK_SUBJECT
static const long X509_CHECK_FLAG_NEVER_CHECK_SUBJECT = 0;
#endif
#else
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 1;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 0;
typedef void *X509_STORE_CTX_get_issuer_fn;
X509_STORE_CTX_get_issuer_fn (*X509_STORE_get_get_issuer)(X509_STORE *) = NULL;
void (*X509_STORE_set_get_issuer)(X509_STORE *,
                                  X509_STORE_CTX_get_issuer_fn) = NULL;
#else
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 1;
#endif

#if CRYPTOGRAPHY_IS_BORINGSSL
static const long X509_V_FLAG_NO_CHECK_TIME = 0;
#endif


#if CRYPTOGRAPHY_IS_BORINGSSL
static const long Cryptography_HAS_PKCS7_FUNCS = 0;

int (*SMIME_write_PKCS7)(BIO *, PKCS7 *, BIO *, int) = NULL;
int (*PEM_write_bio_PKCS7_stream)(BIO *, PKCS7 *, BIO *, int) = NULL;
PKCS7_SIGNER_INFO *(*PKCS7_sign_add_signer)(PKCS7 *, X509 *, EVP_PKEY *,
                                            const EVP_MD *, int) = NULL;
int (*PKCS7_final)(PKCS7 *, BIO *, int);
int (*PKCS7_verify)(PKCS7 *, Cryptography_STACK_OF_X509 *, X509_STORE *, BIO *,
                    BIO *, int) = NULL;
PKCS7 *(*SMIME_read_PKCS7)(BIO *, BIO **) = NULL;
Cryptography_STACK_OF_X509 *(*PKCS7_get0_signers)(PKCS7 *,
                                                  Cryptography_STACK_OF_X509 *,
                                                  int) = NULL;
#else
static const long Cryptography_HAS_PKCS7_FUNCS = 1;
#endif


typedef struct {
    char *password;
    int length;
    int called;
    int error;
    int maxsize;
} CRYPTOGRAPHY_PASSWORD_DATA;

int Cryptography_pem_password_cb(char *buf, int size,
                                  int rwflag, void *userdata) {
    /* The password cb is only invoked if OpenSSL decides the private
       key is encrypted. So this path only occurs if it needs a password */
    CRYPTOGRAPHY_PASSWORD_DATA *st = (CRYPTOGRAPHY_PASSWORD_DATA *)userdata;
    st->called += 1;
    st->maxsize = size;
    if (st->length == 0) {
        st->error = -1;
        return 0;
    } else if (st->length < size) {
        memcpy(buf, st->password, st->length);
        return st->length;
    } else {
        st->error = -2;
        return 0;
    }
}

#define CRYPTOGRAPHY_PACKAGE_VERSION "37.0.4"

/************************************************************/

static void *_cffi_types[] = {
/*  0 */ _CFFI_OP(_CFFI_OP_FUNCTION, 550), // ASN1_ENUMERATED *()(void)
/*  1 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  2 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2067), // ASN1_GENERALIZEDTIME *()(ASN1_OCTET_STRING *, ASN1_GENERALIZEDTIME * *)
/*  3 */ _CFFI_OP(_CFFI_OP_POINTER, 2444), // ASN1_OCTET_STRING *
/*  4 */ _CFFI_OP(_CFFI_OP_POINTER, 2067), // ASN1_GENERALIZEDTIME * *
/*  5 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  6 */ _CFFI_OP(_CFFI_OP_FUNCTION, 8), // ASN1_INTEGER *()(BIGNUM *, ASN1_INTEGER *)
/*  7 */ _CFFI_OP(_CFFI_OP_POINTER, 2447), // BIGNUM *
/*  8 */ _CFFI_OP(_CFFI_OP_POINTER, 2442), // ASN1_INTEGER *
/*  9 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 10 */ _CFFI_OP(_CFFI_OP_FUNCTION, 8), // ASN1_INTEGER *()(X509 *)
/* 11 */ _CFFI_OP(_CFFI_OP_POINTER, 2514), // X509 *
/* 12 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 13 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2441), // ASN1_INTEGER const *()(X509_REVOKED const *)
/* 14 */ _CFFI_OP(_CFFI_OP_POINTER, 2526), // X509_REVOKED const *
/* 15 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 16 */ _CFFI_OP(_CFFI_OP_FUNCTION, 419), // ASN1_OBJECT *()(X509_EXTENSION *)
/* 17 */ _CFFI_OP(_CFFI_OP_POINTER, 2520), // X509_EXTENSION *
/* 18 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 19 */ _CFFI_OP(_CFFI_OP_FUNCTION, 419), // ASN1_OBJECT *()(X509_NAME_ENTRY *)
/* 20 */ _CFFI_OP(_CFFI_OP_POINTER, 2523), // X509_NAME_ENTRY *
/* 21 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 22 */ _CFFI_OP(_CFFI_OP_FUNCTION, 419), // ASN1_OBJECT *()(char const *, int)
/* 23 */ _CFFI_OP(_CFFI_OP_POINTER, 2530), // char const *
/* 24 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7), // int
/* 25 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 26 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(ASN1_OCTET_STRING *, long)
/* 27 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 28 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9), // long
/* 29 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 30 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(X509 const *)
/* 31 */ _CFFI_OP(_CFFI_OP_POINTER, 2514), // X509 const *
/* 32 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 33 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(X509_CRL *)
/* 34 */ _CFFI_OP(_CFFI_OP_POINTER, 2519), // X509_CRL *
/* 35 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 36 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(X509_EXTENSION *)
/* 37 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 38 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 39 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(X509_NAME_ENTRY *)
/* 40 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 41 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 42 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3), // ASN1_OCTET_STRING *()(void)
/* 43 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 44 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1532), // ASN1_OCTET_STRING const *()(X509_CRL const *)
/* 45 */ _CFFI_OP(_CFFI_OP_POINTER, 2519), // X509_CRL const *
/* 46 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 47 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1532), // ASN1_OCTET_STRING const *()(X509_REVOKED const *)
/* 48 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 49 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 50 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(ASN1_INTEGER *, BIGNUM *)
/* 51 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 52 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 53 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 54 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 55 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 56 */ _CFFI_OP(_CFFI_OP_POINTER, 2447), // BIGNUM const *
/* 57 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 58 */ _CFFI_OP(_CFFI_OP_POINTER, 2452), // BN_CTX *
/* 59 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 60 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(BIGNUM const *)
/* 61 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 62 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 63 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(BN_CTX *)
/* 64 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 65 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 66 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(unsigned char const *, int, BIGNUM *)
/* 67 */ _CFFI_OP(_CFFI_OP_POINTER, 2533), // unsigned char const *
/* 68 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 69 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 70 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 71 */ _CFFI_OP(_CFFI_OP_FUNCTION, 7), // BIGNUM *()(void)
/* 72 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 73 */ _CFFI_OP(_CFFI_OP_FUNCTION, 56), // BIGNUM const *()(EC_KEY const *)
/* 74 */ _CFFI_OP(_CFFI_OP_POINTER, 2469), // EC_KEY const *
/* 75 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 76 */ _CFFI_OP(_CFFI_OP_FUNCTION, 56), // BIGNUM const *()(void)
/* 77 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 78 */ _CFFI_OP(_CFFI_OP_FUNCTION, 129), // BIO *()(BIO_METHOD *)
/* 79 */ _CFFI_OP(_CFFI_OP_POINTER, 2451), // BIO_METHOD *
/* 80 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 81 */ _CFFI_OP(_CFFI_OP_FUNCTION, 129), // BIO *()(char const *, char const *)
/* 82 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 83 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 84 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 85 */ _CFFI_OP(_CFFI_OP_FUNCTION, 129), // BIO *()(void const *, int)
/* 86 */ _CFFI_OP(_CFFI_OP_POINTER, 2534), // void const *
/* 87 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 88 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 89 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1287), // BIO_ADDR *()(void)
/* 90 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 91 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2450), // BIO_METHOD const *()(void)
/* 92 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 93 */ _CFFI_OP(_CFFI_OP_FUNCTION, 58), // BN_CTX *()(void)
/* 94 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 95 */ _CFFI_OP(_CFFI_OP_FUNCTION, 603), // BN_MONT_CTX *()(void)
/* 96 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 97 */ _CFFI_OP(_CFFI_OP_FUNCTION, 790), // CMAC_CTX *()(void)
/* 98 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 99 */ _CFFI_OP(_CFFI_OP_FUNCTION, 101), // Cryptography_STACK_OF_X509 *()(PKCS7 *, Cryptography_STACK_OF_X509 *, int)
/* 100 */ _CFFI_OP(_CFFI_OP_POINTER, 2489), // PKCS7 *
/* 101 */ _CFFI_OP(_CFFI_OP_POINTER, 2460), // Cryptography_STACK_OF_X509 *
/* 102 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 103 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 104 */ _CFFI_OP(_CFFI_OP_FUNCTION, 101), // Cryptography_STACK_OF_X509 *()(SSL const *)
/* 105 */ _CFFI_OP(_CFFI_OP_POINTER, 2507), // SSL const *
/* 106 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 107 */ _CFFI_OP(_CFFI_OP_FUNCTION, 101), // Cryptography_STACK_OF_X509 *()(X509_STORE_CTX *)
/* 108 */ _CFFI_OP(_CFFI_OP_POINTER, 2528), // X509_STORE_CTX *
/* 109 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 110 */ _CFFI_OP(_CFFI_OP_FUNCTION, 101), // Cryptography_STACK_OF_X509 *()(void)
/* 111 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 112 */ _CFFI_OP(_CFFI_OP_FUNCTION, 440), // Cryptography_STACK_OF_X509_NAME *()(SSL const *)
/* 113 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 114 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 115 */ _CFFI_OP(_CFFI_OP_FUNCTION, 440), // Cryptography_STACK_OF_X509_NAME *()(char const *)
/* 116 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 117 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 118 */ _CFFI_OP(_CFFI_OP_FUNCTION, 440), // Cryptography_STACK_OF_X509_NAME *()(void)
/* 119 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 120 */ _CFFI_OP(_CFFI_OP_FUNCTION, 828), // Cryptography_STACK_OF_X509_NAME_ENTRY *()(void)
/* 121 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 122 */ _CFFI_OP(_CFFI_OP_FUNCTION, 469), // Cryptography_STACK_OF_X509_OBJECT *()(X509_STORE *)
/* 123 */ _CFFI_OP(_CFFI_OP_POINTER, 2527), // X509_STORE *
/* 124 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 125 */ _CFFI_OP(_CFFI_OP_FUNCTION, 485), // Cryptography_STACK_OF_X509_REVOKED *()(X509_CRL *)
/* 126 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 127 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 128 */ _CFFI_OP(_CFFI_OP_FUNCTION, 139), // DH *()(BIO *, DH * *)
/* 129 */ _CFFI_OP(_CFFI_OP_POINTER, 2448), // BIO *
/* 130 */ _CFFI_OP(_CFFI_OP_POINTER, 139), // DH * *
/* 131 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 132 */ _CFFI_OP(_CFFI_OP_FUNCTION, 139), // DH *()(BIO *, DH * *, int(*)(char *, int, int, void *), void *)
/* 133 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 134 */ _CFFI_OP(_CFFI_OP_NOOP, 130),
/* 135 */ _CFFI_OP(_CFFI_OP_POINTER, 1747), // int(*)(char *, int, int, void *)
/* 136 */ _CFFI_OP(_CFFI_OP_POINTER, 2534), // void *
/* 137 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 138 */ _CFFI_OP(_CFFI_OP_FUNCTION, 139), // DH *()(DH *)
/* 139 */ _CFFI_OP(_CFFI_OP_POINTER, 2466), // DH *
/* 140 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 141 */ _CFFI_OP(_CFFI_OP_FUNCTION, 139), // DH *()(EVP_PKEY *)
/* 142 */ _CFFI_OP(_CFFI_OP_POINTER, 2478), // EVP_PKEY *
/* 143 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 144 */ _CFFI_OP(_CFFI_OP_FUNCTION, 139), // DH *()(void)
/* 145 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 146 */ _CFFI_OP(_CFFI_OP_FUNCTION, 147), // DSA *()(DSA *)
/* 147 */ _CFFI_OP(_CFFI_OP_POINTER, 2467), // DSA *
/* 148 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 149 */ _CFFI_OP(_CFFI_OP_FUNCTION, 147), // DSA *()(EVP_PKEY *)
/* 150 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 151 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 152 */ _CFFI_OP(_CFFI_OP_FUNCTION, 147), // DSA *()(void)
/* 153 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 154 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2135), // EC_GROUP *()(int)
/* 155 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 156 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 157 */ _CFFI_OP(_CFFI_OP_FUNCTION, 173), // EC_GROUP const *()(EC_KEY const *)
/* 158 */ _CFFI_OP(_CFFI_OP_NOOP, 74),
/* 159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 666), // EC_KEY *()(BIO *, EC_KEY * *)
/* 161 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 162 */ _CFFI_OP(_CFFI_OP_POINTER, 666), // EC_KEY * *
/* 163 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 164 */ _CFFI_OP(_CFFI_OP_FUNCTION, 666), // EC_KEY *()(EVP_PKEY *)
/* 165 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 166 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 167 */ _CFFI_OP(_CFFI_OP_FUNCTION, 666), // EC_KEY *()(int)
/* 168 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 169 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 170 */ _CFFI_OP(_CFFI_OP_FUNCTION, 666), // EC_KEY *()(void)
/* 171 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 172 */ _CFFI_OP(_CFFI_OP_FUNCTION, 984), // EC_METHOD const *()(EC_GROUP const *)
/* 173 */ _CFFI_OP(_CFFI_OP_POINTER, 2468), // EC_GROUP const *
/* 174 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 175 */ _CFFI_OP(_CFFI_OP_FUNCTION, 900), // EC_POINT *()(EC_GROUP const *)
/* 176 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 177 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 178 */ _CFFI_OP(_CFFI_OP_FUNCTION, 900), // EC_POINT *()(EC_POINT const *, EC_GROUP const *)
/* 179 */ _CFFI_OP(_CFFI_OP_POINTER, 2471), // EC_POINT const *
/* 180 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 181 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 182 */ _CFFI_OP(_CFFI_OP_FUNCTION, 179), // EC_POINT const *()(EC_GROUP const *)
/* 183 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 184 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 185 */ _CFFI_OP(_CFFI_OP_FUNCTION, 179), // EC_POINT const *()(EC_KEY const *)
/* 186 */ _CFFI_OP(_CFFI_OP_NOOP, 74),
/* 187 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 188 */ _CFFI_OP(_CFFI_OP_FUNCTION, 219), // ENGINE *()(char const *)
/* 189 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 190 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 191 */ _CFFI_OP(_CFFI_OP_FUNCTION, 219), // ENGINE *()(void)
/* 192 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 193 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2151), // EVP_CIPHER *()(OSSL_LIB_CTX *, char const *, char const *)
/* 194 */ _CFFI_OP(_CFFI_OP_POINTER, 2486), // OSSL_LIB_CTX *
/* 195 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 196 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 197 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 198 */ _CFFI_OP(_CFFI_OP_FUNCTION, 658), // EVP_CIPHER const *()(char const *)
/* 199 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 200 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 201 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1004), // EVP_CIPHER_CTX *()(void)
/* 202 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 203 */ _CFFI_OP(_CFFI_OP_FUNCTION, 306), // EVP_MD const *()(char const *)
/* 204 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 205 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 206 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1037), // EVP_MD_CTX *()(void)
/* 207 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 208 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(BIO *, EVP_PKEY * *)
/* 209 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 210 */ _CFFI_OP(_CFFI_OP_POINTER, 142), // EVP_PKEY * *
/* 211 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 212 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(BIO *, EVP_PKEY * *, int(*)(char *, int, int, void *), void *)
/* 213 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 214 */ _CFFI_OP(_CFFI_OP_NOOP, 210),
/* 215 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 216 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 217 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 218 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(ENGINE *, char const *, UI_METHOD *, void *)
/* 219 */ _CFFI_OP(_CFFI_OP_POINTER, 2473), // ENGINE *
/* 220 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 221 */ _CFFI_OP(_CFFI_OP_POINTER, 2513), // UI_METHOD *
/* 222 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 223 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 224 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(NETSCAPE_SPKI *)
/* 225 */ _CFFI_OP(_CFFI_OP_POINTER, 2484), // NETSCAPE_SPKI *
/* 226 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 227 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(X509 *)
/* 228 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 229 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 230 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(X509_REQ *)
/* 231 */ _CFFI_OP(_CFFI_OP_POINTER, 2525), // X509_REQ *
/* 232 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 233 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(int, ENGINE *, unsigned char const *, size_t)
/* 234 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 235 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 236 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 237 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28), // size_t
/* 238 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 239 */ _CFFI_OP(_CFFI_OP_FUNCTION, 142), // EVP_PKEY *()(void)
/* 240 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 241 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1143), // EVP_PKEY_CTX *()(EVP_PKEY *, ENGINE *)
/* 242 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 243 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 244 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 245 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1143), // EVP_PKEY_CTX *()(int, ENGINE *)
/* 246 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 247 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 248 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 249 */ _CFFI_OP(_CFFI_OP_FUNCTION, 701), // GENERAL_NAME *()(GENERAL_NAMES *, int)
/* 250 */ _CFFI_OP(_CFFI_OP_POINTER, 2482), // GENERAL_NAMES *
/* 251 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 252 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 253 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1193), // HMAC_CTX *()(void)
/* 254 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 255 */ _CFFI_OP(_CFFI_OP_FUNCTION, 225), // NETSCAPE_SPKI *()(char const *, int)
/* 256 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 257 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 258 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 259 */ _CFFI_OP(_CFFI_OP_FUNCTION, 225), // NETSCAPE_SPKI *()(void)
/* 260 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 261 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1230), // OSSL_PROVIDER *()(OSSL_LIB_CTX *, char const *)
/* 262 */ _CFFI_OP(_CFFI_OP_NOOP, 194),
/* 263 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 264 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 265 */ _CFFI_OP(_CFFI_OP_FUNCTION, 705), // PKCS12 *()(BIO *, PKCS12 * *)
/* 266 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 267 */ _CFFI_OP(_CFFI_OP_POINTER, 705), // PKCS12 * *
/* 268 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 269 */ _CFFI_OP(_CFFI_OP_FUNCTION, 705), // PKCS12 *()(char *, char *, EVP_PKEY *, X509 *, Cryptography_STACK_OF_X509 *, int, int, int, int, int)
/* 270 */ _CFFI_OP(_CFFI_OP_POINTER, 2530), // char *
/* 271 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 272 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 273 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 274 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 275 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 276 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 277 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 278 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 279 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 280 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 281 */ _CFFI_OP(_CFFI_OP_FUNCTION, 100), // PKCS7 *()(BIO *, BIO * *)
/* 282 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 283 */ _CFFI_OP(_CFFI_OP_POINTER, 129), // BIO * *
/* 284 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 285 */ _CFFI_OP(_CFFI_OP_FUNCTION, 100), // PKCS7 *()(BIO *, PKCS7 * *)
/* 286 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 287 */ _CFFI_OP(_CFFI_OP_POINTER, 100), // PKCS7 * *
/* 288 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 289 */ _CFFI_OP(_CFFI_OP_FUNCTION, 100), // PKCS7 *()(BIO *, PKCS7 * *, int(*)(char *, int, int, void *), void *)
/* 290 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 291 */ _CFFI_OP(_CFFI_OP_NOOP, 287),
/* 292 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 293 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 294 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 295 */ _CFFI_OP(_CFFI_OP_FUNCTION, 100), // PKCS7 *()(X509 *, EVP_PKEY *, Cryptography_STACK_OF_X509 *, BIO *, int)
/* 296 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 297 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 298 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 299 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 300 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 301 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 302 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2498), // PKCS7_SIGNER_INFO *()(PKCS7 *, X509 *, EVP_PKEY *, EVP_MD const *, int)
/* 303 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 304 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 305 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 306 */ _CFFI_OP(_CFFI_OP_POINTER, 2476), // EVP_MD const *
/* 307 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 308 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 309 */ _CFFI_OP(_CFFI_OP_FUNCTION, 323), // RSA *()(BIO *, RSA * *)
/* 310 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 311 */ _CFFI_OP(_CFFI_OP_POINTER, 323), // RSA * *
/* 312 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 313 */ _CFFI_OP(_CFFI_OP_FUNCTION, 323), // RSA *()(BIO *, RSA * *, int(*)(char *, int, int, void *), void *)
/* 314 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 315 */ _CFFI_OP(_CFFI_OP_NOOP, 311),
/* 316 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 317 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 318 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 319 */ _CFFI_OP(_CFFI_OP_FUNCTION, 323), // RSA *()(EVP_PKEY *)
/* 320 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 321 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 322 */ _CFFI_OP(_CFFI_OP_FUNCTION, 323), // RSA *()(RSA *)
/* 323 */ _CFFI_OP(_CFFI_OP_POINTER, 2504), // RSA *
/* 324 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 325 */ _CFFI_OP(_CFFI_OP_FUNCTION, 323), // RSA *()(void)
/* 326 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 327 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2505), // SRTP_PROTECTION_PROFILE *()(SSL *)
/* 328 */ _CFFI_OP(_CFFI_OP_POINTER, 2507), // SSL *
/* 329 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 330 */ _CFFI_OP(_CFFI_OP_FUNCTION, 328), // SSL *()(SSL_CTX *)
/* 331 */ _CFFI_OP(_CFFI_OP_POINTER, 2509), // SSL_CTX *
/* 332 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 333 */ _CFFI_OP(_CFFI_OP_FUNCTION, 536), // SSL_CIPHER const *()(Cryptography_STACK_OF_SSL_CIPHER *, int)
/* 334 */ _CFFI_OP(_CFFI_OP_POINTER, 2459), // Cryptography_STACK_OF_SSL_CIPHER *
/* 335 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 336 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 337 */ _CFFI_OP(_CFFI_OP_FUNCTION, 536), // SSL_CIPHER const *()(SSL *, unsigned char const *)
/* 338 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 339 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 340 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 341 */ _CFFI_OP(_CFFI_OP_FUNCTION, 536), // SSL_CIPHER const *()(SSL const *)
/* 342 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 343 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 344 */ _CFFI_OP(_CFFI_OP_FUNCTION, 331), // SSL_CTX *()(SSL *, SSL_CTX *)
/* 345 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 346 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 347 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 348 */ _CFFI_OP(_CFFI_OP_FUNCTION, 331), // SSL_CTX *()(SSL_METHOD *)
/* 349 */ _CFFI_OP(_CFFI_OP_POINTER, 2511), // SSL_METHOD *
/* 350 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 351 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2510), // SSL_METHOD const *()(void)
/* 352 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 353 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1298), // SSL_SESSION *()(SSL *)
/* 354 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 355 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 356 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1298), // SSL_SESSION *()(SSL const *)
/* 357 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 358 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 359 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1298), // SSL_SESSION *()(void)
/* 360 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 361 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(BIO *, X509 * *)
/* 362 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 363 */ _CFFI_OP(_CFFI_OP_POINTER, 11), // X509 * *
/* 364 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 365 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(BIO *, X509 * *, int(*)(char *, int, int, void *), void *)
/* 366 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 367 */ _CFFI_OP(_CFFI_OP_NOOP, 363),
/* 368 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 369 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 370 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 371 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(Cryptography_STACK_OF_X509 *, int)
/* 372 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 373 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 374 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 375 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(SSL const *)
/* 376 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 377 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 378 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(X509 *)
/* 379 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 380 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 381 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(X509_OBJECT *)
/* 382 */ _CFFI_OP(_CFFI_OP_POINTER, 2524), // X509_OBJECT *
/* 383 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 384 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(X509_STORE_CTX *)
/* 385 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 386 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 387 */ _CFFI_OP(_CFFI_OP_FUNCTION, 11), // X509 *()(void)
/* 388 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 389 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2516), // X509_ALGOR const *()(X509 const *)
/* 390 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 391 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 392 */ _CFFI_OP(_CFFI_OP_FUNCTION, 34), // X509_CRL *()(BIO *, X509_CRL * *)
/* 393 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 394 */ _CFFI_OP(_CFFI_OP_POINTER, 34), // X509_CRL * *
/* 395 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 396 */ _CFFI_OP(_CFFI_OP_FUNCTION, 34), // X509_CRL *()(BIO *, X509_CRL * *, int(*)(char *, int, int, void *), void *)
/* 397 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 398 */ _CFFI_OP(_CFFI_OP_NOOP, 394),
/* 399 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 400 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 401 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 402 */ _CFFI_OP(_CFFI_OP_FUNCTION, 34), // X509_CRL *()(X509_CRL *)
/* 403 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 404 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 405 */ _CFFI_OP(_CFFI_OP_FUNCTION, 34), // X509_CRL *()(void)
/* 406 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 407 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(CONF *, X509V3_CTX *, char *, char *)
/* 408 */ _CFFI_OP(_CFFI_OP_POINTER, 2456), // CONF *
/* 409 */ _CFFI_OP(_CFFI_OP_POINTER, 2515), // X509V3_CTX *
/* 410 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 411 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 412 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 413 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(X509 *, int)
/* 414 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 415 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 416 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 417 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(X509_EXTENSION * *, ASN1_OBJECT *, int, ASN1_OCTET_STRING *)
/* 418 */ _CFFI_OP(_CFFI_OP_POINTER, 17), // X509_EXTENSION * *
/* 419 */ _CFFI_OP(_CFFI_OP_POINTER, 2443), // ASN1_OBJECT *
/* 420 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 421 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 422 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 423 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(X509_EXTENSION *)
/* 424 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 425 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 426 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(X509_EXTENSIONS *, int)
/* 427 */ _CFFI_OP(_CFFI_OP_POINTER, 2521), // X509_EXTENSIONS *
/* 428 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 429 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 430 */ _CFFI_OP(_CFFI_OP_FUNCTION, 17), // X509_EXTENSION *()(X509_REVOKED *, int)
/* 431 */ _CFFI_OP(_CFFI_OP_POINTER, 2526), // X509_REVOKED *
/* 432 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 433 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 434 */ _CFFI_OP(_CFFI_OP_FUNCTION, 427), // X509_EXTENSIONS *()(X509_REQ *)
/* 435 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 436 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 437 */ _CFFI_OP(_CFFI_OP_FUNCTION, 427), // X509_EXTENSIONS *()(void)
/* 438 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 439 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(Cryptography_STACK_OF_X509_NAME *, int)
/* 440 */ _CFFI_OP(_CFFI_OP_POINTER, 2462), // Cryptography_STACK_OF_X509_NAME *
/* 441 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 442 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 443 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(X509 *)
/* 444 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 445 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 446 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(X509_CRL *)
/* 447 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 448 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 449 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(X509_NAME *)
/* 450 */ _CFFI_OP(_CFFI_OP_POINTER, 2522), // X509_NAME *
/* 451 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 452 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(X509_REQ *)
/* 453 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 454 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 455 */ _CFFI_OP(_CFFI_OP_FUNCTION, 450), // X509_NAME *()(void)
/* 456 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 457 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // X509_NAME_ENTRY *()(X509_NAME *, int)
/* 458 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 459 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 460 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 461 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // X509_NAME_ENTRY *()(X509_NAME_ENTRY * *, ASN1_OBJECT *, int, unsigned char const *, int)
/* 462 */ _CFFI_OP(_CFFI_OP_POINTER, 20), // X509_NAME_ENTRY * *
/* 463 */ _CFFI_OP(_CFFI_OP_NOOP, 419),
/* 464 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 465 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 466 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 467 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 468 */ _CFFI_OP(_CFFI_OP_FUNCTION, 382), // X509_OBJECT *()(Cryptography_STACK_OF_X509_OBJECT *, int)
/* 469 */ _CFFI_OP(_CFFI_OP_POINTER, 2464), // Cryptography_STACK_OF_X509_OBJECT *
/* 470 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 471 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 472 */ _CFFI_OP(_CFFI_OP_FUNCTION, 231), // X509_REQ *()(BIO *, X509_REQ * *)
/* 473 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 474 */ _CFFI_OP(_CFFI_OP_POINTER, 231), // X509_REQ * *
/* 475 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 476 */ _CFFI_OP(_CFFI_OP_FUNCTION, 231), // X509_REQ *()(BIO *, X509_REQ * *, int(*)(char *, int, int, void *), void *)
/* 477 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 478 */ _CFFI_OP(_CFFI_OP_NOOP, 474),
/* 479 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 480 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 481 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 482 */ _CFFI_OP(_CFFI_OP_FUNCTION, 231), // X509_REQ *()(void)
/* 483 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 484 */ _CFFI_OP(_CFFI_OP_FUNCTION, 431), // X509_REVOKED *()(Cryptography_STACK_OF_X509_REVOKED *, int)
/* 485 */ _CFFI_OP(_CFFI_OP_POINTER, 2465), // Cryptography_STACK_OF_X509_REVOKED *
/* 486 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 487 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 488 */ _CFFI_OP(_CFFI_OP_FUNCTION, 431), // X509_REVOKED *()(X509_REVOKED *)
/* 489 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 490 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 491 */ _CFFI_OP(_CFFI_OP_FUNCTION, 431), // X509_REVOKED *()(void)
/* 492 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 493 */ _CFFI_OP(_CFFI_OP_FUNCTION, 123), // X509_STORE *()(SSL_CTX const *)
/* 494 */ _CFFI_OP(_CFFI_OP_POINTER, 2509), // SSL_CTX const *
/* 495 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 496 */ _CFFI_OP(_CFFI_OP_FUNCTION, 123), // X509_STORE *()(void)
/* 497 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 498 */ _CFFI_OP(_CFFI_OP_FUNCTION, 108), // X509_STORE_CTX *()(void)
/* 499 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 500 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1674), // X509_VERIFY_PARAM *()(SSL *)
/* 501 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 502 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 503 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1674), // X509_VERIFY_PARAM *()(SSL_CTX *)
/* 504 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 505 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 506 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1674), // X509_VERIFY_PARAM *()(X509_STORE *)
/* 507 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 508 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 509 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1674), // X509_VERIFY_PARAM *()(X509_STORE_CTX *)
/* 510 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 511 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 512 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1674), // X509_VERIFY_PARAM *()(void)
/* 513 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 514 */ _CFFI_OP(_CFFI_OP_FUNCTION, 270), // char *()(BIGNUM const *)
/* 515 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 516 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 517 */ _CFFI_OP(_CFFI_OP_FUNCTION, 270), // char *()(NETSCAPE_SPKI *)
/* 518 */ _CFFI_OP(_CFFI_OP_NOOP, 225),
/* 519 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 520 */ _CFFI_OP(_CFFI_OP_FUNCTION, 270), // char *()(X509_NAME *, char *, int)
/* 521 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 522 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 523 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 524 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 525 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(ENGINE const *)
/* 526 */ _CFFI_OP(_CFFI_OP_POINTER, 2473), // ENGINE const *
/* 527 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 528 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(SSL const *)
/* 529 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 530 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 531 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(SSL const *, int)
/* 532 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 533 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 534 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 535 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(SSL_CIPHER const *)
/* 536 */ _CFFI_OP(_CFFI_OP_POINTER, 2508), // SSL_CIPHER const *
/* 537 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 538 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(int)
/* 539 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 540 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 541 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(long)
/* 542 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 543 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 544 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(unsigned long)
/* 545 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10), // unsigned long
/* 546 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 547 */ _CFFI_OP(_CFFI_OP_FUNCTION, 23), // char const *()(void)
/* 548 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 549 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_ENUMERATED *, long)
/* 550 */ _CFFI_OP(_CFFI_OP_POINTER, 2439), // ASN1_ENUMERATED *
/* 551 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 552 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 553 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_INTEGER *, long)
/* 554 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 555 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 556 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 557 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_OBJECT const *)
/* 558 */ _CFFI_OP(_CFFI_OP_POINTER, 2443), // ASN1_OBJECT const *
/* 559 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 560 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_OCTET_STRING *)
/* 561 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 562 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 563 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_OCTET_STRING *, char const *)
/* 564 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 565 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 566 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 567 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_OCTET_STRING *, unsigned char const *, int)
/* 568 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 569 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 570 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 571 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 572 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ASN1_OCTET_STRING *, void const *, int)
/* 573 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 574 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 575 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 576 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 577 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM * *, char const *)
/* 578 */ _CFFI_OP(_CFFI_OP_POINTER, 7), // BIGNUM * *
/* 579 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 580 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 581 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BIGNUM const *)
/* 582 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 583 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 584 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 585 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *)
/* 586 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 587 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 588 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 589 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 590 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 591 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 592 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 593 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 594 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 595 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 596 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 597 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *, BN_MONT_CTX *)
/* 598 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 599 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 600 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 601 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 602 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 603 */ _CFFI_OP(_CFFI_OP_POINTER, 2454), // BN_MONT_CTX *
/* 604 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 605 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 606 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 607 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 608 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 609 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 610 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 611 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, BN_ULONG)
/* 612 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 613 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(BN_ULONG), (
           ((BN_ULONG)-1) | 0 /* check that BN_ULONG is an integer type */
         ) <= 0)), // BN_ULONG
/* 614 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 615 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM *, int, int, BIGNUM const *, BIGNUM const *, BN_GENCB *)
/* 616 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 617 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 618 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 619 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 620 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 621 */ _CFFI_OP(_CFFI_OP_POINTER, 2453), // BN_GENCB *
/* 622 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 623 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM const *)
/* 624 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 625 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 626 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM const *, BIGNUM const *)
/* 627 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 628 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 629 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 630 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM const *, int, BN_CTX *, BN_GENCB *)
/* 631 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 632 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 633 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 634 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 635 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 636 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIGNUM const *, unsigned char *)
/* 637 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 638 */ _CFFI_OP(_CFFI_OP_POINTER, 2533), // unsigned char *
/* 639 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 640 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *)
/* 641 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 642 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 643 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, ASN1_INTEGER *)
/* 644 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 645 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 646 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 647 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, DH *)
/* 648 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 649 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 650 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 651 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, DSA *)
/* 652 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 653 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 654 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 655 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, DSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 656 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 657 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 658 */ _CFFI_OP(_CFFI_OP_POINTER, 2474), // EVP_CIPHER const *
/* 659 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 660 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 661 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 662 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, EC_KEY *)
/* 665 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 666 */ _CFFI_OP(_CFFI_OP_POINTER, 2469), // EC_KEY *
/* 667 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 668 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, EC_KEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 669 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 670 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 671 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 672 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 673 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 674 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 675 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 676 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 677 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, EVP_PKEY *)
/* 678 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 679 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 680 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 681 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, char *, int, int(*)(char *, int, int, void *), void *)
/* 682 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 683 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 684 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 685 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 686 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 687 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 688 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 689 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 690 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 691 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 692 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 693 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 694 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 695 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 696 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 697 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 698 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 699 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, GENERAL_NAME *)
/* 700 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 701 */ _CFFI_OP(_CFFI_OP_POINTER, 2481), // GENERAL_NAME *
/* 702 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 703 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, PKCS12 *)
/* 704 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 705 */ _CFFI_OP(_CFFI_OP_POINTER, 2488), // PKCS12 *
/* 706 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 707 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, PKCS7 *)
/* 708 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 709 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 710 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 711 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, PKCS7 *, BIO *, int)
/* 712 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 713 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 714 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 715 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 716 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 717 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, RSA *)
/* 718 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 719 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 720 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 721 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, RSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 722 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 723 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 724 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 725 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 726 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 727 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 728 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 729 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 730 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, RSA const *)
/* 731 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 732 */ _CFFI_OP(_CFFI_OP_POINTER, 2504), // RSA const *
/* 733 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 734 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, RSA const *, int)
/* 735 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 736 */ _CFFI_OP(_CFFI_OP_NOOP, 732),
/* 737 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 738 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 739 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509 *)
/* 740 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 741 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 742 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 743 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509 *, unsigned long, unsigned long)
/* 744 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 745 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 746 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 747 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 748 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 749 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509_CRL *)
/* 750 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 751 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 752 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 753 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509_EXTENSION *, unsigned long, int)
/* 754 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 755 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 756 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 757 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 758 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 759 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509_REQ *)
/* 760 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 761 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 762 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 763 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, X509_REQ *, unsigned long, unsigned long)
/* 764 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 765 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 766 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 767 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 768 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 769 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, char *, int)
/* 770 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 771 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 772 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 773 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 774 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, void *, int)
/* 775 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 776 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 777 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 778 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 779 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BIO *, void const *, int)
/* 780 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 781 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 782 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 783 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 784 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(BN_MONT_CTX *, BIGNUM const *, BN_CTX *)
/* 785 */ _CFFI_OP(_CFFI_OP_NOOP, 603),
/* 786 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 787 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 788 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 789 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(CMAC_CTX *, CMAC_CTX const *)
/* 790 */ _CFFI_OP(_CFFI_OP_POINTER, 2455), // CMAC_CTX *
/* 791 */ _CFFI_OP(_CFFI_OP_POINTER, 2455), // CMAC_CTX const *
/* 792 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 793 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(CMAC_CTX *, unsigned char *, size_t *)
/* 794 */ _CFFI_OP(_CFFI_OP_NOOP, 790),
/* 795 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 796 */ _CFFI_OP(_CFFI_OP_POINTER, 237), // size_t *
/* 797 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 798 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(CMAC_CTX *, void const *, size_t)
/* 799 */ _CFFI_OP(_CFFI_OP_NOOP, 790),
/* 800 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 801 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 802 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 803 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(CMAC_CTX *, void const *, size_t, EVP_CIPHER const *, ENGINE *)
/* 804 */ _CFFI_OP(_CFFI_OP_NOOP, 790),
/* 805 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 806 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 807 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 808 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 809 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 810 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_SSL_CIPHER *)
/* 811 */ _CFFI_OP(_CFFI_OP_NOOP, 334),
/* 812 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 813 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509 *)
/* 814 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 815 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 816 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509 *, X509 *)
/* 817 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 818 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 819 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 820 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509_NAME *)
/* 821 */ _CFFI_OP(_CFFI_OP_NOOP, 440),
/* 822 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 823 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509_NAME *, X509_NAME *)
/* 824 */ _CFFI_OP(_CFFI_OP_NOOP, 440),
/* 825 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 826 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 827 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509_NAME_ENTRY *, X509_NAME_ENTRY *)
/* 828 */ _CFFI_OP(_CFFI_OP_POINTER, 2463), // Cryptography_STACK_OF_X509_NAME_ENTRY *
/* 829 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 830 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 831 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509_OBJECT *)
/* 832 */ _CFFI_OP(_CFFI_OP_NOOP, 469),
/* 833 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 834 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(Cryptography_STACK_OF_X509_REVOKED *)
/* 835 */ _CFFI_OP(_CFFI_OP_NOOP, 485),
/* 836 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 837 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH *)
/* 838 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 839 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 840 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH *, BIGNUM *, BIGNUM *)
/* 841 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 842 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 843 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 844 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 845 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 846 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 847 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 848 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 849 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 850 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 851 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH *, int, int, BN_GENCB *)
/* 852 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 853 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 854 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 855 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 856 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 857 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH const *)
/* 858 */ _CFFI_OP(_CFFI_OP_POINTER, 2466), // DH const *
/* 859 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 860 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DH const *, int *)
/* 861 */ _CFFI_OP(_CFFI_OP_NOOP, 858),
/* 862 */ _CFFI_OP(_CFFI_OP_POINTER, 24), // int *
/* 863 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 864 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DSA *)
/* 865 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 866 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 867 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DSA *, BIGNUM *, BIGNUM *)
/* 868 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 869 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 870 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 871 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 872 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 873 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 874 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 875 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 876 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 877 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 878 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DSA *, int, unsigned char *, int, int *, unsigned long *, BN_GENCB *)
/* 879 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 880 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 881 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 882 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 883 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 884 */ _CFFI_OP(_CFFI_OP_POINTER, 545), // unsigned long *
/* 885 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 886 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 887 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(DSA const *)
/* 888 */ _CFFI_OP(_CFFI_OP_POINTER, 2467), // DSA const *
/* 889 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 890 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *)
/* 891 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 892 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 893 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, BIGNUM *, BN_CTX *)
/* 894 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 895 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 896 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 897 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 898 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 899 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 900 */ _CFFI_OP(_CFFI_OP_POINTER, 2471), // EC_POINT *
/* 901 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 902 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 903 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 904 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 905 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, EC_POINT const *, BIGNUM const *, BN_CTX *)
/* 906 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 907 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 908 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 909 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 910 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 911 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 912 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 913 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, BN_CTX *)
/* 914 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 915 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 916 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 917 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 918 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, BN_CTX *)
/* 919 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 920 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 921 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 922 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 923 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 924 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 925 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 926 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 927 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 928 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 929 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 930 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 931 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT *, unsigned char const *, size_t, BN_CTX *)
/* 932 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 933 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 934 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 935 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 936 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT const *)
/* 939 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 940 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 941 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 942 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT const *, BIGNUM *, BIGNUM *, BN_CTX *)
/* 943 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 944 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 945 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 946 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 947 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 948 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 949 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT const *, BN_CTX *)
/* 950 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 951 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 952 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 953 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 954 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_GROUP const *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 955 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 956 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 957 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 958 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 959 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 960 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY *)
/* 961 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 962 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 963 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY *, BIGNUM *, BIGNUM *)
/* 964 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 965 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 966 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 967 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 968 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY *, BIGNUM const *)
/* 969 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 970 */ _CFFI_OP(_CFFI_OP_NOOP, 56),
/* 971 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 972 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY *, EC_GROUP const *)
/* 973 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 974 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 975 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 976 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY *, EC_POINT const *)
/* 977 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 978 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 979 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 980 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_KEY const *)
/* 981 */ _CFFI_OP(_CFFI_OP_NOOP, 74),
/* 982 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 983 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EC_METHOD const *)
/* 984 */ _CFFI_OP(_CFFI_OP_POINTER, 2470), // EC_METHOD const *
/* 985 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 986 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ENGINE *)
/* 987 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 988 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 989 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ENGINE *, char const *, char const *, int)
/* 990 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 991 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 992 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 993 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 994 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 995 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(ENGINE *, char const *, long, void *, void(*)(void), int)
/* 996 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 997 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 998 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 999 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1000 */ _CFFI_OP(_CFFI_OP_POINTER, 2431), // void(*)(void)
/* 1001 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1002 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1003 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *)
/* 1004 */ _CFFI_OP(_CFFI_OP_POINTER, 2475), // EVP_CIPHER_CTX *
/* 1005 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1006 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *, EVP_CIPHER const *, ENGINE *, unsigned char const *, unsigned char const *, int)
/* 1007 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 1008 */ _CFFI_OP(_CFFI_OP_NOOP, 658),
/* 1009 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 1010 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1011 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1012 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1013 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1014 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *, int)
/* 1015 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 1016 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1017 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1018 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *, int, int, void *)
/* 1019 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 1020 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1021 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1022 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1023 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1024 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *, unsigned char *, int *)
/* 1025 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 1026 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1027 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1028 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1029 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_CIPHER_CTX *, unsigned char *, int *, unsigned char const *, int)
/* 1030 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 1031 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1032 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1033 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1034 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1035 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1036 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, EVP_MD const *)
/* 1037 */ _CFFI_OP(_CFFI_OP_POINTER, 2477), // EVP_MD_CTX *
/* 1038 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1039 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1040 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, EVP_MD const *, ENGINE *)
/* 1041 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1042 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1043 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 1044 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1045 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, EVP_MD_CTX const *)
/* 1046 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1047 */ _CFFI_OP(_CFFI_OP_POINTER, 2477), // EVP_MD_CTX const *
/* 1048 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1049 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, EVP_PKEY_CTX * *, EVP_MD const *, ENGINE *, EVP_PKEY *)
/* 1050 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1051 */ _CFFI_OP(_CFFI_OP_POINTER, 1143), // EVP_PKEY_CTX * *
/* 1052 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1053 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 1054 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1055 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1056 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char *, size_t *)
/* 1057 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1058 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1059 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1060 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1061 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1062 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1063 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1064 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1065 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1066 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1067 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1068 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char *, size_t)
/* 1069 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1070 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1071 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1072 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1073 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *)
/* 1074 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1075 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1076 */ _CFFI_OP(_CFFI_OP_POINTER, 1094), // unsigned int *
/* 1077 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1078 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *, EVP_PKEY *)
/* 1079 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1080 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1081 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1082 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1083 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1084 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1085 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1086 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1087 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1088 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1089 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1090 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1091 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, unsigned char const *, unsigned int, EVP_PKEY *)
/* 1092 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1093 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1094 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8), // unsigned int
/* 1095 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1096 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1097 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_MD_CTX *, void const *, size_t)
/* 1098 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1099 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 1100 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1101 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1102 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *)
/* 1103 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1104 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1105 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, DH *)
/* 1106 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1107 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 1108 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1109 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, DSA *)
/* 1110 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1111 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 1112 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1113 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, EC_KEY *)
/* 1114 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1115 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 1116 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1117 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, RSA *)
/* 1118 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1119 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 1120 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1121 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, int)
/* 1122 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1123 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1124 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1125 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY *, unsigned char const *, size_t)
/* 1126 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1127 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1128 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1129 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1130 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY const *)
/* 1131 */ _CFFI_OP(_CFFI_OP_POINTER, 2478), // EVP_PKEY const *
/* 1132 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1133 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY const *, EVP_PKEY const *)
/* 1134 */ _CFFI_OP(_CFFI_OP_NOOP, 1131),
/* 1135 */ _CFFI_OP(_CFFI_OP_NOOP, 1131),
/* 1136 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1137 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY const *, unsigned char *, size_t *)
/* 1138 */ _CFFI_OP(_CFFI_OP_NOOP, 1131),
/* 1139 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1140 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1141 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1142 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *)
/* 1143 */ _CFFI_OP(_CFFI_OP_POINTER, 2479), // EVP_PKEY_CTX *
/* 1144 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1145 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, EVP_MD *)
/* 1146 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1147 */ _CFFI_OP(_CFFI_OP_POINTER, 2476), // EVP_MD *
/* 1148 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1149 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, EVP_MD const *)
/* 1150 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1151 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1152 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1153 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, EVP_PKEY * *)
/* 1154 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1155 */ _CFFI_OP(_CFFI_OP_NOOP, 210),
/* 1156 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1157 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, EVP_PKEY *)
/* 1158 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1159 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1160 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1161 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, int)
/* 1162 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1163 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1164 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1165 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, unsigned char *, int)
/* 1166 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1167 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1168 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1169 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1170 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *)
/* 1171 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1172 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1173 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1174 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1175 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1176 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1177 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1178 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1179 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1180 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1181 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1182 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(EVP_PKEY_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1183 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 1184 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1185 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1186 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1187 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1188 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1189 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(GENERAL_NAMES *)
/* 1190 */ _CFFI_OP(_CFFI_OP_NOOP, 250),
/* 1191 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1192 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(HMAC_CTX *, HMAC_CTX *)
/* 1193 */ _CFFI_OP(_CFFI_OP_POINTER, 2483), // HMAC_CTX *
/* 1194 */ _CFFI_OP(_CFFI_OP_NOOP, 1193),
/* 1195 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1196 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(HMAC_CTX *, unsigned char *, unsigned int *)
/* 1197 */ _CFFI_OP(_CFFI_OP_NOOP, 1193),
/* 1198 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1199 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1200 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1201 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(HMAC_CTX *, unsigned char const *, size_t)
/* 1202 */ _CFFI_OP(_CFFI_OP_NOOP, 1193),
/* 1203 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1204 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1205 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1206 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(HMAC_CTX *, void const *, int, EVP_MD const *, ENGINE *)
/* 1207 */ _CFFI_OP(_CFFI_OP_NOOP, 1193),
/* 1208 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 1209 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1210 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1211 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 1212 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1213 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(NETSCAPE_SPKI *, EVP_PKEY *)
/* 1214 */ _CFFI_OP(_CFFI_OP_NOOP, 225),
/* 1215 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1216 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1217 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(NETSCAPE_SPKI *, EVP_PKEY *, EVP_MD const *)
/* 1218 */ _CFFI_OP(_CFFI_OP_NOOP, 225),
/* 1219 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1220 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1221 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1222 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(OSSL_LIB_CTX *)
/* 1223 */ _CFFI_OP(_CFFI_OP_NOOP, 194),
/* 1224 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1225 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(OSSL_LIB_CTX *, int)
/* 1226 */ _CFFI_OP(_CFFI_OP_NOOP, 194),
/* 1227 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1228 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1229 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(OSSL_PROVIDER *)
/* 1230 */ _CFFI_OP(_CFFI_OP_POINTER, 2487), // OSSL_PROVIDER *
/* 1231 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1232 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(PKCS12 *, char const *, EVP_PKEY * *, X509 * *, Cryptography_STACK_OF_X509 * *)
/* 1233 */ _CFFI_OP(_CFFI_OP_NOOP, 705),
/* 1234 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1235 */ _CFFI_OP(_CFFI_OP_NOOP, 210),
/* 1236 */ _CFFI_OP(_CFFI_OP_NOOP, 363),
/* 1237 */ _CFFI_OP(_CFFI_OP_POINTER, 101), // Cryptography_STACK_OF_X509 * *
/* 1238 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1239 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(PKCS7 *)
/* 1240 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 1241 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1242 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(PKCS7 *, BIO *, int)
/* 1243 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 1244 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1245 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1246 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1247 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(PKCS7 *, Cryptography_STACK_OF_X509 *, X509_STORE *, BIO *, BIO *, int)
/* 1248 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 1249 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 1250 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1251 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1252 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1253 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1254 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1255 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RAND_METHOD const *)
/* 1256 */ _CFFI_OP(_CFFI_OP_POINTER, 2503), // RAND_METHOD const *
/* 1257 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1258 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RSA *, BIGNUM *, BIGNUM *)
/* 1259 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 1260 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1261 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1262 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1263 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1264 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 1265 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1266 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1267 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1268 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1269 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RSA *, BN_CTX *)
/* 1270 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 1271 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 1272 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1273 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RSA *, int, BIGNUM *, BN_GENCB *)
/* 1274 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 1275 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1276 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 1277 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 1278 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1279 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(RSA const *)
/* 1280 */ _CFFI_OP(_CFFI_OP_NOOP, 732),
/* 1281 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1282 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *)
/* 1283 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1284 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1285 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, BIO_ADDR *)
/* 1286 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1287 */ _CFFI_OP(_CFFI_OP_POINTER, 2449), // BIO_ADDR *
/* 1288 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1289 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *)
/* 1290 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1291 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1292 */ _CFFI_OP(_CFFI_OP_POINTER, 67), // unsigned char const * *
/* 1293 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1294 */ _CFFI_OP(_CFFI_OP_POINTER, 1298), // SSL_SESSION * *
/* 1295 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1296 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, SSL_SESSION *)
/* 1297 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1298 */ _CFFI_OP(_CFFI_OP_POINTER, 2512), // SSL_SESSION *
/* 1299 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1300 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, char const *)
/* 1301 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1302 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1303 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1304 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, int *, void *)
/* 1305 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1306 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1307 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1308 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1309 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, int)
/* 1310 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1311 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1312 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1313 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned char *, size_t, char const *, size_t, unsigned char const *, size_t, int)
/* 1314 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1315 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1316 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1317 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1318 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1319 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1320 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1321 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1322 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1323 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned char *, unsigned int *)
/* 1324 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1325 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1326 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1327 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1328 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 1329 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1330 */ _CFFI_OP(_CFFI_OP_NOOP, 1292),
/* 1331 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1332 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1333 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1334 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1335 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1336 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned char const *, size_t, SSL_SESSION * *)
/* 1337 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1338 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1339 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1340 */ _CFFI_OP(_CFFI_OP_NOOP, 1294),
/* 1341 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1342 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned char const *, unsigned int)
/* 1343 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1344 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1345 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1346 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1347 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1348 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1349 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1350 */ _CFFI_OP(_CFFI_OP_NOOP, 1292),
/* 1351 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1352 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1353 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1354 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1355 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1356 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1357 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1358 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1359 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1360 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1361 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1362 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1363 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, void *)
/* 1364 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1365 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1366 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1367 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, void *, int)
/* 1368 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1369 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1370 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1371 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1372 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, void *, size_t, size_t *)
/* 1373 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1374 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1375 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1376 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1377 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1378 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, void const *, int)
/* 1379 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1380 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 1381 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1382 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1383 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL *, void const *, size_t, size_t *)
/* 1384 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1385 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 1386 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1387 */ _CFFI_OP(_CFFI_OP_NOOP, 796),
/* 1388 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1389 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL const *)
/* 1390 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 1391 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1392 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL const *, int)
/* 1393 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 1394 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1395 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1396 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CIPHER const *)
/* 1397 */ _CFFI_OP(_CFFI_OP_NOOP, 536),
/* 1398 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1399 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CIPHER const *, int *)
/* 1400 */ _CFFI_OP(_CFFI_OP_NOOP, 536),
/* 1401 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1402 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1403 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *)
/* 1404 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1405 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1406 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, ENGINE *)
/* 1407 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1408 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 1409 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1410 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, EVP_PKEY *)
/* 1411 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1412 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1413 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1414 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, X509 *)
/* 1415 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1416 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1417 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1418 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, char const *)
/* 1419 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1420 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1421 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1422 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, char const *, char const *)
/* 1423 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1424 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1425 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1426 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1427 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, char const *, int)
/* 1428 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1429 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1430 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1431 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1432 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, uint32_t)
/* 1433 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1434 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 22), // uint32_t
/* 1435 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1436 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, unsigned char const *, unsigned int)
/* 1437 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1438 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1439 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1440 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1441 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX *, unsigned int, int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *), void(*)(SSL *, unsigned int, unsigned char const *, void *), void *, int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *), void *)
/* 1442 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1443 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1444 */ _CFFI_OP(_CFFI_OP_POINTER, 1347), // int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1445 */ _CFFI_OP(_CFFI_OP_POINTER, 2220), // void(*)(SSL *, unsigned int, unsigned char const *, void *)
/* 1446 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1447 */ _CFFI_OP(_CFFI_OP_POINTER, 1355), // int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1448 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1449 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1450 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_CTX const *)
/* 1451 */ _CFFI_OP(_CFFI_OP_NOOP, 494),
/* 1452 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1453 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_SESSION *, SSL_CIPHER const *)
/* 1454 */ _CFFI_OP(_CFFI_OP_NOOP, 1298),
/* 1455 */ _CFFI_OP(_CFFI_OP_NOOP, 536),
/* 1456 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1457 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_SESSION *, int)
/* 1458 */ _CFFI_OP(_CFFI_OP_NOOP, 1298),
/* 1459 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1460 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1461 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_SESSION *, unsigned char const *, size_t)
/* 1462 */ _CFFI_OP(_CFFI_OP_NOOP, 1298),
/* 1463 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1464 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1465 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1466 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_SESSION *, unsigned char const *, unsigned int)
/* 1467 */ _CFFI_OP(_CFFI_OP_NOOP, 1298),
/* 1468 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1469 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1470 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1471 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(SSL_SESSION const *)
/* 1472 */ _CFFI_OP(_CFFI_OP_POINTER, 2512), // SSL_SESSION const *
/* 1473 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1474 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 * *, X509_STORE_CTX *, X509 *)
/* 1475 */ _CFFI_OP(_CFFI_OP_NOOP, 363),
/* 1476 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1477 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1478 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1479 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *)
/* 1480 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1481 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1482 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, ASN1_INTEGER *)
/* 1483 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1484 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 1485 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1486 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, EVP_PKEY *)
/* 1487 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1488 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1489 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1490 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, EVP_PKEY *, EVP_MD const *)
/* 1491 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1492 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1493 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1494 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1495 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, X509_EXTENSION *, int)
/* 1496 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1497 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1498 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1499 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1500 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, X509_NAME *)
/* 1501 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1502 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1503 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1504 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, long)
/* 1505 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1506 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1507 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1508 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 *, unsigned char const *, int)
/* 1509 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1510 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1511 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1512 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1513 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 const *, EVP_MD const *, unsigned char *, unsigned int *)
/* 1514 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 1515 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1516 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1517 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1518 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1519 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509 const *, X509 const *)
/* 1520 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 1521 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 1522 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1523 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *)
/* 1524 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1525 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1526 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, ASN1_OCTET_STRING *)
/* 1527 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1528 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 1529 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1530 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, ASN1_OCTET_STRING const *)
/* 1531 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1532 */ _CFFI_OP(_CFFI_OP_POINTER, 2444), // ASN1_OCTET_STRING const *
/* 1533 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1534 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, EVP_PKEY *)
/* 1535 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1536 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1537 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1538 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, EVP_PKEY *, EVP_MD const *)
/* 1539 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1540 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1541 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1542 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1543 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, X509_EXTENSION *, int)
/* 1544 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1545 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1546 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1547 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1548 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, X509_NAME *)
/* 1549 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1550 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1551 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1552 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, X509_REVOKED *)
/* 1553 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1554 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1555 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1556 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL *, long)
/* 1557 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1558 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1559 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1560 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_CRL const *, X509_CRL const *)
/* 1561 */ _CFFI_OP(_CFFI_OP_NOOP, 45),
/* 1562 */ _CFFI_OP(_CFFI_OP_NOOP, 45),
/* 1563 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1564 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_EXTENSION *)
/* 1565 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1566 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1567 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_EXTENSIONS *)
/* 1568 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 1569 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1570 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_EXTENSIONS *, X509_EXTENSION *)
/* 1571 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 1572 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1573 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1574 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_EXTENSIONS *, X509_EXTENSION *, int)
/* 1575 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 1576 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1577 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1578 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1579 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME *)
/* 1580 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1581 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1582 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME *, X509_NAME_ENTRY *, int, int)
/* 1583 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1584 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1585 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1586 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1587 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1588 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME *, int, int)
/* 1589 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1590 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1591 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1592 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1593 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME *, int, int, unsigned char *, int, int, int)
/* 1594 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1595 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1596 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1597 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1598 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1599 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1600 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1601 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1602 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME *, unsigned char * *)
/* 1603 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1604 */ _CFFI_OP(_CFFI_OP_POINTER, 638), // unsigned char * *
/* 1605 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1606 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_NAME const *, X509_NAME const *)
/* 1607 */ _CFFI_OP(_CFFI_OP_POINTER, 2522), // X509_NAME const *
/* 1608 */ _CFFI_OP(_CFFI_OP_NOOP, 1607),
/* 1609 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1610 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, ASN1_OBJECT const *, int, unsigned char const *, int)
/* 1611 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1612 */ _CFFI_OP(_CFFI_OP_NOOP, 558),
/* 1613 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1614 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1615 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1616 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1617 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, EVP_PKEY *)
/* 1618 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1619 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1620 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1621 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, EVP_PKEY *, EVP_MD const *)
/* 1622 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1623 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1624 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1625 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1626 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, X509_EXTENSIONS *)
/* 1627 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1628 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 1629 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1630 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, X509_NAME *)
/* 1631 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1632 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 1633 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1634 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REQ *, long)
/* 1635 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1636 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1637 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1638 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REVOKED *)
/* 1639 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1640 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1641 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REVOKED *, ASN1_INTEGER *)
/* 1642 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1643 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 1644 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1645 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REVOKED *, ASN1_OCTET_STRING *)
/* 1646 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1647 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 1648 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1649 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REVOKED *, X509_EXTENSION *, int)
/* 1650 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1651 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1652 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1653 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1654 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_REVOKED *, int, void *, int, unsigned long)
/* 1655 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 1656 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1657 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1658 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1659 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1660 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1661 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *)
/* 1662 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, X509 *)
/* 1665 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1666 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1667 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1668 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, X509_CRL *)
/* 1669 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1670 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1671 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1672 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, X509_VERIFY_PARAM *)
/* 1673 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1674 */ _CFFI_OP(_CFFI_OP_POINTER, 2529), // X509_VERIFY_PARAM *
/* 1675 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1676 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, char const *, char const *)
/* 1677 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1678 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1679 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1680 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1681 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, int)
/* 1682 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1683 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1684 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1685 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE *, unsigned long)
/* 1686 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1687 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1688 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1689 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE_CTX *)
/* 1690 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1691 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1692 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE_CTX *, X509_STORE *, X509 *, Cryptography_STACK_OF_X509 *)
/* 1693 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1694 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1695 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1696 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 1697 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1698 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE_CTX *, char const *)
/* 1699 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1700 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1701 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1702 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE_CTX *, int, void *)
/* 1703 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1704 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1705 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1706 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1707 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_STORE_CTX *, void *)
/* 1708 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1709 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1710 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1711 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, ASN1_OBJECT *)
/* 1712 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1713 */ _CFFI_OP(_CFFI_OP_NOOP, 419),
/* 1714 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1715 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, Cryptography_STACK_OF_ASN1_OBJECT *)
/* 1716 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1717 */ _CFFI_OP(_CFFI_OP_POINTER, 2458), // Cryptography_STACK_OF_ASN1_OBJECT *
/* 1718 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1719 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, char const *)
/* 1720 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1721 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1722 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1723 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, char const *, size_t)
/* 1724 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1725 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1726 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1727 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1728 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, int)
/* 1729 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1730 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1731 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1732 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, unsigned char const *, size_t)
/* 1733 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1734 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1735 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1736 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1737 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM *, unsigned long)
/* 1738 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 1739 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1740 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1741 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(X509_VERIFY_PARAM const *)
/* 1742 */ _CFFI_OP(_CFFI_OP_POINTER, 2529), // X509_VERIFY_PARAM const *
/* 1743 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1744 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(char *)
/* 1745 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 1746 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1747 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(char *, int, int, void *)
/* 1748 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 1749 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1750 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1751 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1752 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1753 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(char const *)
/* 1754 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1755 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1756 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(char const *, int, unsigned char const *, int, int, EVP_MD const *, int, unsigned char *)
/* 1757 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1758 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1759 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1760 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1761 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1762 */ _CFFI_OP(_CFFI_OP_NOOP, 306),
/* 1763 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1764 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1765 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1766 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(char const *, size_t, unsigned char const *, size_t, uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *, size_t)
/* 1767 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1768 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1769 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1770 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1771 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24), // uint64_t
/* 1772 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 1773 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 1774 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 1775 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1776 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1777 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1778 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int)
/* 1779 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1780 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1781 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int, X509_STORE_CTX *)
/* 1782 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1783 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 1784 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1785 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, DSA *)
/* 1786 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1787 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1788 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1789 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1790 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1791 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 1792 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1793 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, EC_KEY *)
/* 1794 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1795 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1796 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1797 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1798 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1799 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 1800 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1801 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int, unsigned char const *, int, unsigned char const *, int, DSA *)
/* 1802 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1803 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1804 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1805 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1806 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1807 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 1808 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1809 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(int, unsigned char const *, int, unsigned char const *, int, EC_KEY *)
/* 1810 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1811 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1812 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1813 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1814 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1815 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 1816 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1817 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(unsigned char * *, ASN1_OCTET_STRING *)
/* 1818 */ _CFFI_OP(_CFFI_OP_NOOP, 1604),
/* 1819 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 1820 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1821 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(unsigned char * *, unsigned char *, unsigned char const *, unsigned int, unsigned char const *, unsigned int)
/* 1822 */ _CFFI_OP(_CFFI_OP_NOOP, 1604),
/* 1823 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1824 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1825 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1826 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 1827 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1828 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1829 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(unsigned char *, int)
/* 1830 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1831 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1832 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1833 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(unsigned int)
/* 1834 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1835 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1836 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(unsigned long)
/* 1837 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1838 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1839 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(void *(*)(size_t, char const *, int), void *(*)(void *, size_t, char const *, int), void(*)(void *, char const *, int))
/* 1840 */ _CFFI_OP(_CFFI_OP_POINTER, 2052), // void *(*)(size_t, char const *, int)
/* 1841 */ _CFFI_OP(_CFFI_OP_POINTER, 2057), // void *(*)(void *, size_t, char const *, int)
/* 1842 */ _CFFI_OP(_CFFI_OP_POINTER, 2421), // void(*)(void *, char const *, int)
/* 1843 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1844 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // int()(void)
/* 1845 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1846 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2363), // int(*()(X509_STORE *))(X509 * *, X509_STORE_CTX *, X509 *)
/* 1847 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 1848 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1849 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(BIO *, char * *)
/* 1850 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1851 */ _CFFI_OP(_CFFI_OP_POINTER, 270), // char * *
/* 1852 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1853 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(BIO *, int)
/* 1854 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1855 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1856 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1857 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *)
/* 1858 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1859 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1860 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, EVP_PKEY * *)
/* 1861 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1862 */ _CFFI_OP(_CFFI_OP_NOOP, 210),
/* 1863 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1864 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, int)
/* 1865 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1866 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1867 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1868 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, long)
/* 1869 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1870 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1871 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1872 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, time_t *, long *)
/* 1873 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1874 */ _CFFI_OP(_CFFI_OP_POINTER, 2397), // time_t *
/* 1875 */ _CFFI_OP(_CFFI_OP_POINTER, 28), // long *
/* 1876 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1877 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, unsigned char *, int)
/* 1878 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1879 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1880 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1881 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1882 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL *, unsigned char const * *)
/* 1883 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1884 */ _CFFI_OP(_CFFI_OP_NOOP, 1292),
/* 1885 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1886 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL const *)
/* 1887 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 1888 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1889 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX *)
/* 1890 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1891 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1892 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX *, int(*)(SSL *, void *))
/* 1893 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1894 */ _CFFI_OP(_CFFI_OP_POINTER, 1363), // int(*)(SSL *, void *)
/* 1895 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1896 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX *, int)
/* 1897 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1898 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1899 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1900 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX *, long)
/* 1901 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1902 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1903 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1904 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX *, void *)
/* 1905 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 1906 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1907 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1908 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_CTX const *)
/* 1909 */ _CFFI_OP(_CFFI_OP_NOOP, 494),
/* 1910 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1911 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(SSL_SESSION const *)
/* 1912 */ _CFFI_OP(_CFFI_OP_NOOP, 1472),
/* 1913 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1914 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(X509 *)
/* 1915 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1916 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1917 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(X509_CRL *)
/* 1918 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 1919 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1920 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // long()(X509_REQ *)
/* 1921 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 1922 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1923 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(BIO *)
/* 1924 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 1925 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1926 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(EC_GROUP const *, EC_POINT const *, point_conversion_form_t, unsigned char *, size_t, BN_CTX *)
/* 1927 */ _CFFI_OP(_CFFI_OP_NOOP, 173),
/* 1928 */ _CFFI_OP(_CFFI_OP_NOOP, 179),
/* 1929 */ _CFFI_OP(_CFFI_OP_ENUM, 0), // point_conversion_form_t
/* 1930 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1931 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1932 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 1933 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1934 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(EC_builtin_curve *, size_t)
/* 1935 */ _CFFI_OP(_CFFI_OP_POINTER, 2472), // EC_builtin_curve *
/* 1936 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(EVP_PKEY *, unsigned char * *)
/* 1939 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 1940 */ _CFFI_OP(_CFFI_OP_NOOP, 1604),
/* 1941 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1942 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(SSL *)
/* 1943 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1944 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1945 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(SSL const *, unsigned char *, size_t)
/* 1946 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 1947 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1948 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1949 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1950 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(SSL const *, void *, size_t)
/* 1951 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 1952 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1953 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1954 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1955 */ _CFFI_OP(_CFFI_OP_FUNCTION, 237), // size_t()(SSL_SESSION const *, unsigned char *, size_t)
/* 1956 */ _CFFI_OP(_CFFI_OP_NOOP, 1472),
/* 1957 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1958 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1959 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1960 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1434), // uint32_t()(SSL_SESSION const *)
/* 1961 */ _CFFI_OP(_CFFI_OP_NOOP, 1472),
/* 1962 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1963 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1771), // uint64_t()(SSL_CIPHER const *)
/* 1964 */ _CFFI_OP(_CFFI_OP_NOOP, 536),
/* 1965 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1966 */ _CFFI_OP(_CFFI_OP_FUNCTION, 638), // unsigned char *()(ASN1_OCTET_STRING *)
/* 1967 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 1968 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1969 */ _CFFI_OP(_CFFI_OP_FUNCTION, 638), // unsigned char *()(X509 *, int *)
/* 1970 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 1971 */ _CFFI_OP(_CFFI_OP_NOOP, 862),
/* 1972 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1973 */ _CFFI_OP(_CFFI_OP_FUNCTION, 67), // unsigned char const *()(ASN1_OCTET_STRING const *)
/* 1974 */ _CFFI_OP(_CFFI_OP_NOOP, 1532),
/* 1975 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1976 */ _CFFI_OP(_CFFI_OP_FUNCTION, 67), // unsigned char const *()(SSL_SESSION const *, unsigned int *)
/* 1977 */ _CFFI_OP(_CFFI_OP_NOOP, 1472),
/* 1978 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 1979 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1980 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1094), // unsigned int()(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 1981 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1982 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1983 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 1984 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1985 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1986 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1987 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1988 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1094), // unsigned int()(SSL *, char const *, unsigned char *, unsigned int)
/* 1989 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1990 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 1991 */ _CFFI_OP(_CFFI_OP_NOOP, 638),
/* 1992 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1993 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1994 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL *)
/* 1995 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1996 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1997 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL *, unsigned long)
/* 1998 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 1999 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2000 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2001 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL_CTX *)
/* 2002 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2003 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2004 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL_CTX *, DH *)
/* 2005 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2006 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 2007 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2008 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL_CTX *, EC_KEY *)
/* 2009 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2010 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 2011 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2012 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL_CTX *, X509 *)
/* 2013 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2014 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2015 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2016 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(SSL_CTX *, unsigned long)
/* 2017 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2018 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2019 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2020 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(X509 *)
/* 2021 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2022 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2023 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(X509_NAME *)
/* 2024 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 2025 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2026 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(X509_VERIFY_PARAM *)
/* 2027 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2028 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2029 */ _CFFI_OP(_CFFI_OP_FUNCTION, 545), // unsigned long()(void)
/* 2030 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2031 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(SSL const *, int)
/* 2032 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 2033 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2034 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2035 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(SSL_CTX const *, int)
/* 2036 */ _CFFI_OP(_CFFI_OP_NOOP, 494),
/* 2037 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2038 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2039 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(X509V3_CTX *)
/* 2040 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2041 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2042 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(X509_EXTENSION *)
/* 2043 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 2044 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2045 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(X509_STORE_CTX *, int)
/* 2046 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2047 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2048 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2049 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(size_t)
/* 2050 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2051 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2052 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(size_t, char const *, int)
/* 2053 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2054 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2055 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2056 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2057 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // void *()(void *, size_t, char const *, int)
/* 2058 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2059 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2060 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2061 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2062 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2063 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ASN1_ENUMERATED *)
/* 2064 */ _CFFI_OP(_CFFI_OP_NOOP, 550),
/* 2065 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2066 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ASN1_GENERALIZEDTIME *)
/* 2067 */ _CFFI_OP(_CFFI_OP_POINTER, 2440), // ASN1_GENERALIZEDTIME *
/* 2068 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2069 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ASN1_INTEGER *)
/* 2070 */ _CFFI_OP(_CFFI_OP_NOOP, 8),
/* 2071 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2072 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ASN1_OBJECT *)
/* 2073 */ _CFFI_OP(_CFFI_OP_NOOP, 419),
/* 2074 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2075 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ASN1_OCTET_STRING *)
/* 2076 */ _CFFI_OP(_CFFI_OP_NOOP, 3),
/* 2077 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2078 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BIGNUM *)
/* 2079 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 2080 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2081 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BIGNUM *, int)
/* 2082 */ _CFFI_OP(_CFFI_OP_NOOP, 7),
/* 2083 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2084 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2085 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BIO *)
/* 2086 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 2087 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2088 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BIO_ADDR *)
/* 2089 */ _CFFI_OP(_CFFI_OP_NOOP, 1287),
/* 2090 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2091 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BN_CTX *)
/* 2092 */ _CFFI_OP(_CFFI_OP_NOOP, 58),
/* 2093 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2094 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(BN_MONT_CTX *)
/* 2095 */ _CFFI_OP(_CFFI_OP_NOOP, 603),
/* 2096 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2097 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(CMAC_CTX *)
/* 2098 */ _CFFI_OP(_CFFI_OP_NOOP, 790),
/* 2099 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2100 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(Cryptography_STACK_OF_X509 *)
/* 2101 */ _CFFI_OP(_CFFI_OP_NOOP, 101),
/* 2102 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2103 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(Cryptography_STACK_OF_X509_NAME *)
/* 2104 */ _CFFI_OP(_CFFI_OP_NOOP, 440),
/* 2105 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2106 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DH *)
/* 2107 */ _CFFI_OP(_CFFI_OP_NOOP, 139),
/* 2108 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2109 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DH const *, BIGNUM const * *, BIGNUM const * *)
/* 2110 */ _CFFI_OP(_CFFI_OP_NOOP, 858),
/* 2111 */ _CFFI_OP(_CFFI_OP_POINTER, 56), // BIGNUM const * *
/* 2112 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2113 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2114 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DH const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2115 */ _CFFI_OP(_CFFI_OP_NOOP, 858),
/* 2116 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2117 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2118 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2119 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2120 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DSA *)
/* 2121 */ _CFFI_OP(_CFFI_OP_NOOP, 147),
/* 2122 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2123 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2124 */ _CFFI_OP(_CFFI_OP_NOOP, 888),
/* 2125 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2126 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2127 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2128 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2129 */ _CFFI_OP(_CFFI_OP_NOOP, 888),
/* 2130 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2131 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2132 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2133 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2134 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EC_GROUP *)
/* 2135 */ _CFFI_OP(_CFFI_OP_POINTER, 2468), // EC_GROUP *
/* 2136 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2137 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EC_KEY *)
/* 2138 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 2139 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2140 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EC_KEY *, int)
/* 2141 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 2142 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2143 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2144 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EC_POINT *)
/* 2145 */ _CFFI_OP(_CFFI_OP_NOOP, 900),
/* 2146 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2147 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(ENGINE *)
/* 2148 */ _CFFI_OP(_CFFI_OP_NOOP, 219),
/* 2149 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2150 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EVP_CIPHER *)
/* 2151 */ _CFFI_OP(_CFFI_OP_POINTER, 2474), // EVP_CIPHER *
/* 2152 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2153 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EVP_CIPHER_CTX *)
/* 2154 */ _CFFI_OP(_CFFI_OP_NOOP, 1004),
/* 2155 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2156 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EVP_MD_CTX *)
/* 2157 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 2158 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2159 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EVP_PKEY *)
/* 2160 */ _CFFI_OP(_CFFI_OP_NOOP, 142),
/* 2161 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2162 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(EVP_PKEY_CTX *)
/* 2163 */ _CFFI_OP(_CFFI_OP_NOOP, 1143),
/* 2164 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2165 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(GENERAL_NAMES *)
/* 2166 */ _CFFI_OP(_CFFI_OP_NOOP, 250),
/* 2167 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2168 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(HMAC_CTX *)
/* 2169 */ _CFFI_OP(_CFFI_OP_NOOP, 1193),
/* 2170 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2171 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(NETSCAPE_SPKI *)
/* 2172 */ _CFFI_OP(_CFFI_OP_NOOP, 225),
/* 2173 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2174 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(PKCS12 *)
/* 2175 */ _CFFI_OP(_CFFI_OP_NOOP, 705),
/* 2176 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2177 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(PKCS7 *)
/* 2178 */ _CFFI_OP(_CFFI_OP_NOOP, 100),
/* 2179 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2180 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(RSA *)
/* 2181 */ _CFFI_OP(_CFFI_OP_NOOP, 323),
/* 2182 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2183 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2184 */ _CFFI_OP(_CFFI_OP_NOOP, 732),
/* 2185 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2186 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2187 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2188 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2189 */ _CFFI_OP(_CFFI_OP_NOOP, 732),
/* 2190 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2191 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2192 */ _CFFI_OP(_CFFI_OP_NOOP, 2111),
/* 2193 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2194 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *)
/* 2195 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2196 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2197 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, BIO *, BIO *)
/* 2198 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2199 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 2200 */ _CFFI_OP(_CFFI_OP_NOOP, 129),
/* 2201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, char *)
/* 2203 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2204 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 2205 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2206 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, int(*)(SSL *, void *), void *)
/* 2207 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2208 */ _CFFI_OP(_CFFI_OP_NOOP, 1894),
/* 2209 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2210 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2211 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, int)
/* 2212 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2213 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2214 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2215 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, int, int(*)(int, X509_STORE_CTX *))
/* 2216 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2217 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2218 */ _CFFI_OP(_CFFI_OP_POINTER, 1781), // int(*)(int, X509_STORE_CTX *)
/* 2219 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2220 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL *, unsigned int, unsigned char const *, void *)
/* 2221 */ _CFFI_OP(_CFFI_OP_NOOP, 328),
/* 2222 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2223 */ _CFFI_OP(_CFFI_OP_NOOP, 67),
/* 2224 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2225 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2226 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL const *, char const *)
/* 2227 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 2228 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2229 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2230 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL const *, int, int)
/* 2231 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 2232 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2233 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2234 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2235 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL const *, unsigned char const * *, unsigned int *)
/* 2236 */ _CFFI_OP(_CFFI_OP_NOOP, 105),
/* 2237 */ _CFFI_OP(_CFFI_OP_NOOP, 1292),
/* 2238 */ _CFFI_OP(_CFFI_OP_NOOP, 1076),
/* 2239 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2240 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *)
/* 2241 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2242 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2243 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, Cryptography_STACK_OF_X509_NAME *)
/* 2244 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2245 */ _CFFI_OP(_CFFI_OP_NOOP, 440),
/* 2246 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2247 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, X509_STORE *)
/* 2248 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2249 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 2250 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2251 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *))
/* 2252 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2253 */ _CFFI_OP(_CFFI_OP_POINTER, 1289), // int(*)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *)
/* 2254 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2255 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, int *, void *))
/* 2256 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2257 */ _CFFI_OP(_CFFI_OP_POINTER, 1304), // int(*)(SSL *, int *, void *)
/* 2258 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2259 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, unsigned char *, unsigned int *))
/* 2260 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2261 */ _CFFI_OP(_CFFI_OP_POINTER, 1323), // int(*)(SSL *, unsigned char *, unsigned int *)
/* 2262 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2263 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *), void *)
/* 2264 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2265 */ _CFFI_OP(_CFFI_OP_POINTER, 1328), // int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 2266 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2267 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2268 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const *, size_t, SSL_SESSION * *))
/* 2269 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2270 */ _CFFI_OP(_CFFI_OP_POINTER, 1336), // int(*)(SSL *, unsigned char const *, size_t, SSL_SESSION * *)
/* 2271 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2272 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const *, unsigned int))
/* 2273 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2274 */ _CFFI_OP(_CFFI_OP_POINTER, 1342), // int(*)(SSL *, unsigned char const *, unsigned int)
/* 2275 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2276 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(SSL *, void *), void *)
/* 2277 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2278 */ _CFFI_OP(_CFFI_OP_NOOP, 1894),
/* 2279 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2280 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2281 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(X509_STORE_CTX *, void *), void *)
/* 2282 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2283 */ _CFFI_OP(_CFFI_OP_POINTER, 1707), // int(*)(X509_STORE_CTX *, void *)
/* 2284 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2285 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2286 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int(*)(char *, int, int, void *))
/* 2287 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2288 */ _CFFI_OP(_CFFI_OP_NOOP, 135),
/* 2289 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2290 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int)
/* 2291 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2292 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2293 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2294 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, int, int(*)(int, X509_STORE_CTX *))
/* 2295 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2296 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2297 */ _CFFI_OP(_CFFI_OP_NOOP, 2218),
/* 2298 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2299 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))
/* 2300 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2301 */ _CFFI_OP(_CFFI_OP_POINTER, 1980), // unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 2302 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2303 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int))
/* 2304 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2305 */ _CFFI_OP(_CFFI_OP_POINTER, 1988), // unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int)
/* 2306 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2307 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, void *)
/* 2308 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2309 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2310 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2311 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, void(*)(SSL const *, char const *))
/* 2312 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2313 */ _CFFI_OP(_CFFI_OP_POINTER, 2226), // void(*)(SSL const *, char const *)
/* 2314 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2315 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_CTX *, void(*)(SSL const *, int, int))
/* 2316 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2317 */ _CFFI_OP(_CFFI_OP_POINTER, 2230), // void(*)(SSL const *, int, int)
/* 2318 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2319 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(SSL_SESSION *)
/* 2320 */ _CFFI_OP(_CFFI_OP_NOOP, 1298),
/* 2321 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2322 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509 *)
/* 2323 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2324 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2325 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509V3_CTX *, X509 *, X509 *, X509_REQ *, X509_CRL *, int)
/* 2326 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2327 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2328 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2329 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 2330 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 2331 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2332 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2333 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_CRL *)
/* 2334 */ _CFFI_OP(_CFFI_OP_NOOP, 34),
/* 2335 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2336 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_EXTENSION *)
/* 2337 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 2338 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2339 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_EXTENSIONS *)
/* 2340 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 2341 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2342 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_EXTENSIONS *, void(*)(X509_EXTENSION *))
/* 2343 */ _CFFI_OP(_CFFI_OP_NOOP, 427),
/* 2344 */ _CFFI_OP(_CFFI_OP_POINTER, 2336), // void(*)(X509_EXTENSION *)
/* 2345 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2346 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_NAME *)
/* 2347 */ _CFFI_OP(_CFFI_OP_NOOP, 450),
/* 2348 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2349 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_NAME_ENTRY *)
/* 2350 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 2351 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2352 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_REQ *)
/* 2353 */ _CFFI_OP(_CFFI_OP_NOOP, 231),
/* 2354 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2355 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_REVOKED *)
/* 2356 */ _CFFI_OP(_CFFI_OP_NOOP, 431),
/* 2357 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2358 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE *)
/* 2359 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 2360 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2361 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE *, int(*)(X509 * *, X509_STORE_CTX *, X509 *))
/* 2362 */ _CFFI_OP(_CFFI_OP_NOOP, 123),
/* 2363 */ _CFFI_OP(_CFFI_OP_POINTER, 1474), // int(*)(X509 * *, X509_STORE_CTX *, X509 *)
/* 2364 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2365 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *)
/* 2366 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2367 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2368 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *, Cryptography_STACK_OF_X509_CRL *)
/* 2369 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2370 */ _CFFI_OP(_CFFI_OP_POINTER, 2461), // Cryptography_STACK_OF_X509_CRL *
/* 2371 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2372 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *, X509 *)
/* 2373 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2374 */ _CFFI_OP(_CFFI_OP_NOOP, 11),
/* 2375 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2376 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *, X509_VERIFY_PARAM *)
/* 2377 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2378 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2379 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2380 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *, int(*)(int, X509_STORE_CTX *))
/* 2381 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2382 */ _CFFI_OP(_CFFI_OP_NOOP, 2218),
/* 2383 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2384 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_STORE_CTX *, int)
/* 2385 */ _CFFI_OP(_CFFI_OP_NOOP, 108),
/* 2386 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2387 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2388 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_VERIFY_PARAM *)
/* 2389 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2390 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2391 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_VERIFY_PARAM *, int)
/* 2392 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2393 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2394 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2395 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_VERIFY_PARAM *, time_t)
/* 2396 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2397 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(time_t), (
           ((time_t)-1) | 0 /* check that time_t is an integer type */
         ) <= 0)), // time_t
/* 2398 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2399 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(X509_VERIFY_PARAM *, unsigned int)
/* 2400 */ _CFFI_OP(_CFFI_OP_NOOP, 1674),
/* 2401 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2402 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2403 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(char const *)
/* 2404 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2405 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2406 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(int, int, int, char const *, int)
/* 2407 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2408 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2409 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2410 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2411 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2412 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2413 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(unsigned long, char *, size_t)
/* 2414 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2415 */ _CFFI_OP(_CFFI_OP_NOOP, 270),
/* 2416 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2417 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2418 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(void *)
/* 2419 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2420 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2421 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(void *, char const *, int)
/* 2422 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2423 */ _CFFI_OP(_CFFI_OP_NOOP, 23),
/* 2424 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2425 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2426 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(void const *, int, double)
/* 2427 */ _CFFI_OP(_CFFI_OP_NOOP, 86),
/* 2428 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2429 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 14), // double
/* 2430 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2431 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2534), // void()(void)
/* 2432 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2433 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2313), // void(*()(SSL_CTX *))(SSL const *, char const *)
/* 2434 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2435 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2436 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2317), // void(*()(SSL_CTX *))(SSL const *, int, int)
/* 2437 */ _CFFI_OP(_CFFI_OP_NOOP, 331),
/* 2438 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2439 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 2), // ASN1_ENUMERATED
/* 2440 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 3), // ASN1_GENERALIZEDTIME
/* 2441 */ _CFFI_OP(_CFFI_OP_POINTER, 2442), // ASN1_INTEGER const *
/* 2442 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 4), // ASN1_INTEGER
/* 2443 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 5), // ASN1_OBJECT
/* 2444 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 79), // ASN1_OCTET_STRING
/* 2445 */ _CFFI_OP(_CFFI_OP_POINTER, 2446), // ASN1_TYPE *
/* 2446 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 6), // ASN1_TYPE
/* 2447 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 7), // BIGNUM
/* 2448 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 8), // BIO
/* 2449 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 9), // BIO_ADDR
/* 2450 */ _CFFI_OP(_CFFI_OP_POINTER, 2451), // BIO_METHOD const *
/* 2451 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 10), // BIO_METHOD
/* 2452 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 11), // BN_CTX
/* 2453 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 12), // BN_GENCB
/* 2454 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 13), // BN_MONT_CTX
/* 2455 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 14), // CMAC_CTX
/* 2456 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 15), // CONF
/* 2457 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 16), // CRYPTOGRAPHY_PASSWORD_DATA
/* 2458 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 17), // Cryptography_STACK_OF_ASN1_OBJECT
/* 2459 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 18), // Cryptography_STACK_OF_SSL_CIPHER
/* 2460 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 19), // Cryptography_STACK_OF_X509
/* 2461 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 20), // Cryptography_STACK_OF_X509_CRL
/* 2462 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 21), // Cryptography_STACK_OF_X509_NAME
/* 2463 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 22), // Cryptography_STACK_OF_X509_NAME_ENTRY
/* 2464 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 23), // Cryptography_STACK_OF_X509_OBJECT
/* 2465 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 24), // Cryptography_STACK_OF_X509_REVOKED
/* 2466 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 25), // DH
/* 2467 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 26), // DSA
/* 2468 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 27), // EC_GROUP
/* 2469 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 28), // EC_KEY
/* 2470 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 29), // EC_METHOD
/* 2471 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 30), // EC_POINT
/* 2472 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 31), // EC_builtin_curve
/* 2473 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 32), // ENGINE
/* 2474 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 33), // EVP_CIPHER
/* 2475 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 34), // EVP_CIPHER_CTX
/* 2476 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 35), // EVP_MD
/* 2477 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 36), // EVP_MD_CTX
/* 2478 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 37), // EVP_PKEY
/* 2479 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 38), // EVP_PKEY_CTX
/* 2480 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 39), // EXTENDED_KEY_USAGE
/* 2481 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 40), // GENERAL_NAME
/* 2482 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 80), // GENERAL_NAMES
/* 2483 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 41), // HMAC_CTX
/* 2484 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 42), // NETSCAPE_SPKI
/* 2485 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 43), // OBJ_NAME
/* 2486 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 44), // OSSL_LIB_CTX
/* 2487 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 45), // OSSL_PROVIDER
/* 2488 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 46), // PKCS12
/* 2489 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 47), // PKCS7
/* 2490 */ _CFFI_OP(_CFFI_OP_POINTER, 2491), // PKCS7_DIGEST *
/* 2491 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 48), // PKCS7_DIGEST
/* 2492 */ _CFFI_OP(_CFFI_OP_POINTER, 2493), // PKCS7_ENCRYPT *
/* 2493 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 49), // PKCS7_ENCRYPT
/* 2494 */ _CFFI_OP(_CFFI_OP_POINTER, 2495), // PKCS7_ENVELOPE *
/* 2495 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 50), // PKCS7_ENVELOPE
/* 2496 */ _CFFI_OP(_CFFI_OP_POINTER, 2497), // PKCS7_SIGNED *
/* 2497 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 51), // PKCS7_SIGNED
/* 2498 */ _CFFI_OP(_CFFI_OP_POINTER, 2499), // PKCS7_SIGNER_INFO *
/* 2499 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 52), // PKCS7_SIGNER_INFO
/* 2500 */ _CFFI_OP(_CFFI_OP_POINTER, 2501), // PKCS7_SIGN_ENVELOPE *
/* 2501 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 53), // PKCS7_SIGN_ENVELOPE
/* 2502 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 54), // PKCS8_PRIV_KEY_INFO
/* 2503 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 55), // RAND_METHOD
/* 2504 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 56), // RSA
/* 2505 */ _CFFI_OP(_CFFI_OP_POINTER, 2506), // SRTP_PROTECTION_PROFILE *
/* 2506 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 57), // SRTP_PROTECTION_PROFILE
/* 2507 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 58), // SSL
/* 2508 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 59), // SSL_CIPHER
/* 2509 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 60), // SSL_CTX
/* 2510 */ _CFFI_OP(_CFFI_OP_POINTER, 2511), // SSL_METHOD const *
/* 2511 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 61), // SSL_METHOD
/* 2512 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 62), // SSL_SESSION
/* 2513 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 63), // UI_METHOD
/* 2514 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 64), // X509
/* 2515 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 65), // X509V3_CTX
/* 2516 */ _CFFI_OP(_CFFI_OP_POINTER, 2517), // X509_ALGOR const *
/* 2517 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 66), // X509_ALGOR
/* 2518 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 67), // X509_ATTRIBUTE
/* 2519 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 68), // X509_CRL
/* 2520 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 69), // X509_EXTENSION
/* 2521 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 70), // X509_EXTENSIONS
/* 2522 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 71), // X509_NAME
/* 2523 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 72), // X509_NAME_ENTRY
/* 2524 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 73), // X509_OBJECT
/* 2525 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 74), // X509_REQ
/* 2526 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 75), // X509_REVOKED
/* 2527 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 76), // X509_STORE
/* 2528 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 77), // X509_STORE_CTX
/* 2529 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 78), // X509_VERIFY_PARAM
/* 2530 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 2), // char
/* 2531 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 0), // union $1
/* 2532 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 1), // union $2
/* 2533 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 4), // unsigned char
/* 2534 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 0), // void
};

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ASN1_TYPE(ASN1_TYPE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'ASN1_TYPE.type' is an integer */
}
struct _cffi_align_typedef_ASN1_TYPE { char x; ASN1_TYPE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_CRYPTOGRAPHY_PASSWORD_DATA(CRYPTOGRAPHY_PASSWORD_DATA *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char * *tmp = &p->password; (void)tmp; }
  (void)((p->length) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.length' is an integer */
  (void)((p->called) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.called' is an integer */
  (void)((p->error) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.error' is an integer */
  (void)((p->maxsize) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.maxsize' is an integer */
}
struct _cffi_align_typedef_CRYPTOGRAPHY_PASSWORD_DATA { char x; CRYPTOGRAPHY_PASSWORD_DATA y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_EC_builtin_curve(EC_builtin_curve *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->nid) | 0);  /* check that 'EC_builtin_curve.nid' is an integer */
  { char const * *tmp = &p->comment; (void)tmp; }
}
struct _cffi_align_typedef_EC_builtin_curve { char x; EC_builtin_curve y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_GENERAL_NAME(GENERAL_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'GENERAL_NAME.type' is an integer */
  /* cannot generate 'union $1' in field 'd': unknown type name */
}
struct _cffi_align_typedef_GENERAL_NAME { char x; GENERAL_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_OBJ_NAME(OBJ_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'OBJ_NAME.type' is an integer */
  (void)((p->alias) | 0);  /* check that 'OBJ_NAME.alias' is an integer */
  { char const * *tmp = &p->name; (void)tmp; }
  { char const * *tmp = &p->data; (void)tmp; }
}
struct _cffi_align_typedef_OBJ_NAME { char x; OBJ_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7(PKCS7 *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->type; (void)tmp; }
  /* cannot generate 'union $2' in field 'd': unknown type name */
}
struct _cffi_align_typedef_PKCS7 { char x; PKCS7 y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGNED(PKCS7_SIGNED *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGNED { char x; PKCS7_SIGNED y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGN_ENVELOPE(PKCS7_SIGN_ENVELOPE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGN_ENVELOPE { char x; PKCS7_SIGN_ENVELOPE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_SRTP_PROTECTION_PROFILE(SRTP_PROTECTION_PROFILE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char const * *tmp = &p->name; (void)tmp; }
  (void)((p->id) | 0);  /* check that 'SRTP_PROTECTION_PROFILE.id' is an integer */
}
struct _cffi_align_typedef_SRTP_PROTECTION_PROFILE { char x; SRTP_PROTECTION_PROFILE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509V3_CTX(X509V3_CTX *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { X509 * *tmp = &p->issuer_cert; (void)tmp; }
  { X509 * *tmp = &p->subject_cert; (void)tmp; }
}
struct _cffi_align_typedef_X509V3_CTX { char x; X509V3_CTX y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509_ALGOR(X509_ALGOR *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->algorithm; (void)tmp; }
}
struct _cffi_align_typedef_X509_ALGOR { char x; X509_ALGOR y; };

static int _cffi_const_POINT_CONVERSION_COMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_COMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_COMPRESSED) | 0);  /* check that POINT_CONVERSION_COMPRESSED is an integer */
  return n;
}

static int _cffi_const_POINT_CONVERSION_UNCOMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_UNCOMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_UNCOMPRESSED) | 0);  /* check that POINT_CONVERSION_UNCOMPRESSED is an integer */
  return n;
}

static int _cffi_const_CIPHER_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (CIPHER_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((CIPHER_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) | 0);  /* check that CIPHER_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_IS_BORINGSSL(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_IS_BORINGSSL) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_IS_BORINGSSL) | 0);  /* check that CRYPTOGRAPHY_IS_BORINGSSL is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_IS_LIBRESSL(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_IS_LIBRESSL) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_IS_LIBRESSL) | 0);  /* check that CRYPTOGRAPHY_IS_LIBRESSL is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340) | 0);  /* check that CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350) | 0);  /* check that CRYPTOGRAPHY_LIBRESSL_LESS_THAN_350 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) | 0);  /* check that CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_300_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_300_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_300_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_300_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111E(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111E) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111E) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111E is an integer */
  return n;
}

static void _cffi_const_CRYPTOGRAPHY_PACKAGE_VERSION(char *o)
{
  *(char const * *)o = CRYPTOGRAPHY_PACKAGE_VERSION;
}

static int _cffi_const_Cryptography_HAS_110_VERIFICATION_PARAMS(unsigned long long *o)
{
  int n = (Cryptography_HAS_110_VERIFICATION_PARAMS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_110_VERIFICATION_PARAMS) | 0);  /* check that Cryptography_HAS_110_VERIFICATION_PARAMS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_300_EVP_CIPHER(unsigned long long *o)
{
  int n = (Cryptography_HAS_300_EVP_CIPHER) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_300_EVP_CIPHER) | 0);  /* check that Cryptography_HAS_300_EVP_CIPHER is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_300_FIPS(unsigned long long *o)
{
  int n = (Cryptography_HAS_300_FIPS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_300_FIPS) | 0);  /* check that Cryptography_HAS_300_FIPS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ALPN(unsigned long long *o)
{
  int n = (Cryptography_HAS_ALPN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ALPN) | 0);  /* check that Cryptography_HAS_ALPN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_BN_FLAGS(unsigned long long *o)
{
  int n = (Cryptography_HAS_BN_FLAGS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_BN_FLAGS) | 0);  /* check that Cryptography_HAS_BN_FLAGS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_CUSTOM_EXT(unsigned long long *o)
{
  int n = (Cryptography_HAS_CUSTOM_EXT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_CUSTOM_EXT) | 0);  /* check that Cryptography_HAS_CUSTOM_EXT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_DTLS(unsigned long long *o)
{
  int n = (Cryptography_HAS_DTLS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_DTLS) | 0);  /* check that Cryptography_HAS_DTLS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_DTLS_GET_DATA_MTU(unsigned long long *o)
{
  int n = (Cryptography_HAS_DTLS_GET_DATA_MTU) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_DTLS_GET_DATA_MTU) | 0);  /* check that Cryptography_HAS_DTLS_GET_DATA_MTU is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EC2M(unsigned long long *o)
{
  int n = (Cryptography_HAS_EC2M) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EC2M) | 0);  /* check that Cryptography_HAS_EC2M is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED25519(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED25519) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED25519) | 0);  /* check that Cryptography_HAS_ED25519 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED448(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED448) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED448) | 0);  /* check that Cryptography_HAS_ED448 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ENGINE(unsigned long long *o)
{
  int n = (Cryptography_HAS_ENGINE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ENGINE) | 0);  /* check that Cryptography_HAS_ENGINE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_DIGESTFINAL_XOF(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_DIGESTFINAL_XOF) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_DIGESTFINAL_XOF) | 0);  /* check that Cryptography_HAS_EVP_DIGESTFINAL_XOF is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_DH(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_DH) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_DH) | 0);  /* check that Cryptography_HAS_EVP_PKEY_DH is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_DHX) | 0);  /* check that Cryptography_HAS_EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) | 0);  /* check that Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_FIPS(unsigned long long *o)
{
  int n = (Cryptography_HAS_FIPS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_FIPS) | 0);  /* check that Cryptography_HAS_FIPS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_GET_PROTO_VERSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_GET_PROTO_VERSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_GET_PROTO_VERSION) | 0);  /* check that Cryptography_HAS_GET_PROTO_VERSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_KEYLOG(unsigned long long *o)
{
  int n = (Cryptography_HAS_KEYLOG) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_KEYLOG) | 0);  /* check that Cryptography_HAS_KEYLOG is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_MEM_FUNCTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_MEM_FUNCTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_MEM_FUNCTIONS) | 0);  /* check that Cryptography_HAS_MEM_FUNCTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_NEXTPROTONEG(unsigned long long *o)
{
  int n = (Cryptography_HAS_NEXTPROTONEG) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_NEXTPROTONEG) | 0);  /* check that Cryptography_HAS_NEXTPROTONEG is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY(unsigned long long *o)
{
  int n = (Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) | 0);  /* check that Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OPENSSL_CLEANUP(unsigned long long *o)
{
  int n = (Cryptography_HAS_OPENSSL_CLEANUP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OPENSSL_CLEANUP) | 0);  /* check that Cryptography_HAS_OPENSSL_CLEANUP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OP_NO_COMPRESSION) | 0);  /* check that Cryptography_HAS_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OP_NO_RENEGOTIATION(unsigned long long *o)
{
  int n = (Cryptography_HAS_OP_NO_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OP_NO_RENEGOTIATION) | 0);  /* check that Cryptography_HAS_OP_NO_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PKCS7_FUNCS(unsigned long long *o)
{
  int n = (Cryptography_HAS_PKCS7_FUNCS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PKCS7_FUNCS) | 0);  /* check that Cryptography_HAS_PKCS7_FUNCS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_POLY1305(unsigned long long *o)
{
  int n = (Cryptography_HAS_POLY1305) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_POLY1305) | 0);  /* check that Cryptography_HAS_POLY1305 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PROVIDERS(unsigned long long *o)
{
  int n = (Cryptography_HAS_PROVIDERS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PROVIDERS) | 0);  /* check that Cryptography_HAS_PROVIDERS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PSK(unsigned long long *o)
{
  int n = (Cryptography_HAS_PSK) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PSK) | 0);  /* check that Cryptography_HAS_PSK is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PSK_TLSv1_3(unsigned long long *o)
{
  int n = (Cryptography_HAS_PSK_TLSv1_3) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PSK_TLSv1_3) | 0);  /* check that Cryptography_HAS_PSK_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RAW_KEY(unsigned long long *o)
{
  int n = (Cryptography_HAS_RAW_KEY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RAW_KEY) | 0);  /* check that Cryptography_HAS_RAW_KEY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (Cryptography_HAS_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RELEASE_BUFFERS) | 0);  /* check that Cryptography_HAS_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SCRYPT(unsigned long long *o)
{
  int n = (Cryptography_HAS_SCRYPT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SCRYPT) | 0);  /* check that Cryptography_HAS_SCRYPT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SECURE_RENEGOTIATION(unsigned long long *o)
{
  int n = (Cryptography_HAS_SECURE_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SECURE_RENEGOTIATION) | 0);  /* check that Cryptography_HAS_SECURE_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SET_CERT_CB(unsigned long long *o)
{
  int n = (Cryptography_HAS_SET_CERT_CB) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SET_CERT_CB) | 0);  /* check that Cryptography_HAS_SET_CERT_CB is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SRTP(unsigned long long *o)
{
  int n = (Cryptography_HAS_SRTP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SRTP) | 0);  /* check that Cryptography_HAS_SRTP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL3_METHOD(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL3_METHOD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL3_METHOD) | 0);  /* check that Cryptography_HAS_SSL3_METHOD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_COOKIE(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_COOKIE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_COOKIE) | 0);  /* check that Cryptography_HAS_SSL_COOKIE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) | 0);  /* check that Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_NO_TICKET) | 0);  /* check that Cryptography_HAS_SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_SET_SSL_CTX(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_SET_SSL_CTX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_SET_SSL_CTX) | 0);  /* check that Cryptography_HAS_SSL_SET_SSL_CTX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_ST) | 0);  /* check that Cryptography_HAS_SSL_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSEXT_HOSTNAME(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSEXT_HOSTNAME) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSEXT_HOSTNAME) | 0);  /* check that Cryptography_HAS_TLSEXT_HOSTNAME is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLS_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLS_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLS_ST) | 0);  /* check that Cryptography_HAS_TLS_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_1(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_1) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_1) | 0);  /* check that Cryptography_HAS_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_2(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_2) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_2) | 0);  /* check that Cryptography_HAS_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_3(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_3) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_3) | 0);  /* check that Cryptography_HAS_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_3_FUNCTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_3_FUNCTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_3_FUNCTIONS) | 0);  /* check that Cryptography_HAS_TLSv1_3_FUNCTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING(unsigned long long *o)
{
  int n = (Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING) | 0);  /* check that Cryptography_HAS_UNEXPECTED_EOF_WHILE_READING is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_VERIFIED_CHAIN(unsigned long long *o)
{
  int n = (Cryptography_HAS_VERIFIED_CHAIN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_VERIFIED_CHAIN) | 0);  /* check that Cryptography_HAS_VERIFIED_CHAIN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_STORE_CTX_GET_ISSUER(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) | 0);  /* check that Cryptography_HAS_X509_STORE_CTX_GET_ISSUER is an integer */
  return n;
}

static void _cffi_const_Cryptography_osrandom_engine_id(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_id;
}

static void _cffi_const_Cryptography_osrandom_engine_name(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_name;
}

static int _cffi_const_DH_NOT_SUITABLE_GENERATOR(unsigned long long *o)
{
  int n = (DH_NOT_SUITABLE_GENERATOR) <= 0;
  *o = (unsigned long long)((DH_NOT_SUITABLE_GENERATOR) | 0);  /* check that DH_NOT_SUITABLE_GENERATOR is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ASN1(unsigned long long *o)
{
  int n = (ERR_LIB_ASN1) <= 0;
  *o = (unsigned long long)((ERR_LIB_ASN1) | 0);  /* check that ERR_LIB_ASN1 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_EVP(unsigned long long *o)
{
  int n = (ERR_LIB_EVP) <= 0;
  *o = (unsigned long long)((ERR_LIB_EVP) | 0);  /* check that ERR_LIB_EVP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PEM(unsigned long long *o)
{
  int n = (ERR_LIB_PEM) <= 0;
  *o = (unsigned long long)((ERR_LIB_PEM) | 0);  /* check that ERR_LIB_PEM is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PKCS12(unsigned long long *o)
{
  int n = (ERR_LIB_PKCS12) <= 0;
  *o = (unsigned long long)((ERR_LIB_PKCS12) | 0);  /* check that ERR_LIB_PKCS12 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PROV(unsigned long long *o)
{
  int n = (ERR_LIB_PROV) <= 0;
  *o = (unsigned long long)((ERR_LIB_PROV) | 0);  /* check that ERR_LIB_PROV is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_GET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_GET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_GET_TAG) | 0);  /* check that EVP_CTRL_AEAD_GET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_IVLEN(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_IVLEN) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_IVLEN) | 0);  /* check that EVP_CTRL_AEAD_SET_IVLEN is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_TAG) | 0);  /* check that EVP_CTRL_AEAD_SET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_F_EVP_ENCRYPTFINAL_EX(unsigned long long *o)
{
  int n = (EVP_F_EVP_ENCRYPTFINAL_EX) <= 0;
  *o = (unsigned long long)((EVP_F_EVP_ENCRYPTFINAL_EX) | 0);  /* check that EVP_F_EVP_ENCRYPTFINAL_EX is an integer */
  return n;
}

static int _cffi_const_EVP_MAX_MD_SIZE(unsigned long long *o)
{
  int n = (EVP_MAX_MD_SIZE) <= 0;
  *o = (unsigned long long)((EVP_MAX_MD_SIZE) | 0);  /* check that EVP_MAX_MD_SIZE is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DH(unsigned long long *o)
{
  int n = (EVP_PKEY_DH) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DH) | 0);  /* check that EVP_PKEY_DH is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DHX) | 0);  /* check that EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DSA(unsigned long long *o)
{
  int n = (EVP_PKEY_DSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DSA) | 0);  /* check that EVP_PKEY_DSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_EC(unsigned long long *o)
{
  int n = (EVP_PKEY_EC) <= 0;
  *o = (unsigned long long)((EVP_PKEY_EC) | 0);  /* check that EVP_PKEY_EC is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED25519(unsigned long long *o)
{
  int n = (EVP_PKEY_ED25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED25519) | 0);  /* check that EVP_PKEY_ED25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED448(unsigned long long *o)
{
  int n = (EVP_PKEY_ED448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED448) | 0);  /* check that EVP_PKEY_ED448 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_POLY1305(unsigned long long *o)
{
  int n = (EVP_PKEY_POLY1305) <= 0;
  *o = (unsigned long long)((EVP_PKEY_POLY1305) | 0);  /* check that EVP_PKEY_POLY1305 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_RSA(unsigned long long *o)
{
  int n = (EVP_PKEY_RSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_RSA) | 0);  /* check that EVP_PKEY_RSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_RSA_PSS(unsigned long long *o)
{
  int n = (EVP_PKEY_RSA_PSS) <= 0;
  *o = (unsigned long long)((EVP_PKEY_RSA_PSS) | 0);  /* check that EVP_PKEY_RSA_PSS is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X25519(unsigned long long *o)
{
  int n = (EVP_PKEY_X25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X25519) | 0);  /* check that EVP_PKEY_X25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X448(unsigned long long *o)
{
  int n = (EVP_PKEY_X448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X448) | 0);  /* check that EVP_PKEY_X448 is an integer */
  return n;
}

static int _cffi_const_EVP_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (EVP_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((EVP_R_BAD_DECRYPT) | 0);  /* check that EVP_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) | 0);  /* check that EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) | 0);  /* check that EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_EVP_R_XTS_DUPLICATED_KEYS(unsigned long long *o)
{
  int n = (EVP_R_XTS_DUPLICATED_KEYS) <= 0;
  *o = (unsigned long long)((EVP_R_XTS_DUPLICATED_KEYS) | 0);  /* check that EVP_R_XTS_DUPLICATED_KEYS is an integer */
  return n;
}

static int _cffi_const_GEN_DNS(unsigned long long *o)
{
  int n = (GEN_DNS) <= 0;
  *o = (unsigned long long)((GEN_DNS) | 0);  /* check that GEN_DNS is an integer */
  return n;
}

static int _cffi_const_GEN_EMAIL(unsigned long long *o)
{
  int n = (GEN_EMAIL) <= 0;
  *o = (unsigned long long)((GEN_EMAIL) | 0);  /* check that GEN_EMAIL is an integer */
  return n;
}

static int _cffi_const_GEN_URI(unsigned long long *o)
{
  int n = (GEN_URI) <= 0;
  *o = (unsigned long long)((GEN_URI) | 0);  /* check that GEN_URI is an integer */
  return n;
}

static int _cffi_const_MBSTRING_UTF8(unsigned long long *o)
{
  int n = (MBSTRING_UTF8) <= 0;
  *o = (unsigned long long)((MBSTRING_UTF8) | 0);  /* check that MBSTRING_UTF8 is an integer */
  return n;
}

static int _cffi_const_NID_ED25519(unsigned long long *o)
{
  int n = (NID_ED25519) <= 0;
  *o = (unsigned long long)((NID_ED25519) | 0);  /* check that NID_ED25519 is an integer */
  return n;
}

static int _cffi_const_NID_ED448(unsigned long long *o)
{
  int n = (NID_ED448) <= 0;
  *o = (unsigned long long)((NID_ED448) | 0);  /* check that NID_ED448 is an integer */
  return n;
}

static int _cffi_const_NID_X25519(unsigned long long *o)
{
  int n = (NID_X25519) <= 0;
  *o = (unsigned long long)((NID_X25519) | 0);  /* check that NID_X25519 is an integer */
  return n;
}

static int _cffi_const_NID_X448(unsigned long long *o)
{
  int n = (NID_X448) <= 0;
  *o = (unsigned long long)((NID_X448) | 0);  /* check that NID_X448 is an integer */
  return n;
}

static int _cffi_const_NID_crl_reason(unsigned long long *o)
{
  int n = (NID_crl_reason) <= 0;
  *o = (unsigned long long)((NID_crl_reason) | 0);  /* check that NID_crl_reason is an integer */
  return n;
}

static int _cffi_const_NID_pbe_WithSHA1And3_Key_TripleDES_CBC(unsigned long long *o)
{
  int n = (NID_pbe_WithSHA1And3_Key_TripleDES_CBC) <= 0;
  *o = (unsigned long long)((NID_pbe_WithSHA1And3_Key_TripleDES_CBC) | 0);  /* check that NID_pbe_WithSHA1And3_Key_TripleDES_CBC is an integer */
  return n;
}

static int _cffi_const_NID_pkcs7_signed(unsigned long long *o)
{
  int n = (NID_pkcs7_signed) <= 0;
  *o = (unsigned long long)((NID_pkcs7_signed) | 0);  /* check that NID_pkcs7_signed is an integer */
  return n;
}

static int _cffi_const_NID_poly1305(unsigned long long *o)
{
  int n = (NID_poly1305) <= 0;
  *o = (unsigned long long)((NID_poly1305) | 0);  /* check that NID_poly1305 is an integer */
  return n;
}

static int _cffi_const_NID_subject_alt_name(unsigned long long *o)
{
  int n = (NID_subject_alt_name) <= 0;
  *o = (unsigned long long)((NID_subject_alt_name) | 0);  /* check that NID_subject_alt_name is an integer */
  return n;
}

static int _cffi_const_NID_undef(unsigned long long *o)
{
  int n = (NID_undef) <= 0;
  *o = (unsigned long long)((NID_undef) | 0);  /* check that NID_undef is an integer */
  return n;
}

static int _cffi_const_OBJ_NAME_TYPE_MD_METH(unsigned long long *o)
{
  int n = (OBJ_NAME_TYPE_MD_METH) <= 0;
  *o = (unsigned long long)((OBJ_NAME_TYPE_MD_METH) | 0);  /* check that OBJ_NAME_TYPE_MD_METH is an integer */
  return n;
}

static int _cffi_const_OPENSSL_BUILT_ON(unsigned long long *o)
{
  int n = (OPENSSL_BUILT_ON) <= 0;
  *o = (unsigned long long)((OPENSSL_BUILT_ON) | 0);  /* check that OPENSSL_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_OPENSSL_CFLAGS(unsigned long long *o)
{
  int n = (OPENSSL_CFLAGS) <= 0;
  *o = (unsigned long long)((OPENSSL_CFLAGS) | 0);  /* check that OPENSSL_CFLAGS is an integer */
  return n;
}

static int _cffi_const_OPENSSL_DIR(unsigned long long *o)
{
  int n = (OPENSSL_DIR) <= 0;
  *o = (unsigned long long)((OPENSSL_DIR) | 0);  /* check that OPENSSL_DIR is an integer */
  return n;
}

static int _cffi_const_OPENSSL_EC_NAMED_CURVE(unsigned long long *o)
{
  int n = (OPENSSL_EC_NAMED_CURVE) <= 0;
  *o = (unsigned long long)((OPENSSL_EC_NAMED_CURVE) | 0);  /* check that OPENSSL_EC_NAMED_CURVE is an integer */
  return n;
}

static int _cffi_const_OPENSSL_PLATFORM(unsigned long long *o)
{
  int n = (OPENSSL_PLATFORM) <= 0;
  *o = (unsigned long long)((OPENSSL_PLATFORM) | 0);  /* check that OPENSSL_PLATFORM is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION(unsigned long long *o)
{
  int n = (OPENSSL_VERSION) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION) | 0);  /* check that OPENSSL_VERSION is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION_NUMBER(unsigned long long *o)
{
  int n = (OPENSSL_VERSION_NUMBER) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION_NUMBER) | 0);  /* check that OPENSSL_VERSION_NUMBER is an integer */
  return n;
}

static void _cffi_const_OPENSSL_VERSION_TEXT(char *o)
{
  *(char const * *)o = OPENSSL_VERSION_TEXT;
}

static int _cffi_const_PEM_R_UNSUPPORTED_ENCRYPTION(unsigned long long *o)
{
  int n = (PEM_R_UNSUPPORTED_ENCRYPTION) <= 0;
  *o = (unsigned long long)((PEM_R_UNSUPPORTED_ENCRYPTION) | 0);  /* check that PEM_R_UNSUPPORTED_ENCRYPTION is an integer */
  return n;
}

static int _cffi_const_PKCS12_R_PKCS12_CIPHERFINAL_ERROR(unsigned long long *o)
{
  int n = (PKCS12_R_PKCS12_CIPHERFINAL_ERROR) <= 0;
  *o = (unsigned long long)((PKCS12_R_PKCS12_CIPHERFINAL_ERROR) | 0);  /* check that PKCS12_R_PKCS12_CIPHERFINAL_ERROR is an integer */
  return n;
}

static int _cffi_const_PKCS7_BINARY(unsigned long long *o)
{
  int n = (PKCS7_BINARY) <= 0;
  *o = (unsigned long long)((PKCS7_BINARY) | 0);  /* check that PKCS7_BINARY is an integer */
  return n;
}

static int _cffi_const_PKCS7_DETACHED(unsigned long long *o)
{
  int n = (PKCS7_DETACHED) <= 0;
  *o = (unsigned long long)((PKCS7_DETACHED) | 0);  /* check that PKCS7_DETACHED is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOATTR(unsigned long long *o)
{
  int n = (PKCS7_NOATTR) <= 0;
  *o = (unsigned long long)((PKCS7_NOATTR) | 0);  /* check that PKCS7_NOATTR is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCERTS(unsigned long long *o)
{
  int n = (PKCS7_NOCERTS) <= 0;
  *o = (unsigned long long)((PKCS7_NOCERTS) | 0);  /* check that PKCS7_NOCERTS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCHAIN(unsigned long long *o)
{
  int n = (PKCS7_NOCHAIN) <= 0;
  *o = (unsigned long long)((PKCS7_NOCHAIN) | 0);  /* check that PKCS7_NOCHAIN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOINTERN(unsigned long long *o)
{
  int n = (PKCS7_NOINTERN) <= 0;
  *o = (unsigned long long)((PKCS7_NOINTERN) | 0);  /* check that PKCS7_NOINTERN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSIGS(unsigned long long *o)
{
  int n = (PKCS7_NOSIGS) <= 0;
  *o = (unsigned long long)((PKCS7_NOSIGS) | 0);  /* check that PKCS7_NOSIGS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSMIMECAP(unsigned long long *o)
{
  int n = (PKCS7_NOSMIMECAP) <= 0;
  *o = (unsigned long long)((PKCS7_NOSMIMECAP) | 0);  /* check that PKCS7_NOSMIMECAP is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOVERIFY(unsigned long long *o)
{
  int n = (PKCS7_NOVERIFY) <= 0;
  *o = (unsigned long long)((PKCS7_NOVERIFY) | 0);  /* check that PKCS7_NOVERIFY is an integer */
  return n;
}

static int _cffi_const_PKCS7_PARTIAL(unsigned long long *o)
{
  int n = (PKCS7_PARTIAL) <= 0;
  *o = (unsigned long long)((PKCS7_PARTIAL) | 0);  /* check that PKCS7_PARTIAL is an integer */
  return n;
}

static int _cffi_const_PKCS7_STREAM(unsigned long long *o)
{
  int n = (PKCS7_STREAM) <= 0;
  *o = (unsigned long long)((PKCS7_STREAM) | 0);  /* check that PKCS7_STREAM is an integer */
  return n;
}

static int _cffi_const_PKCS7_TEXT(unsigned long long *o)
{
  int n = (PKCS7_TEXT) <= 0;
  *o = (unsigned long long)((PKCS7_TEXT) | 0);  /* check that PKCS7_TEXT is an integer */
  return n;
}

static int _cffi_const_PROV_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (PROV_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((PROV_R_BAD_DECRYPT) | 0);  /* check that PROV_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_PROV_R_WRONG_FINAL_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (PROV_R_WRONG_FINAL_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((PROV_R_WRONG_FINAL_BLOCK_LENGTH) | 0);  /* check that PROV_R_WRONG_FINAL_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_PROV_R_XTS_DUPLICATED_KEYS(unsigned long long *o)
{
  int n = (PROV_R_XTS_DUPLICATED_KEYS) <= 0;
  *o = (unsigned long long)((PROV_R_XTS_DUPLICATED_KEYS) | 0);  /* check that PROV_R_XTS_DUPLICATED_KEYS is an integer */
  return n;
}

static int _cffi_const_RSA_F4(unsigned long long *o)
{
  int n = (RSA_F4) <= 0;
  *o = (unsigned long long)((RSA_F4) | 0);  /* check that RSA_F4 is an integer */
  return n;
}

static int _cffi_const_RSA_NO_PADDING(unsigned long long *o)
{
  int n = (RSA_NO_PADDING) <= 0;
  *o = (unsigned long long)((RSA_NO_PADDING) | 0);  /* check that RSA_NO_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_OAEP_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_OAEP_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_OAEP_PADDING) | 0);  /* check that RSA_PKCS1_OAEP_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PADDING) | 0);  /* check that RSA_PKCS1_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PSS_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PSS_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PSS_PADDING) | 0);  /* check that RSA_PKCS1_PSS_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PSS_SALTLEN_AUTO(unsigned long long *o)
{
  int n = (RSA_PSS_SALTLEN_AUTO) <= 0;
  *o = (unsigned long long)((RSA_PSS_SALTLEN_AUTO) | 0);  /* check that RSA_PSS_SALTLEN_AUTO is an integer */
  return n;
}

static int _cffi_const_SSL3_RANDOM_SIZE(unsigned long long *o)
{
  int n = (SSL3_RANDOM_SIZE) <= 0;
  *o = (unsigned long long)((SSL3_RANDOM_SIZE) | 0);  /* check that SSL3_RANDOM_SIZE is an integer */
  return n;
}

static int _cffi_const_SSL3_VERSION(unsigned long long *o)
{
  int n = (SSL3_VERSION) <= 0;
  *o = (unsigned long long)((SSL3_VERSION) | 0);  /* check that SSL3_VERSION is an integer */
  return n;
}

static int _cffi_const_SSLEAY_BUILT_ON(unsigned long long *o)
{
  int n = (SSLEAY_BUILT_ON) <= 0;
  *o = (unsigned long long)((SSLEAY_BUILT_ON) | 0);  /* check that SSLEAY_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_SSLEAY_CFLAGS(unsigned long long *o)
{
  int n = (SSLEAY_CFLAGS) <= 0;
  *o = (unsigned long long)((SSLEAY_CFLAGS) | 0);  /* check that SSLEAY_CFLAGS is an integer */
  return n;
}

static int _cffi_const_SSLEAY_DIR(unsigned long long *o)
{
  int n = (SSLEAY_DIR) <= 0;
  *o = (unsigned long long)((SSLEAY_DIR) | 0);  /* check that SSLEAY_DIR is an integer */
  return n;
}

static int _cffi_const_SSLEAY_PLATFORM(unsigned long long *o)
{
  int n = (SSLEAY_PLATFORM) <= 0;
  *o = (unsigned long long)((SSLEAY_PLATFORM) | 0);  /* check that SSLEAY_PLATFORM is an integer */
  return n;
}

static int _cffi_const_SSLEAY_VERSION(unsigned long long *o)
{
  int n = (SSLEAY_VERSION) <= 0;
  *o = (unsigned long long)((SSLEAY_VERSION) | 0);  /* check that SSLEAY_VERSION is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_EXIT) | 0);  /* check that SSL_CB_ACCEPT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_LOOP) | 0);  /* check that SSL_CB_ACCEPT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_ALERT) | 0);  /* check that SSL_CB_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_EXIT) | 0);  /* check that SSL_CB_CONNECT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_LOOP) | 0);  /* check that SSL_CB_CONNECT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_EXIT) | 0);  /* check that SSL_CB_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_DONE(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_DONE) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_DONE) | 0);  /* check that SSL_CB_HANDSHAKE_DONE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_START(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_START) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_START) | 0);  /* check that SSL_CB_HANDSHAKE_START is an integer */
  return n;
}

static int _cffi_const_SSL_CB_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_LOOP) | 0);  /* check that SSL_CB_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ(unsigned long long *o)
{
  int n = (SSL_CB_READ) <= 0;
  *o = (unsigned long long)((SSL_CB_READ) | 0);  /* check that SSL_CB_READ is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_READ_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_READ_ALERT) | 0);  /* check that SSL_CB_READ_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE(unsigned long long *o)
{
  int n = (SSL_CB_WRITE) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE) | 0);  /* check that SSL_CB_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_WRITE_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE_ALERT) | 0);  /* check that SSL_CB_WRITE_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_NONE(unsigned long long *o)
{
  int n = (SSL_ERROR_NONE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_NONE) | 0);  /* check that SSL_ERROR_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SSL(unsigned long long *o)
{
  int n = (SSL_ERROR_SSL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SSL) | 0);  /* check that SSL_ERROR_SSL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SYSCALL(unsigned long long *o)
{
  int n = (SSL_ERROR_SYSCALL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SYSCALL) | 0);  /* check that SSL_ERROR_SYSCALL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_CONNECT(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_CONNECT) | 0);  /* check that SSL_ERROR_WANT_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_READ(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_READ) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_READ) | 0);  /* check that SSL_ERROR_WANT_READ is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_WRITE(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_WRITE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_WRITE) | 0);  /* check that SSL_ERROR_WANT_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_X509_LOOKUP(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_X509_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_X509_LOOKUP) | 0);  /* check that SSL_ERROR_WANT_X509_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_ZERO_RETURN(unsigned long long *o)
{
  int n = (SSL_ERROR_ZERO_RETURN) <= 0;
  *o = (unsigned long long)((SSL_ERROR_ZERO_RETURN) | 0);  /* check that SSL_ERROR_ZERO_RETURN is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_ASN1(unsigned long long *o)
{
  int n = (SSL_FILETYPE_ASN1) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_ASN1) | 0);  /* check that SSL_FILETYPE_ASN1 is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_PEM(unsigned long long *o)
{
  int n = (SSL_FILETYPE_PEM) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_PEM) | 0);  /* check that SSL_FILETYPE_PEM is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER(unsigned long long *o)
{
  int n = (SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) | 0);  /* check that SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_AUTO_RETRY(unsigned long long *o)
{
  int n = (SSL_MODE_AUTO_RETRY) <= 0;
  *o = (unsigned long long)((SSL_MODE_AUTO_RETRY) | 0);  /* check that SSL_MODE_AUTO_RETRY is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ENABLE_PARTIAL_WRITE(unsigned long long *o)
{
  int n = (SSL_MODE_ENABLE_PARTIAL_WRITE) <= 0;
  *o = (unsigned long long)((SSL_MODE_ENABLE_PARTIAL_WRITE) | 0);  /* check that SSL_MODE_ENABLE_PARTIAL_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (SSL_MODE_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((SSL_MODE_RELEASE_BUFFERS) | 0);  /* check that SSL_MODE_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALL(unsigned long long *o)
{
  int n = (SSL_OP_ALL) <= 0;
  *o = (unsigned long long)((SSL_OP_ALL) | 0);  /* check that SSL_OP_ALL is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION(unsigned long long *o)
{
  int n = (SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) | 0);  /* check that SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_CIPHER_SERVER_PREFERENCE(unsigned long long *o)
{
  int n = (SSL_OP_CIPHER_SERVER_PREFERENCE) <= 0;
  *o = (unsigned long long)((SSL_OP_CIPHER_SERVER_PREFERENCE) | 0);  /* check that SSL_OP_CIPHER_SERVER_PREFERENCE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_COOKIE_EXCHANGE(unsigned long long *o)
{
  int n = (SSL_OP_COOKIE_EXCHANGE) <= 0;
  *o = (unsigned long long)((SSL_OP_COOKIE_EXCHANGE) | 0);  /* check that SSL_OP_COOKIE_EXCHANGE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS(unsigned long long *o)
{
  int n = (SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) <= 0;
  *o = (unsigned long long)((SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) | 0);  /* check that SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_EPHEMERAL_RSA(unsigned long long *o)
{
  int n = (SSL_OP_EPHEMERAL_RSA) <= 0;
  *o = (unsigned long long)((SSL_OP_EPHEMERAL_RSA) | 0);  /* check that SSL_OP_EPHEMERAL_RSA is an integer */
  return n;
}

static int _cffi_const_SSL_OP_LEGACY_SERVER_CONNECT(unsigned long long *o)
{
  int n = (SSL_OP_LEGACY_SERVER_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_OP_LEGACY_SERVER_CONNECT) | 0);  /* check that SSL_OP_LEGACY_SERVER_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) | 0);  /* check that SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_SESS_ID_BUG(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_SESS_ID_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_SESS_ID_BUG) | 0);  /* check that SSL_OP_MICROSOFT_SESS_ID_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CA_DN_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CA_DN_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CA_DN_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CA_DN_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CHALLENGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CHALLENGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CHALLENGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CHALLENGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (SSL_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_COMPRESSION) | 0);  /* check that SSL_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1) | 0);  /* check that SSL_OP_NO_DTLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1_2) | 0);  /* check that SSL_OP_NO_DTLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_QUERY_MTU(unsigned long long *o)
{
  int n = (SSL_OP_NO_QUERY_MTU) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_QUERY_MTU) | 0);  /* check that SSL_OP_NO_QUERY_MTU is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_RENEGOTIATION(unsigned long long *o)
{
  int n = (SSL_OP_NO_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_RENEGOTIATION) | 0);  /* check that SSL_OP_NO_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv2(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv2) | 0);  /* check that SSL_OP_NO_SSLv2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv3(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv3) | 0);  /* check that SSL_OP_NO_SSLv3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TICKET) | 0);  /* check that SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1) | 0);  /* check that SSL_OP_NO_TLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_1) | 0);  /* check that SSL_OP_NO_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_2) | 0);  /* check that SSL_OP_NO_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_3(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_3) | 0);  /* check that SSL_OP_NO_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_1(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_1) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_1) | 0);  /* check that SSL_OP_PKCS1_CHECK_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_2(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_2) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_2) | 0);  /* check that SSL_OP_PKCS1_CHECK_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_DH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_DH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_DH_USE) | 0);  /* check that SSL_OP_SINGLE_DH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_ECDH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_ECDH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_ECDH_USE) | 0);  /* check that SSL_OP_SINGLE_ECDH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLEAY_080_CLIENT_DH_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLEAY_080_CLIENT_DH_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLEAY_080_CLIENT_DH_BUG) | 0);  /* check that SSL_OP_SSLEAY_080_CLIENT_DH_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) | 0);  /* check that SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_BLOCK_PADDING_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_BLOCK_PADDING_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_BLOCK_PADDING_BUG) | 0);  /* check that SSL_OP_TLS_BLOCK_PADDING_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_D5_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_D5_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_D5_BUG) | 0);  /* check that SSL_OP_TLS_D5_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_ROLLBACK_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_ROLLBACK_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_ROLLBACK_BUG) | 0);  /* check that SSL_OP_TLS_ROLLBACK_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_RECEIVED_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_RECEIVED_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_RECEIVED_SHUTDOWN) | 0);  /* check that SSL_RECEIVED_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_R_UNEXPECTED_EOF_WHILE_READING(unsigned long long *o)
{
  int n = (SSL_R_UNEXPECTED_EOF_WHILE_READING) <= 0;
  *o = (unsigned long long)((SSL_R_UNEXPECTED_EOF_WHILE_READING) | 0);  /* check that SSL_R_UNEXPECTED_EOF_WHILE_READING is an integer */
  return n;
}

static int _cffi_const_SSL_SENT_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_SENT_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_SENT_SHUTDOWN) | 0);  /* check that SSL_SENT_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_BOTH(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_BOTH) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_BOTH) | 0);  /* check that SSL_SESS_CACHE_BOTH is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_CLIENT(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_CLIENT) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_CLIENT) | 0);  /* check that SSL_SESS_CACHE_CLIENT is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_AUTO_CLEAR(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_AUTO_CLEAR) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_AUTO_CLEAR) | 0);  /* check that SSL_SESS_CACHE_NO_AUTO_CLEAR is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_LOOKUP(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_STORE(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_STORE) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_STORE) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_STORE is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_OFF(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_OFF) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_OFF) | 0);  /* check that SSL_SESS_CACHE_OFF is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_SERVER(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_SERVER) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_SERVER) | 0);  /* check that SSL_SESS_CACHE_SERVER is an integer */
  return n;
}

static int _cffi_const_SSL_ST_ACCEPT(unsigned long long *o)
{
  int n = (SSL_ST_ACCEPT) <= 0;
  *o = (unsigned long long)((SSL_ST_ACCEPT) | 0);  /* check that SSL_ST_ACCEPT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_BEFORE(unsigned long long *o)
{
  int n = (SSL_ST_BEFORE) <= 0;
  *o = (unsigned long long)((SSL_ST_BEFORE) | 0);  /* check that SSL_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_SSL_ST_CONNECT(unsigned long long *o)
{
  int n = (SSL_ST_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ST_CONNECT) | 0);  /* check that SSL_ST_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_INIT(unsigned long long *o)
{
  int n = (SSL_ST_INIT) <= 0;
  *o = (unsigned long long)((SSL_ST_INIT) | 0);  /* check that SSL_ST_INIT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_MASK(unsigned long long *o)
{
  int n = (SSL_ST_MASK) <= 0;
  *o = (unsigned long long)((SSL_ST_MASK) | 0);  /* check that SSL_ST_MASK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_OK(unsigned long long *o)
{
  int n = (SSL_ST_OK) <= 0;
  *o = (unsigned long long)((SSL_ST_OK) | 0);  /* check that SSL_ST_OK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_RENEGOTIATE(unsigned long long *o)
{
  int n = (SSL_ST_RENEGOTIATE) <= 0;
  *o = (unsigned long long)((SSL_ST_RENEGOTIATE) | 0);  /* check that SSL_ST_RENEGOTIATE is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_ALERT_FATAL(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_ALERT_FATAL) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_ALERT_FATAL) | 0);  /* check that SSL_TLSEXT_ERR_ALERT_FATAL is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_NOACK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_NOACK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_NOACK) | 0);  /* check that SSL_TLSEXT_ERR_NOACK is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_OK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_OK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_OK) | 0);  /* check that SSL_TLSEXT_ERR_OK is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_CLIENT_ONCE(unsigned long long *o)
{
  int n = (SSL_VERIFY_CLIENT_ONCE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_CLIENT_ONCE) | 0);  /* check that SSL_VERIFY_CLIENT_ONCE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_FAIL_IF_NO_PEER_CERT(unsigned long long *o)
{
  int n = (SSL_VERIFY_FAIL_IF_NO_PEER_CERT) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_FAIL_IF_NO_PEER_CERT) | 0);  /* check that SSL_VERIFY_FAIL_IF_NO_PEER_CERT is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_NONE(unsigned long long *o)
{
  int n = (SSL_VERIFY_NONE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_NONE) | 0);  /* check that SSL_VERIFY_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_PEER(unsigned long long *o)
{
  int n = (SSL_VERIFY_PEER) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_PEER) | 0);  /* check that SSL_VERIFY_PEER is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_POST_HANDSHAKE(unsigned long long *o)
{
  int n = (SSL_VERIFY_POST_HANDSHAKE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_POST_HANDSHAKE) | 0);  /* check that SSL_VERIFY_POST_HANDSHAKE is an integer */
  return n;
}

static int _cffi_const_TLS1_1_VERSION(unsigned long long *o)
{
  int n = (TLS1_1_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_1_VERSION) | 0);  /* check that TLS1_1_VERSION is an integer */
  return n;
}

static int _cffi_const_TLS1_2_VERSION(unsigned long long *o)
{
  int n = (TLS1_2_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_2_VERSION) | 0);  /* check that TLS1_2_VERSION is an integer */
  return n;
}

static int _cffi_const_TLS1_3_VERSION(unsigned long long *o)
{
  int n = (TLS1_3_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_3_VERSION) | 0);  /* check that TLS1_3_VERSION is an integer */
  return n;
}

static int _cffi_const_TLS1_VERSION(unsigned long long *o)
{
  int n = (TLS1_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_VERSION) | 0);  /* check that TLS1_VERSION is an integer */
  return n;
}

static int _cffi_const_TLSEXT_NAMETYPE_host_name(unsigned long long *o)
{
  int n = (TLSEXT_NAMETYPE_host_name) <= 0;
  *o = (unsigned long long)((TLSEXT_NAMETYPE_host_name) | 0);  /* check that TLSEXT_NAMETYPE_host_name is an integer */
  return n;
}

static int _cffi_const_TLSEXT_STATUSTYPE_ocsp(unsigned long long *o)
{
  int n = (TLSEXT_STATUSTYPE_ocsp) <= 0;
  *o = (unsigned long long)((TLSEXT_STATUSTYPE_ocsp) | 0);  /* check that TLSEXT_STATUSTYPE_ocsp is an integer */
  return n;
}

static int _cffi_const_TLS_ST_BEFORE(unsigned long long *o)
{
  int n = (TLS_ST_BEFORE) <= 0;
  *o = (unsigned long long)((TLS_ST_BEFORE) | 0);  /* check that TLS_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_TLS_ST_OK(unsigned long long *o)
{
  int n = (TLS_ST_OK) <= 0;
  *o = (unsigned long long)((TLS_ST_OK) | 0);  /* check that TLS_ST_OK is an integer */
  return n;
}

static int _cffi_const_V_ASN1_GENERALIZEDTIME(unsigned long long *o)
{
  int n = (V_ASN1_GENERALIZEDTIME) <= 0;
  *o = (unsigned long long)((V_ASN1_GENERALIZEDTIME) | 0);  /* check that V_ASN1_GENERALIZEDTIME is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_NEVER_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) | 0);  /* check that X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS is an integer */
  return n;
}

static int _cffi_const_X509_LU_CRL(unsigned long long *o)
{
  int n = (X509_LU_CRL) <= 0;
  *o = (unsigned long long)((X509_LU_CRL) | 0);  /* check that X509_LU_CRL is an integer */
  return n;
}

static int _cffi_const_X509_LU_X509(unsigned long long *o)
{
  int n = (X509_LU_X509) <= 0;
  *o = (unsigned long long)((X509_LU_X509) | 0);  /* check that X509_LU_X509 is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_ANY(unsigned long long *o)
{
  int n = (X509_PURPOSE_ANY) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_ANY) | 0);  /* check that X509_PURPOSE_ANY is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_CRL_SIGN(unsigned long long *o)
{
  int n = (X509_PURPOSE_CRL_SIGN) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_CRL_SIGN) | 0);  /* check that X509_PURPOSE_CRL_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_MAX(unsigned long long *o)
{
  int n = (X509_PURPOSE_MAX) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_MAX) | 0);  /* check that X509_PURPOSE_MAX is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_MIN(unsigned long long *o)
{
  int n = (X509_PURPOSE_MIN) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_MIN) | 0);  /* check that X509_PURPOSE_MIN is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_NS_SSL_SERVER(unsigned long long *o)
{
  int n = (X509_PURPOSE_NS_SSL_SERVER) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_NS_SSL_SERVER) | 0);  /* check that X509_PURPOSE_NS_SSL_SERVER is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_OCSP_HELPER(unsigned long long *o)
{
  int n = (X509_PURPOSE_OCSP_HELPER) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_OCSP_HELPER) | 0);  /* check that X509_PURPOSE_OCSP_HELPER is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_SMIME_ENCRYPT(unsigned long long *o)
{
  int n = (X509_PURPOSE_SMIME_ENCRYPT) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_SMIME_ENCRYPT) | 0);  /* check that X509_PURPOSE_SMIME_ENCRYPT is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_SMIME_SIGN(unsigned long long *o)
{
  int n = (X509_PURPOSE_SMIME_SIGN) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_SMIME_SIGN) | 0);  /* check that X509_PURPOSE_SMIME_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_SSL_CLIENT(unsigned long long *o)
{
  int n = (X509_PURPOSE_SSL_CLIENT) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_SSL_CLIENT) | 0);  /* check that X509_PURPOSE_SSL_CLIENT is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_SSL_SERVER(unsigned long long *o)
{
  int n = (X509_PURPOSE_SSL_SERVER) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_SSL_SERVER) | 0);  /* check that X509_PURPOSE_SSL_SERVER is an integer */
  return n;
}

static int _cffi_const_X509_PURPOSE_TIMESTAMP_SIGN(unsigned long long *o)
{
  int n = (X509_PURPOSE_TIMESTAMP_SIGN) <= 0;
  *o = (unsigned long long)((X509_PURPOSE_TIMESTAMP_SIGN) | 0);  /* check that X509_PURPOSE_TIMESTAMP_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_R_CERT_ALREADY_IN_HASH_TABLE(unsigned long long *o)
{
  int n = (X509_R_CERT_ALREADY_IN_HASH_TABLE) <= 0;
  *o = (unsigned long long)((X509_R_CERT_ALREADY_IN_HASH_TABLE) | 0);  /* check that X509_R_CERT_ALREADY_IN_HASH_TABLE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_SKID_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_SKID_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_SKID_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_SKID_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_APPLICATION_VERIFICATION(unsigned long long *o)
{
  int n = (X509_V_ERR_APPLICATION_VERIFICATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_APPLICATION_VERIFICATION) | 0);  /* check that X509_V_ERR_APPLICATION_VERIFICATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_CHAIN_TOO_LONG(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_CHAIN_TOO_LONG) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_CHAIN_TOO_LONG) | 0);  /* check that X509_V_ERR_CERT_CHAIN_TOO_LONG is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CERT_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CERT_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REJECTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REJECTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REJECTED) | 0);  /* check that X509_V_ERR_CERT_REJECTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REVOKED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REVOKED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REVOKED) | 0);  /* check that X509_V_ERR_CERT_REVOKED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CERT_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_UNTRUSTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_UNTRUSTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_UNTRUSTED) | 0);  /* check that X509_V_ERR_CERT_UNTRUSTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CRL_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CRL_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_PATH_VALIDATION_ERROR(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_PATH_VALIDATION_ERROR) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_PATH_VALIDATION_ERROR) | 0);  /* check that X509_V_ERR_CRL_PATH_VALIDATION_ERROR is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CRL_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) | 0);  /* check that X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DIFFERENT_CRL_SCOPE(unsigned long long *o)
{
  int n = (X509_V_ERR_DIFFERENT_CRL_SCOPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DIFFERENT_CRL_SCOPE) | 0);  /* check that X509_V_ERR_DIFFERENT_CRL_SCOPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EMAIL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_EMAIL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EMAIL_MISMATCH) | 0);  /* check that X509_V_ERR_EMAIL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EXCLUDED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_EXCLUDED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EXCLUDED_VIOLATION) | 0);  /* check that X509_V_ERR_EXCLUDED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_HOSTNAME_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_HOSTNAME_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_HOSTNAME_MISMATCH) | 0);  /* check that X509_V_ERR_HOSTNAME_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_CA) | 0);  /* check that X509_V_ERR_INVALID_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_NON_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_NON_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_NON_CA) | 0);  /* check that X509_V_ERR_INVALID_NON_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_POLICY_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_POLICY_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_POLICY_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_POLICY_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_PURPOSE(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_PURPOSE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_PURPOSE) | 0);  /* check that X509_V_ERR_INVALID_PURPOSE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_IP_ADDRESS_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_IP_ADDRESS_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_IP_ADDRESS_MISMATCH) | 0);  /* check that X509_V_ERR_IP_ADDRESS_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CERTSIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CERTSIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CERTSIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CERTSIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CRL_SIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CRL_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_NO_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_ERR_NO_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_NO_EXPLICIT_POLICY) | 0);  /* check that X509_V_ERR_NO_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_OUT_OF_MEM(unsigned long long *o)
{
  int n = (X509_V_ERR_OUT_OF_MEM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_OUT_OF_MEM) | 0);  /* check that X509_V_ERR_OUT_OF_MEM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PERMITTED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_PERMITTED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PERMITTED_VIOLATION) | 0);  /* check that X509_V_ERR_PERMITTED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) | 0);  /* check that X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN(unsigned long long *o)
{
  int n = (X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) | 0);  /* check that X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBJECT_ISSUER_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBJECT_ISSUER_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBJECT_ISSUER_MISMATCH) | 0);  /* check that X509_V_ERR_SUBJECT_ISSUER_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBTREE_MINMAX(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBTREE_MINMAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBTREE_MINMAX) | 0);  /* check that X509_V_ERR_SUBTREE_MINMAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNNESTED_RESOURCE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNNESTED_RESOURCE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNNESTED_RESOURCE) | 0);  /* check that X509_V_ERR_UNNESTED_RESOURCE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_NAME_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_NAME_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_ALLOW_PROXY_CERTS(unsigned long long *o)
{
  int n = (X509_V_FLAG_ALLOW_PROXY_CERTS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_ALLOW_PROXY_CERTS) | 0);  /* check that X509_V_FLAG_ALLOW_PROXY_CERTS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CB_ISSUER_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CB_ISSUER_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CB_ISSUER_CHECK) | 0);  /* check that X509_V_FLAG_CB_ISSUER_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CHECK_SS_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_FLAG_CHECK_SS_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CHECK_SS_SIGNATURE) | 0);  /* check that X509_V_FLAG_CHECK_SS_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK) | 0);  /* check that X509_V_FLAG_CRL_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK_ALL(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK_ALL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK_ALL) | 0);  /* check that X509_V_FLAG_CRL_CHECK_ALL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXPLICIT_POLICY) | 0);  /* check that X509_V_FLAG_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXTENDED_CRL_SUPPORT(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXTENDED_CRL_SUPPORT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXTENDED_CRL_SUPPORT) | 0);  /* check that X509_V_FLAG_EXTENDED_CRL_SUPPORT is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_IGNORE_CRITICAL(unsigned long long *o)
{
  int n = (X509_V_FLAG_IGNORE_CRITICAL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_IGNORE_CRITICAL) | 0);  /* check that X509_V_FLAG_IGNORE_CRITICAL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_ANY(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_ANY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_ANY) | 0);  /* check that X509_V_FLAG_INHIBIT_ANY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_MAP(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_MAP) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_MAP) | 0);  /* check that X509_V_FLAG_INHIBIT_MAP is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_NOTIFY_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_NOTIFY_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_NOTIFY_POLICY) | 0);  /* check that X509_V_FLAG_NOTIFY_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_NO_ALT_CHAINS(unsigned long long *o)
{
  int n = (X509_V_FLAG_NO_ALT_CHAINS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_NO_ALT_CHAINS) | 0);  /* check that X509_V_FLAG_NO_ALT_CHAINS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_NO_CHECK_TIME(unsigned long long *o)
{
  int n = (X509_V_FLAG_NO_CHECK_TIME) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_NO_CHECK_TIME) | 0);  /* check that X509_V_FLAG_NO_CHECK_TIME is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_PARTIAL_CHAIN(unsigned long long *o)
{
  int n = (X509_V_FLAG_PARTIAL_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_PARTIAL_CHAIN) | 0);  /* check that X509_V_FLAG_PARTIAL_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_POLICY_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_POLICY_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_POLICY_CHECK) | 0);  /* check that X509_V_FLAG_POLICY_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_TRUSTED_FIRST(unsigned long long *o)
{
  int n = (X509_V_FLAG_TRUSTED_FIRST) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_TRUSTED_FIRST) | 0);  /* check that X509_V_FLAG_TRUSTED_FIRST is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_CHECK_TIME(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_CHECK_TIME) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_CHECK_TIME) | 0);  /* check that X509_V_FLAG_USE_CHECK_TIME is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_DELTAS(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_DELTAS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_DELTAS) | 0);  /* check that X509_V_FLAG_USE_DELTAS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_X509_STRICT(unsigned long long *o)
{
  int n = (X509_V_FLAG_X509_STRICT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_X509_STRICT) | 0);  /* check that X509_V_FLAG_X509_STRICT is an integer */
  return n;
}

static int _cffi_const_X509_V_OK(unsigned long long *o)
{
  int n = (X509_V_OK) <= 0;
  *o = (unsigned long long)((X509_V_OK) | 0);  /* check that X509_V_OK is an integer */
  return n;
}

static void _cffi_d_ASN1_ENUMERATED_free(ASN1_ENUMERATED * x0)
{
  ASN1_ENUMERATED_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_free(PyObject *self, PyObject *arg0)
{
  ASN1_ENUMERATED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(550), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(550), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_ENUMERATED_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_free _cffi_d_ASN1_ENUMERATED_free
#endif

static ASN1_ENUMERATED * _cffi_d_ASN1_ENUMERATED_new(void)
{
  return ASN1_ENUMERATED_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_new(PyObject *self, PyObject *noarg)
{
  ASN1_ENUMERATED * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(550));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_new _cffi_d_ASN1_ENUMERATED_new
#endif

static int _cffi_d_ASN1_ENUMERATED_set(ASN1_ENUMERATED * x0, long x1)
{
  return ASN1_ENUMERATED_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_set(PyObject *self, PyObject *args)
{
  ASN1_ENUMERATED * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_ENUMERATED_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(550), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(550), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_set _cffi_d_ASN1_ENUMERATED_set
#endif

static void _cffi_d_ASN1_GENERALIZEDTIME_free(ASN1_GENERALIZEDTIME * x0)
{
  ASN1_GENERALIZEDTIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_GENERALIZEDTIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_GENERALIZEDTIME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2067), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2067), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_GENERALIZEDTIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_GENERALIZEDTIME_free _cffi_d_ASN1_GENERALIZEDTIME_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_IA5STRING_new(void)
{
  return ASN1_IA5STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_IA5STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_IA5STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_IA5STRING_new _cffi_d_ASN1_IA5STRING_new
#endif

static void _cffi_d_ASN1_INTEGER_free(ASN1_INTEGER * x0)
{
  ASN1_INTEGER_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_free(PyObject *self, PyObject *arg0)
{
  ASN1_INTEGER * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(8), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(8), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_INTEGER_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_INTEGER_free _cffi_d_ASN1_INTEGER_free
#endif

static int _cffi_d_ASN1_INTEGER_set(ASN1_INTEGER * x0, long x1)
{
  return ASN1_INTEGER_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_set(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(8), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(8), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_set _cffi_d_ASN1_INTEGER_set
#endif

static BIGNUM * _cffi_d_ASN1_INTEGER_to_BN(ASN1_INTEGER * x0, BIGNUM * x1)
{
  return ASN1_INTEGER_to_BN(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_to_BN(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  BIGNUM * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_to_BN", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(8), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(8), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_to_BN(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_to_BN _cffi_d_ASN1_INTEGER_to_BN
#endif

static void _cffi_d_ASN1_OBJECT_free(ASN1_OBJECT * x0)
{
  ASN1_OBJECT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OBJECT_free(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(419), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(419), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OBJECT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OBJECT_free _cffi_d_ASN1_OBJECT_free
#endif

static void _cffi_d_ASN1_OCTET_STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_OCTET_STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OCTET_STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_free _cffi_d_ASN1_OCTET_STRING_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_OCTET_STRING_new(void)
{
  return ASN1_OCTET_STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_new _cffi_d_ASN1_OCTET_STRING_new
#endif

static int _cffi_d_ASN1_OCTET_STRING_set(ASN1_OCTET_STRING * x0, unsigned char const * x1, int x2)
{
  return ASN1_OCTET_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  unsigned char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_OCTET_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_set _cffi_d_ASN1_OCTET_STRING_set
#endif

static unsigned char * _cffi_d_ASN1_STRING_data(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_data(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(638));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_data _cffi_d_ASN1_STRING_data
#endif

static unsigned char const * _cffi_d_ASN1_STRING_get0_data(ASN1_OCTET_STRING const * x0)
{
  return ASN1_STRING_get0_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_get0_data(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1532), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1532), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_get0_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(67));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_get0_data _cffi_d_ASN1_STRING_get0_data
#endif

static int _cffi_d_ASN1_STRING_length(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_length(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_length(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_length(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_length _cffi_d_ASN1_STRING_length
#endif

static int _cffi_d_ASN1_STRING_set(ASN1_OCTET_STRING * x0, void const * x1, int x2)
{
  return ASN1_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set _cffi_d_ASN1_STRING_set
#endif

static int _cffi_d_ASN1_STRING_set_default_mask_asc(char * x0)
{
  return ASN1_STRING_set_default_mask_asc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set_default_mask_asc(PyObject *self, PyObject *arg0)
{
  char * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set_default_mask_asc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set_default_mask_asc _cffi_d_ASN1_STRING_set_default_mask_asc
#endif

static int _cffi_d_ASN1_STRING_to_UTF8(unsigned char * * x0, ASN1_OCTET_STRING * x1)
{
  return ASN1_STRING_to_UTF8(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_to_UTF8(PyObject *self, PyObject *args)
{
  unsigned char * * x0;
  ASN1_OCTET_STRING * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_to_UTF8", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1604), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1604), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_to_UTF8(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_to_UTF8 _cffi_d_ASN1_STRING_to_UTF8
#endif

static int _cffi_d_ASN1_STRING_type(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_type(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_type _cffi_d_ASN1_STRING_type
#endif

static void _cffi_d_ASN1_TIME_free(ASN1_OCTET_STRING * x0)
{
  ASN1_TIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_TIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_TIME_free _cffi_d_ASN1_TIME_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_TIME_new(void)
{
  return ASN1_TIME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_new _cffi_d_ASN1_TIME_new
#endif

static int _cffi_d_ASN1_TIME_set_string(ASN1_OCTET_STRING * x0, char const * x1)
{
  return ASN1_TIME_set_string(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_set_string(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_set_string", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_set_string(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_set_string _cffi_d_ASN1_TIME_set_string
#endif

static ASN1_GENERALIZEDTIME * _cffi_d_ASN1_TIME_to_generalizedtime(ASN1_OCTET_STRING * x0, ASN1_GENERALIZEDTIME * * x1)
{
  return ASN1_TIME_to_generalizedtime(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_to_generalizedtime(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  ASN1_GENERALIZEDTIME * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_to_generalizedtime", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(3), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(3), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_to_generalizedtime(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2067));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_to_generalizedtime _cffi_d_ASN1_TIME_to_generalizedtime
#endif

static void _cffi_d_BIO_ADDR_free(BIO_ADDR * x0)
{
  BIO_ADDR_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_ADDR_free(PyObject *self, PyObject *arg0)
{
  BIO_ADDR * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1287), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO_ADDR *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1287), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_ADDR_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_ADDR_free _cffi_d_BIO_ADDR_free
#endif

static BIO_ADDR * _cffi_d_BIO_ADDR_new(void)
{
  return BIO_ADDR_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_ADDR_new(PyObject *self, PyObject *noarg)
{
  BIO_ADDR * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_ADDR_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1287));
  return pyresult;
}
#else
#  define _cffi_f_BIO_ADDR_new _cffi_d_BIO_ADDR_new
#endif

static void _cffi_d_BIO_clear_retry_flags(BIO * x0)
{
  BIO_clear_retry_flags(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_clear_retry_flags(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_clear_retry_flags(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_clear_retry_flags _cffi_d_BIO_clear_retry_flags
#endif

static size_t _cffi_d_BIO_ctrl_pending(BIO * x0)
{
  return BIO_ctrl_pending(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_ctrl_pending(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_ctrl_pending(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_ctrl_pending _cffi_d_BIO_ctrl_pending
#endif

static int _cffi_d_BIO_free(BIO * x0)
{
  return BIO_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_free _cffi_d_BIO_free
#endif

static void _cffi_d_BIO_free_all(BIO * x0)
{
  BIO_free_all(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free_all(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_free_all(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_free_all _cffi_d_BIO_free_all
#endif

static long _cffi_d_BIO_get_mem_data(BIO * x0, char * * x1)
{
  return BIO_get_mem_data(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_get_mem_data(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_get_mem_data", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1851), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1851), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_get_mem_data(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_get_mem_data _cffi_d_BIO_get_mem_data
#endif

static int _cffi_d_BIO_gets(BIO * x0, char * x1, int x2)
{
  return BIO_gets(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_gets(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_gets", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_gets(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_gets _cffi_d_BIO_gets
#endif

static BIO * _cffi_d_BIO_new(BIO_METHOD * x0)
{
  return BIO_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new(PyObject *self, PyObject *arg0)
{
  BIO_METHOD * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(79), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(79), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(129));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new _cffi_d_BIO_new
#endif

static BIO * _cffi_d_BIO_new_file(char const * x0, char const * x1)
{
  return BIO_new_file(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_file(PyObject *self, PyObject *args)
{
  char const * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_file", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_file(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(129));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_file _cffi_d_BIO_new_file
#endif

static BIO * _cffi_d_BIO_new_mem_buf(void const * x0, int x1)
{
  return BIO_new_mem_buf(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_mem_buf(PyObject *self, PyObject *args)
{
  void const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_mem_buf", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_mem_buf(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(129));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_mem_buf _cffi_d_BIO_new_mem_buf
#endif

static int _cffi_d_BIO_read(BIO * x0, void * x1, int x2)
{
  return BIO_read(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_read(PyObject *self, PyObject *args)
{
  BIO * x0;
  void * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_read", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_read(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_read _cffi_d_BIO_read
#endif

static int _cffi_d_BIO_reset(BIO * x0)
{
  return BIO_reset(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_reset(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_reset(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_reset _cffi_d_BIO_reset
#endif

static BIO_METHOD const * _cffi_d_BIO_s_mem(void)
{
  return BIO_s_mem();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_mem(PyObject *self, PyObject *noarg)
{
  BIO_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_mem(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2450));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_mem _cffi_d_BIO_s_mem
#endif

static long _cffi_d_BIO_set_mem_eof_return(BIO * x0, int x1)
{
  return BIO_set_mem_eof_return(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_mem_eof_return(PyObject *self, PyObject *args)
{
  BIO * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_set_mem_eof_return", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_set_mem_eof_return(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_set_mem_eof_return _cffi_d_BIO_set_mem_eof_return
#endif

static void _cffi_d_BIO_set_retry_read(BIO * x0)
{
  BIO_set_retry_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_retry_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_set_retry_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_set_retry_read _cffi_d_BIO_set_retry_read
#endif

static int _cffi_d_BIO_should_io_special(BIO * x0)
{
  return BIO_should_io_special(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_io_special(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_io_special(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_io_special _cffi_d_BIO_should_io_special
#endif

static int _cffi_d_BIO_should_read(BIO * x0)
{
  return BIO_should_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_read _cffi_d_BIO_should_read
#endif

static int _cffi_d_BIO_should_retry(BIO * x0)
{
  return BIO_should_retry(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_retry(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_retry(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_retry _cffi_d_BIO_should_retry
#endif

static int _cffi_d_BIO_should_write(BIO * x0)
{
  return BIO_should_write(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_write(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_write(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_write _cffi_d_BIO_should_write
#endif

static int _cffi_d_BIO_up_ref(BIO * x0)
{
  return BIO_up_ref(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_up_ref(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_up_ref(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_up_ref _cffi_d_BIO_up_ref
#endif

static int _cffi_d_BIO_write(BIO * x0, void const * x1, int x2)
{
  return BIO_write(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_write(PyObject *self, PyObject *args)
{
  BIO * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_write", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_write(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_write _cffi_d_BIO_write
#endif

static void _cffi_d_BN_CTX_end(BN_CTX * x0)
{
  BN_CTX_end(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_end(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_end(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_end _cffi_d_BN_CTX_end
#endif

static void _cffi_d_BN_CTX_free(BN_CTX * x0)
{
  BN_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_free _cffi_d_BN_CTX_free
#endif

static BIGNUM * _cffi_d_BN_CTX_get(BN_CTX * x0)
{
  return BN_CTX_get(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_get(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_get(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_get _cffi_d_BN_CTX_get
#endif

static BN_CTX * _cffi_d_BN_CTX_new(void)
{
  return BN_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(58));
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_new _cffi_d_BN_CTX_new
#endif

static void _cffi_d_BN_CTX_start(BN_CTX * x0)
{
  BN_CTX_start(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_start(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_start(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_start _cffi_d_BN_CTX_start
#endif

static void _cffi_d_BN_MONT_CTX_free(BN_MONT_CTX * x0)
{
  BN_MONT_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_MONT_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(603), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(603), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_MONT_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_MONT_CTX_free _cffi_d_BN_MONT_CTX_free
#endif

static BN_MONT_CTX * _cffi_d_BN_MONT_CTX_new(void)
{
  return BN_MONT_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_MONT_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(603));
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_new _cffi_d_BN_MONT_CTX_new
#endif

static int _cffi_d_BN_MONT_CTX_set(BN_MONT_CTX * x0, BIGNUM const * x1, BN_CTX * x2)
{
  return BN_MONT_CTX_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_set(PyObject *self, PyObject *args)
{
  BN_MONT_CTX * x0;
  BIGNUM const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_MONT_CTX_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(603), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(603), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_set _cffi_d_BN_MONT_CTX_set
#endif

static int _cffi_d_BN_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_add(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_add", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_add(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_add _cffi_d_BN_add
#endif

static BIGNUM * _cffi_d_BN_bin2bn(unsigned char const * x0, int x1, BIGNUM * x2)
{
  return BN_bin2bn(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bin2bn(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_bin2bn", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bin2bn(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bin2bn _cffi_d_BN_bin2bn
#endif

static int _cffi_d_BN_bn2bin(BIGNUM const * x0, unsigned char * x1)
{
  return BN_bn2bin(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2bin(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  unsigned char * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_bn2bin", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2bin(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2bin _cffi_d_BN_bn2bin
#endif

static char * _cffi_d_BN_bn2hex(BIGNUM const * x0)
{
  return BN_bn2hex(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2hex(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2hex(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(270));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2hex _cffi_d_BN_bn2hex
#endif

static void _cffi_d_BN_clear_free(BIGNUM * x0)
{
  BN_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_clear_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_clear_free _cffi_d_BN_clear_free
#endif

static int _cffi_d_BN_cmp(BIGNUM const * x0, BIGNUM const * x1)
{
  return BN_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_cmp(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_cmp _cffi_d_BN_cmp
#endif

static BIGNUM * _cffi_d_BN_dup(BIGNUM const * x0)
{
  return BN_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_dup(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_dup _cffi_d_BN_dup
#endif

static void _cffi_d_BN_free(BIGNUM * x0)
{
  BN_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_free _cffi_d_BN_free
#endif

static int _cffi_d_BN_generate_prime_ex(BIGNUM * x0, int x1, int x2, BIGNUM const * x3, BIGNUM const * x4, BN_GENCB * x5)
{
  return BN_generate_prime_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_generate_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  int x2;
  BIGNUM const * x3;
  BIGNUM const * x4;
  BN_GENCB * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_generate_prime_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(621), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(621), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_generate_prime_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_generate_prime_ex _cffi_d_BN_generate_prime_ex
#endif

static int _cffi_d_BN_hex2bn(BIGNUM * * x0, char const * x1)
{
  return BN_hex2bn(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_hex2bn(PyObject *self, PyObject *args)
{
  BIGNUM * * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_hex2bn", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(578), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(578), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_hex2bn(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_hex2bn _cffi_d_BN_hex2bn
#endif

static int _cffi_d_BN_is_negative(BIGNUM const * x0)
{
  return BN_is_negative(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_negative(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_negative(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_negative _cffi_d_BN_is_negative
#endif

static int _cffi_d_BN_is_odd(BIGNUM const * x0)
{
  return BN_is_odd(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_odd(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_odd(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_odd _cffi_d_BN_is_odd
#endif

static int _cffi_d_BN_is_prime_ex(BIGNUM const * x0, int x1, BN_CTX * x2, BN_GENCB * x3)
{
  return BN_is_prime_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  int x1;
  BN_CTX * x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_is_prime_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(621), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(621), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_prime_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_prime_ex _cffi_d_BN_is_prime_ex
#endif

static int _cffi_d_BN_mod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod _cffi_d_BN_mod
#endif

static int _cffi_d_BN_mod_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_add _cffi_d_BN_mod_add
#endif

static int _cffi_d_BN_mod_exp(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_exp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp _cffi_d_BN_mod_exp
#endif

static int _cffi_d_BN_mod_exp_mont(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(603), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(603), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont _cffi_d_BN_mod_exp_mont
#endif

static int _cffi_d_BN_mod_exp_mont_consttime(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont_consttime(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont_consttime", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(603), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(603), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont_consttime _cffi_d_BN_mod_exp_mont_consttime
#endif

static BIGNUM * _cffi_d_BN_mod_inverse(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod_inverse(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_inverse(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod_inverse", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_inverse(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_inverse _cffi_d_BN_mod_inverse
#endif

static int _cffi_d_BN_mod_mul(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_mul(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_mul(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_mul", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_mul(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_mul _cffi_d_BN_mod_mul
#endif

static int _cffi_d_BN_mod_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_sub(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_sub", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_sub(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_sub _cffi_d_BN_mod_sub
#endif

static BIGNUM * _cffi_d_BN_new(void)
{
  return BN_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_new(PyObject *self, PyObject *noarg)
{
  BIGNUM * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(7));
  return pyresult;
}
#else
#  define _cffi_f_BN_new _cffi_d_BN_new
#endif

static int _cffi_d_BN_nnmod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_nnmod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_nnmod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_nnmod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_nnmod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_nnmod _cffi_d_BN_nnmod
#endif

static int _cffi_d_BN_num_bits(BIGNUM const * x0)
{
  return BN_num_bits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bits(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bits _cffi_d_BN_num_bits
#endif

static int _cffi_d_BN_num_bytes(BIGNUM const * x0)
{
  return BN_num_bytes(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bytes(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bytes(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bytes _cffi_d_BN_num_bytes
#endif

static int _cffi_d_BN_prime_checks_for_size(int x0)
{
  return BN_prime_checks_for_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_prime_checks_for_size(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_prime_checks_for_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_BN_prime_checks_for_size _cffi_d_BN_prime_checks_for_size
#endif

static int _cffi_d_BN_rand_range(BIGNUM * x0, BIGNUM const * x1)
{
  return BN_rand_range(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_rand_range(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_rand_range", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_rand_range(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_rand_range _cffi_d_BN_rand_range
#endif

static void _cffi_d_BN_set_flags(BIGNUM * x0, int x1)
{
  BN_set_flags(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_flags(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_flags", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_set_flags(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_set_flags _cffi_d_BN_set_flags
#endif

static int _cffi_d_BN_set_word(BIGNUM * x0, BN_ULONG x1)
{
  return BN_set_word(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_word(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BN_ULONG x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_word", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, BN_ULONG);
  if (x1 == (BN_ULONG)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_set_word(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_set_word _cffi_d_BN_set_word
#endif

static int _cffi_d_BN_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_sub(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_sub", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_sub(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_sub _cffi_d_BN_sub
#endif

static ASN1_INTEGER * _cffi_d_BN_to_ASN1_INTEGER(BIGNUM * x0, ASN1_INTEGER * x1)
{
  return BN_to_ASN1_INTEGER(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_to_ASN1_INTEGER(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  ASN1_INTEGER * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_INTEGER * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_to_ASN1_INTEGER", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(8), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(8), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_to_ASN1_INTEGER(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(8));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_to_ASN1_INTEGER _cffi_d_BN_to_ASN1_INTEGER
#endif

static BIGNUM const * _cffi_d_BN_value_one(void)
{
  return BN_value_one();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_value_one(PyObject *self, PyObject *noarg)
{
  BIGNUM const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_value_one(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(56));
  return pyresult;
}
#else
#  define _cffi_f_BN_value_one _cffi_d_BN_value_one
#endif

static int _cffi_d_CMAC_CTX_copy(CMAC_CTX * x0, CMAC_CTX const * x1)
{
  return CMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_copy(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  CMAC_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "CMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(790), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(790), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(791), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (CMAC_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(791), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_copy _cffi_d_CMAC_CTX_copy
#endif

static void _cffi_d_CMAC_CTX_free(CMAC_CTX * x0)
{
  CMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  CMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(790), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(790), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CMAC_CTX_free _cffi_d_CMAC_CTX_free
#endif

static CMAC_CTX * _cffi_d_CMAC_CTX_new(void)
{
  return CMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  CMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(790));
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_new _cffi_d_CMAC_CTX_new
#endif

static int _cffi_d_CMAC_Final(CMAC_CTX * x0, unsigned char * x1, size_t * x2)
{
  return CMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Final(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(790), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(790), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Final _cffi_d_CMAC_Final
#endif

static int _cffi_d_CMAC_Init(CMAC_CTX * x0, void const * x1, size_t x2, EVP_CIPHER const * x3, ENGINE * x4)
{
  return CMAC_Init(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Init(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  EVP_CIPHER const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "CMAC_Init", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(790), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(790), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Init(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Init _cffi_d_CMAC_Init
#endif

static int _cffi_d_CMAC_Update(CMAC_CTX * x0, void const * x1, size_t x2)
{
  return CMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Update(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(790), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(790), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Update _cffi_d_CMAC_Update
#endif

static int _cffi_d_Cryptography_CRYPTO_set_mem_functions(void *(* x0)(size_t, char const *, int), void *(* x1)(void *, size_t, char const *, int), void(* x2)(void *, char const *, int))
{
  return Cryptography_CRYPTO_set_mem_functions(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_CRYPTO_set_mem_functions(PyObject *self, PyObject *args)
{
  void *(* x0)(size_t, char const *, int);
  void *(* x1)(void *, size_t, char const *, int);
  void(* x2)(void *, char const *, int);
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_CRYPTO_set_mem_functions", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = (void *(*)(size_t, char const *, int))_cffi_to_c_pointer(arg0, _cffi_type(1840));
  if (x0 == (void *(*)(size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x1 = (void *(*)(void *, size_t, char const *, int))_cffi_to_c_pointer(arg1, _cffi_type(1841));
  if (x1 == (void *(*)(void *, size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x2 = (void(*)(void *, char const *, int))_cffi_to_c_pointer(arg2, _cffi_type(1842));
  if (x2 == (void(*)(void *, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_CRYPTO_set_mem_functions(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_CRYPTO_set_mem_functions _cffi_d_Cryptography_CRYPTO_set_mem_functions
#endif

static int _cffi_d_Cryptography_DH_check(DH const * x0, int * x1)
{
  return Cryptography_DH_check(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DH_check(PyObject *self, PyObject *args)
{
  DH const * x0;
  int * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_DH_check", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(858), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(858), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(862), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(862), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DH_check(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DH_check _cffi_d_Cryptography_DH_check
#endif

static long _cffi_d_Cryptography_DTLSv1_get_timeout(SSL * x0, time_t * x1, long * x2)
{
  return Cryptography_DTLSv1_get_timeout(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DTLSv1_get_timeout(PyObject *self, PyObject *args)
{
  SSL * x0;
  time_t * x1;
  long * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_DTLSv1_get_timeout", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1874), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (time_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1874), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1875), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1875), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DTLSv1_get_timeout(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DTLSv1_get_timeout _cffi_d_Cryptography_DTLSv1_get_timeout
#endif

static void _cffi_d_Cryptography_EVP_MD_CTX_free(EVP_MD_CTX * x0)
{
  Cryptography_EVP_MD_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_EVP_MD_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_free _cffi_d_Cryptography_EVP_MD_CTX_free
#endif

static EVP_MD_CTX * _cffi_d_Cryptography_EVP_MD_CTX_new(void)
{
  return Cryptography_EVP_MD_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_MD_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_EVP_MD_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1037));
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_new _cffi_d_Cryptography_EVP_MD_CTX_new
#endif

static SSL_SESSION * _cffi_d_Cryptography_SSL_SESSION_new(void)
{
  return Cryptography_SSL_SESSION_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_SSL_SESSION_new(PyObject *self, PyObject *noarg)
{
  SSL_SESSION * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_SSL_SESSION_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1298));
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_SSL_SESSION_new _cffi_d_Cryptography_SSL_SESSION_new
#endif

static X509_REVOKED * _cffi_d_Cryptography_X509_REVOKED_dup(X509_REVOKED * x0)
{
  return Cryptography_X509_REVOKED_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_X509_REVOKED_dup(PyObject *self, PyObject *arg0)
{
  X509_REVOKED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_REVOKED * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(431), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509_REVOKED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(431), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_X509_REVOKED_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(431));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_X509_REVOKED_dup _cffi_d_Cryptography_X509_REVOKED_dup
#endif

static int _cffi_d_Cryptography_add_osrandom_engine(void)
{
  return Cryptography_add_osrandom_engine();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_add_osrandom_engine(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_add_osrandom_engine(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_add_osrandom_engine _cffi_d_Cryptography_add_osrandom_engine
#endif

static DH * _cffi_d_Cryptography_d2i_DHxparams_bio(BIO * x0, DH * * x1)
{
  return Cryptography_d2i_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_d2i_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_d2i_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(130), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(130), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_d2i_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(139));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_d2i_DHxparams_bio _cffi_d_Cryptography_d2i_DHxparams_bio
#endif

static void _cffi_d_Cryptography_free_wrapper(void * x0, char const * x1, int x2)
{
  Cryptography_free_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_free_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_free_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_free_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_free_wrapper _cffi_d_Cryptography_free_wrapper
#endif

static int _cffi_d_Cryptography_i2d_DHxparams_bio(BIO * x0, DH * x1)
{
  return Cryptography_i2d_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_i2d_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_i2d_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_i2d_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_i2d_DHxparams_bio _cffi_d_Cryptography_i2d_DHxparams_bio
#endif

static void * _cffi_d_Cryptography_malloc_wrapper(size_t x0, char const * x1, int x2)
{
  return Cryptography_malloc_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_malloc_wrapper(PyObject *self, PyObject *args)
{
  size_t x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_malloc_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_malloc_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(136));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_malloc_wrapper _cffi_d_Cryptography_malloc_wrapper
#endif

static int _cffi_d_Cryptography_pem_password_cb(char * x0, int x1, int x2, void * x3)
{
  return Cryptography_pem_password_cb(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_pem_password_cb(PyObject *self, PyObject *args)
{
  char * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_pem_password_cb", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_pem_password_cb(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_pem_password_cb _cffi_d_Cryptography_pem_password_cb
#endif

static void * _cffi_d_Cryptography_realloc_wrapper(void * x0, size_t x1, char const * x2, int x3)
{
  return Cryptography_realloc_wrapper(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_realloc_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  size_t x1;
  char const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_realloc_wrapper", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_realloc_wrapper(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(136));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_realloc_wrapper _cffi_d_Cryptography_realloc_wrapper
#endif

static void _cffi_d_DH_free(DH * x0)
{
  DH_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_free(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_free _cffi_d_DH_free
#endif

static int _cffi_d_DH_generate_key(DH * x0)
{
  return DH_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_key(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_key _cffi_d_DH_generate_key
#endif

static int _cffi_d_DH_generate_parameters_ex(DH * x0, int x1, int x2, BN_GENCB * x3)
{
  return DH_generate_parameters_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DH * x0;
  int x1;
  int x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_generate_parameters_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(621), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(621), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_parameters_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_parameters_ex _cffi_d_DH_generate_parameters_ex
#endif

static void _cffi_d_DH_get0_key(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DH_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_key(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(858), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(858), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_key _cffi_d_DH_get0_key
#endif

static void _cffi_d_DH_get0_pqg(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DH_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_pqg(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(858), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(858), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_pqg _cffi_d_DH_get0_pqg
#endif

static DH * _cffi_d_DH_new(void)
{
  return DH_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_new(PyObject *self, PyObject *noarg)
{
  DH * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(139));
  return pyresult;
}
#else
#  define _cffi_f_DH_new _cffi_d_DH_new
#endif

static int _cffi_d_DH_set0_key(DH * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DH_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_key(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_key _cffi_d_DH_set0_key
#endif

static int _cffi_d_DH_set0_pqg(DH * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DH_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_pqg(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_pqg _cffi_d_DH_set0_pqg
#endif

static int _cffi_d_DH_size(DH const * x0)
{
  return DH_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_size(PyObject *self, PyObject *arg0)
{
  DH const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(858), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(858), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_size _cffi_d_DH_size
#endif

static DH * _cffi_d_DHparams_dup(DH * x0)
{
  return DHparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DHparams_dup(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DHparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(139));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DHparams_dup _cffi_d_DHparams_dup
#endif

static void _cffi_d_DSA_free(DSA * x0)
{
  DSA_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_free(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_free _cffi_d_DSA_free
#endif

static int _cffi_d_DSA_generate_key(DSA * x0)
{
  return DSA_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_key(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_key _cffi_d_DSA_generate_key
#endif

static int _cffi_d_DSA_generate_parameters_ex(DSA * x0, int x1, unsigned char * x2, int x3, int * x4, unsigned long * x5, BN_GENCB * x6)
{
  return DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DSA * x0;
  int x1;
  unsigned char * x2;
  int x3;
  int * x4;
  unsigned long * x5;
  BN_GENCB * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "DSA_generate_parameters_ex", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(862), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(862), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(884), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(884), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(621), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(621), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_parameters_ex _cffi_d_DSA_generate_parameters_ex
#endif

static void _cffi_d_DSA_get0_key(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DSA_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_key(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(888), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(888), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_key _cffi_d_DSA_get0_key
#endif

static void _cffi_d_DSA_get0_pqg(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DSA_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_pqg(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(888), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(888), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_pqg _cffi_d_DSA_get0_pqg
#endif

static DSA * _cffi_d_DSA_new(void)
{
  return DSA_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_new(PyObject *self, PyObject *noarg)
{
  DSA * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(147));
  return pyresult;
}
#else
#  define _cffi_f_DSA_new _cffi_d_DSA_new
#endif

static int _cffi_d_DSA_set0_key(DSA * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DSA_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_key(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_key _cffi_d_DSA_set0_key
#endif

static int _cffi_d_DSA_set0_pqg(DSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DSA_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_pqg(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_pqg _cffi_d_DSA_set0_pqg
#endif

static int _cffi_d_DSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, DSA * x5)
{
  return DSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_sign _cffi_d_DSA_sign
#endif

static int _cffi_d_DSA_size(DSA const * x0)
{
  return DSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_size(PyObject *self, PyObject *arg0)
{
  DSA const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(888), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(888), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_size _cffi_d_DSA_size
#endif

static int _cffi_d_DSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, DSA * x5)
{
  return DSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_verify _cffi_d_DSA_verify
#endif

static DSA * _cffi_d_DSAparams_dup(DSA * x0)
{
  return DSAparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSAparams_dup(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSAparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(147));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSAparams_dup _cffi_d_DSAparams_dup
#endif

static SSL_METHOD const * _cffi_d_DTLS_client_method(void)
{
  return DTLS_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_client_method _cffi_d_DTLS_client_method
#endif

static size_t _cffi_d_DTLS_get_data_mtu(SSL * x0)
{
  return DTLS_get_data_mtu(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_get_data_mtu(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_get_data_mtu(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_get_data_mtu _cffi_d_DTLS_get_data_mtu
#endif

static long _cffi_d_DTLS_get_link_min_mtu(SSL * x0)
{
  return DTLS_get_link_min_mtu(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_get_link_min_mtu(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_get_link_min_mtu(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_get_link_min_mtu _cffi_d_DTLS_get_link_min_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLS_method(void)
{
  return DTLS_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_method _cffi_d_DTLS_method
#endif

static SSL_METHOD const * _cffi_d_DTLS_server_method(void)
{
  return DTLS_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_server_method _cffi_d_DTLS_server_method
#endif

static long _cffi_d_DTLS_set_link_mtu(SSL * x0, long x1)
{
  return DTLS_set_link_mtu(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_set_link_mtu(PyObject *self, PyObject *args)
{
  SSL * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "DTLS_set_link_mtu", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_set_link_mtu(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_set_link_mtu _cffi_d_DTLS_set_link_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_client_method(void)
{
  return DTLSv1_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_client_method _cffi_d_DTLSv1_client_method
#endif

static long _cffi_d_DTLSv1_handle_timeout(SSL * x0)
{
  return DTLSv1_handle_timeout(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_handle_timeout(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_handle_timeout(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_handle_timeout _cffi_d_DTLSv1_handle_timeout
#endif

static int _cffi_d_DTLSv1_listen(SSL * x0, BIO_ADDR * x1)
{
  return DTLSv1_listen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_listen(PyObject *self, PyObject *args)
{
  SSL * x0;
  BIO_ADDR * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "DTLSv1_listen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1287), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIO_ADDR *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1287), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_listen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_listen _cffi_d_DTLSv1_listen
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_method(void)
{
  return DTLSv1_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_method _cffi_d_DTLSv1_method
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_server_method(void)
{
  return DTLSv1_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2510));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_server_method _cffi_d_DTLSv1_server_method
#endif

static int _cffi_d_ECDSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, EC_KEY * x5)
{
  return ECDSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_sign _cffi_d_ECDSA_sign
#endif

static int _cffi_d_ECDSA_size(EC_KEY const * x0)
{
  return ECDSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_size(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(74), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(74), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_size _cffi_d_ECDSA_size
#endif

static int _cffi_d_ECDSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, EC_KEY * x5)
{
  return ECDSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_verify _cffi_d_ECDSA_verify
#endif

static void _cffi_d_EC_GROUP_free(EC_GROUP * x0)
{
  EC_GROUP_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_free(PyObject *self, PyObject *arg0)
{
  EC_GROUP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2135), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2135), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_GROUP_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_GROUP_free _cffi_d_EC_GROUP_free
#endif

static EC_POINT const * _cffi_d_EC_GROUP_get0_generator(EC_GROUP const * x0)
{
  return EC_GROUP_get0_generator(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get0_generator(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get0_generator(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(179));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get0_generator _cffi_d_EC_GROUP_get0_generator
#endif

static int _cffi_d_EC_GROUP_get_asn1_flag(EC_GROUP const * x0)
{
  return EC_GROUP_get_asn1_flag(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_asn1_flag(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_asn1_flag(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_asn1_flag _cffi_d_EC_GROUP_get_asn1_flag
#endif

static int _cffi_d_EC_GROUP_get_curve_name(EC_GROUP const * x0)
{
  return EC_GROUP_get_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_curve_name(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_curve_name _cffi_d_EC_GROUP_get_curve_name
#endif

static int _cffi_d_EC_GROUP_get_order(EC_GROUP const * x0, BIGNUM * x1, BN_CTX * x2)
{
  return EC_GROUP_get_order(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_order(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  BIGNUM * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_GROUP_get_order", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_order(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_order _cffi_d_EC_GROUP_get_order
#endif

static EC_METHOD const * _cffi_d_EC_GROUP_method_of(EC_GROUP const * x0)
{
  return EC_GROUP_method_of(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_method_of(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_METHOD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_method_of(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(984));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_method_of _cffi_d_EC_GROUP_method_of
#endif

static EC_GROUP * _cffi_d_EC_GROUP_new_by_curve_name(int x0)
{
  return EC_GROUP_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_GROUP * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2135));
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_new_by_curve_name _cffi_d_EC_GROUP_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_free(EC_KEY * x0)
{
  EC_KEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_free(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_free _cffi_d_EC_KEY_free
#endif

static int _cffi_d_EC_KEY_generate_key(EC_KEY * x0)
{
  return EC_KEY_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_generate_key(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_generate_key _cffi_d_EC_KEY_generate_key
#endif

static EC_GROUP const * _cffi_d_EC_KEY_get0_group(EC_KEY const * x0)
{
  return EC_KEY_get0_group(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_group(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_GROUP const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(74), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(74), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_group(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(173));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_group _cffi_d_EC_KEY_get0_group
#endif

static BIGNUM const * _cffi_d_EC_KEY_get0_private_key(EC_KEY const * x0)
{
  return EC_KEY_get0_private_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_private_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(74), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(74), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_private_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(56));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_private_key _cffi_d_EC_KEY_get0_private_key
#endif

static EC_POINT const * _cffi_d_EC_KEY_get0_public_key(EC_KEY const * x0)
{
  return EC_KEY_get0_public_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_public_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(74), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(74), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_public_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(179));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_public_key _cffi_d_EC_KEY_get0_public_key
#endif

static EC_KEY * _cffi_d_EC_KEY_new(void)
{
  return EC_KEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new(PyObject *self, PyObject *noarg)
{
  EC_KEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(666));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new _cffi_d_EC_KEY_new
#endif

static EC_KEY * _cffi_d_EC_KEY_new_by_curve_name(int x0)
{
  return EC_KEY_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_KEY * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(666));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new_by_curve_name _cffi_d_EC_KEY_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_set_asn1_flag(EC_KEY * x0, int x1)
{
  EC_KEY_set_asn1_flag(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_asn1_flag(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_asn1_flag", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_set_asn1_flag(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_set_asn1_flag _cffi_d_EC_KEY_set_asn1_flag
#endif

static int _cffi_d_EC_KEY_set_group(EC_KEY * x0, EC_GROUP const * x1)
{
  return EC_KEY_set_group(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_group(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_group", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_group(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_group _cffi_d_EC_KEY_set_group
#endif

static int _cffi_d_EC_KEY_set_private_key(EC_KEY * x0, BIGNUM const * x1)
{
  return EC_KEY_set_private_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_private_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_private_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_private_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_private_key _cffi_d_EC_KEY_set_private_key
#endif

static int _cffi_d_EC_KEY_set_public_key(EC_KEY * x0, EC_POINT const * x1)
{
  return EC_KEY_set_public_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key _cffi_d_EC_KEY_set_public_key
#endif

static int _cffi_d_EC_KEY_set_public_key_affine_coordinates(EC_KEY * x0, BIGNUM * x1, BIGNUM * x2)
{
  return EC_KEY_set_public_key_affine_coordinates(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key_affine_coordinates(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key_affine_coordinates", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key_affine_coordinates(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key_affine_coordinates _cffi_d_EC_KEY_set_public_key_affine_coordinates
#endif

static int _cffi_d_EC_METHOD_get_field_type(EC_METHOD const * x0)
{
  return EC_METHOD_get_field_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_METHOD_get_field_type(PyObject *self, PyObject *arg0)
{
  EC_METHOD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(984), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_METHOD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(984), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_METHOD_get_field_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_METHOD_get_field_type _cffi_d_EC_METHOD_get_field_type
#endif

static int _cffi_d_EC_POINT_add(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, EC_POINT const * x3, BN_CTX * x4)
{
  return EC_POINT_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_add(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  EC_POINT const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_add _cffi_d_EC_POINT_add
#endif

static void _cffi_d_EC_POINT_clear_free(EC_POINT * x0)
{
  EC_POINT_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_clear_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_clear_free _cffi_d_EC_POINT_clear_free
#endif

static int _cffi_d_EC_POINT_cmp(EC_GROUP const * x0, EC_POINT const * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_cmp(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_cmp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_cmp", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_cmp(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_cmp _cffi_d_EC_POINT_cmp
#endif

static int _cffi_d_EC_POINT_dbl(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_dbl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dbl(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dbl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dbl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dbl _cffi_d_EC_POINT_dbl
#endif

static EC_POINT * _cffi_d_EC_POINT_dup(EC_POINT const * x0, EC_GROUP const * x1)
{
  return EC_POINT_dup(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dup(PyObject *self, PyObject *args)
{
  EC_POINT const * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dup", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dup(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(900));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dup _cffi_d_EC_POINT_dup
#endif

static void _cffi_d_EC_POINT_free(EC_POINT * x0)
{
  EC_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_free _cffi_d_EC_POINT_free
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GF2m(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GF2m _cffi_d_EC_POINT_get_affine_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GFp _cffi_d_EC_POINT_get_affine_coordinates_GFp
#endif

static int _cffi_d_EC_POINT_invert(EC_GROUP const * x0, EC_POINT * x1, BN_CTX * x2)
{
  return EC_POINT_invert(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_invert(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_invert", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_invert(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_invert _cffi_d_EC_POINT_invert
#endif

static int _cffi_d_EC_POINT_is_at_infinity(EC_GROUP const * x0, EC_POINT const * x1)
{
  return EC_POINT_is_at_infinity(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_at_infinity(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_at_infinity", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_at_infinity(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_at_infinity _cffi_d_EC_POINT_is_at_infinity
#endif

static int _cffi_d_EC_POINT_is_on_curve(EC_GROUP const * x0, EC_POINT const * x1, BN_CTX * x2)
{
  return EC_POINT_is_on_curve(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_on_curve(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_on_curve", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_on_curve(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_on_curve _cffi_d_EC_POINT_is_on_curve
#endif

static int _cffi_d_EC_POINT_mul(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, EC_POINT const * x3, BIGNUM const * x4, BN_CTX * x5)
{
  return EC_POINT_mul(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_mul(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  EC_POINT const * x3;
  BIGNUM const * x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_mul", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_mul(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_mul _cffi_d_EC_POINT_mul
#endif

static EC_POINT * _cffi_d_EC_POINT_new(EC_GROUP const * x0)
{
  return EC_POINT_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_new(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(900));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_new _cffi_d_EC_POINT_new
#endif

static int _cffi_d_EC_POINT_oct2point(EC_GROUP const * x0, EC_POINT * x1, unsigned char const * x2, size_t x3, BN_CTX * x4)
{
  return EC_POINT_oct2point(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_oct2point(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  unsigned char const * x2;
  size_t x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_oct2point", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_oct2point(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_oct2point _cffi_d_EC_POINT_oct2point
#endif

static size_t _cffi_d_EC_POINT_point2oct(EC_GROUP const * x0, EC_POINT const * x1, point_conversion_form_t x2, unsigned char * x3, size_t x4, BN_CTX * x5)
{
  return EC_POINT_point2oct(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_point2oct(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  point_conversion_form_t x2;
  unsigned char * x3;
  size_t x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_point2oct", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(179), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(179), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  if (_cffi_to_c((char *)&x2, _cffi_type(1929), arg2) < 0)
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_point2oct(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_point2oct _cffi_d_EC_POINT_point2oct
#endif

static int _cffi_d_EC_POINT_set_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(173), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(173), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(900), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(900), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(56), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(56), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_affine_coordinates_GFp _cffi_d_EC_POINT_set_affine_coordinates_GFp
#endif

static char const * _cffi_d_EC_curve_nid2nist(int x0)
{
  return EC_curve_nid2nist(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_curve_nid2nist(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_curve_nid2nist(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_EC_curve_nid2nist _cffi_d_EC_curve_nid2nist
#endif

static size_t _cffi_d_EC_get_builtin_curves(EC_builtin_curve * x0, size_t x1)
{
  return EC_get_builtin_curves(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_get_builtin_curves(PyObject *self, PyObject *args)
{
  EC_builtin_curve * x0;
  size_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_get_builtin_curves", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1935), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_builtin_curve *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1935), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_get_builtin_curves(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_get_builtin_curves _cffi_d_EC_get_builtin_curves
#endif

static ENGINE * _cffi_d_ENGINE_by_id(char const * x0)
{
  return ENGINE_by_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_by_id(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ENGINE * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_by_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(219));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_by_id _cffi_d_ENGINE_by_id
#endif

static int _cffi_d_ENGINE_ctrl_cmd(ENGINE * x0, char const * x1, long x2, void * x3, void(* x4)(void), int x5)
{
  return ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_ctrl_cmd(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  long x2;
  void * x3;
  void(* x4)(void);
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ENGINE_ctrl_cmd", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, long);
  if (x2 == (long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = (void(*)(void))_cffi_to_c_pointer(arg4, _cffi_type(1000));
  if (x4 == (void(*)(void))NULL && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_ctrl_cmd _cffi_d_ENGINE_ctrl_cmd
#endif

static int _cffi_d_ENGINE_ctrl_cmd_string(ENGINE * x0, char const * x1, char const * x2, int x3)
{
  return ENGINE_ctrl_cmd_string(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_ctrl_cmd_string(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  char const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_ctrl_cmd_string", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_ctrl_cmd_string(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_ctrl_cmd_string _cffi_d_ENGINE_ctrl_cmd_string
#endif

static int _cffi_d_ENGINE_finish(ENGINE * x0)
{
  return ENGINE_finish(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_finish(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_finish(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_finish _cffi_d_ENGINE_finish
#endif

static int _cffi_d_ENGINE_free(ENGINE * x0)
{
  return ENGINE_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_free(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_free _cffi_d_ENGINE_free
#endif

static ENGINE * _cffi_d_ENGINE_get_default_RAND(void)
{
  return ENGINE_get_default_RAND();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_default_RAND(PyObject *self, PyObject *noarg)
{
  ENGINE * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_default_RAND(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(219));
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_default_RAND _cffi_d_ENGINE_get_default_RAND
#endif

static char const * _cffi_d_ENGINE_get_name(ENGINE const * x0)
{
  return ENGINE_get_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_name(PyObject *self, PyObject *arg0)
{
  ENGINE const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(526), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(526), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_name _cffi_d_ENGINE_get_name
#endif

static int _cffi_d_ENGINE_init(ENGINE * x0)
{
  return ENGINE_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_init(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_init _cffi_d_ENGINE_init
#endif

static void _cffi_d_ENGINE_load_builtin_engines(void)
{
  ENGINE_load_builtin_engines();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_builtin_engines(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ENGINE_load_builtin_engines(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ENGINE_load_builtin_engines _cffi_d_ENGINE_load_builtin_engines
#endif

static EVP_PKEY * _cffi_d_ENGINE_load_private_key(ENGINE * x0, char const * x1, UI_METHOD * x2, void * x3)
{
  return ENGINE_load_private_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_private_key(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  UI_METHOD * x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_load_private_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(221), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (UI_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(221), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_load_private_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_load_private_key _cffi_d_ENGINE_load_private_key
#endif

static EVP_PKEY * _cffi_d_ENGINE_load_public_key(ENGINE * x0, char const * x1, UI_METHOD * x2, void * x3)
{
  return ENGINE_load_public_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_public_key(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  UI_METHOD * x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_load_public_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(221), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (UI_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(221), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_load_public_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_load_public_key _cffi_d_ENGINE_load_public_key
#endif

static int _cffi_d_ENGINE_set_default_RAND(ENGINE * x0)
{
  return ENGINE_set_default_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_set_default_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_set_default_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_set_default_RAND _cffi_d_ENGINE_set_default_RAND
#endif

static void _cffi_d_ENGINE_unregister_RAND(ENGINE * x0)
{
  ENGINE_unregister_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_unregister_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ENGINE_unregister_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ENGINE_unregister_RAND _cffi_d_ENGINE_unregister_RAND
#endif

static int _cffi_d_ERR_GET_LIB(unsigned long x0)
{
  return ERR_GET_LIB(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_LIB(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_LIB(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_LIB _cffi_d_ERR_GET_LIB
#endif

static int _cffi_d_ERR_GET_REASON(unsigned long x0)
{
  return ERR_GET_REASON(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_REASON(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_REASON(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_REASON _cffi_d_ERR_GET_REASON
#endif

static void _cffi_d_ERR_clear_error(void)
{
  ERR_clear_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_clear_error(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_clear_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_clear_error _cffi_d_ERR_clear_error
#endif

static void _cffi_d_ERR_error_string_n(unsigned long x0, char * x1, size_t x2)
{
  ERR_error_string_n(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_error_string_n(PyObject *self, PyObject *args)
{
  unsigned long x0;
  char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ERR_error_string_n", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_error_string_n(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_error_string_n _cffi_d_ERR_error_string_n
#endif

static char const * _cffi_d_ERR_func_error_string(unsigned long x0)
{
  return ERR_func_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_func_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_func_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_ERR_func_error_string _cffi_d_ERR_func_error_string
#endif

static unsigned long _cffi_d_ERR_get_error(void)
{
  return ERR_get_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_get_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_get_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_get_error _cffi_d_ERR_get_error
#endif

static char const * _cffi_d_ERR_lib_error_string(unsigned long x0)
{
  return ERR_lib_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_lib_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_lib_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_ERR_lib_error_string _cffi_d_ERR_lib_error_string
#endif

static unsigned long _cffi_d_ERR_peek_error(void)
{
  return ERR_peek_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_peek_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_peek_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_peek_error _cffi_d_ERR_peek_error
#endif

static void _cffi_d_ERR_put_error(int x0, int x1, int x2, char const * x3, int x4)
{
  ERR_put_error(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_put_error(PyObject *self, PyObject *args)
{
  int x0;
  int x1;
  int x2;
  char const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "ERR_put_error", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_put_error(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_put_error _cffi_d_ERR_put_error
#endif

static char const * _cffi_d_ERR_reason_error_string(unsigned long x0)
{
  return ERR_reason_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_reason_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_reason_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_ERR_reason_error_string _cffi_d_ERR_reason_error_string
#endif

static int _cffi_d_EVP_CIPHER_CTX_ctrl(EVP_CIPHER_CTX * x0, int x1, int x2, void * x3)
{
  return EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_ctrl(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_ctrl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_ctrl _cffi_d_EVP_CIPHER_CTX_ctrl
#endif

static void _cffi_d_EVP_CIPHER_CTX_free(EVP_CIPHER_CTX * x0)
{
  EVP_CIPHER_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_CIPHER_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_free _cffi_d_EVP_CIPHER_CTX_free
#endif

static EVP_CIPHER_CTX * _cffi_d_EVP_CIPHER_CTX_new(void)
{
  return EVP_CIPHER_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_CIPHER_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1004));
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_new _cffi_d_EVP_CIPHER_CTX_new
#endif

static int _cffi_d_EVP_CIPHER_CTX_reset(EVP_CIPHER_CTX * x0)
{
  return EVP_CIPHER_CTX_reset(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_reset(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_reset(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_reset _cffi_d_EVP_CIPHER_CTX_reset
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_key_length(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_key_length(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_key_length(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_key_length", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_key_length(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_key_length _cffi_d_EVP_CIPHER_CTX_set_key_length
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_padding(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_padding(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_padding(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_padding", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_padding(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_padding _cffi_d_EVP_CIPHER_CTX_set_padding
#endif

static EVP_CIPHER * _cffi_d_EVP_CIPHER_fetch(OSSL_LIB_CTX * x0, char const * x1, char const * x2)
{
  return EVP_CIPHER_fetch(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_fetch(PyObject *self, PyObject *args)
{
  OSSL_LIB_CTX * x0;
  char const * x1;
  char const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_fetch", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(194), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OSSL_LIB_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(194), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_fetch(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2151));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_fetch _cffi_d_EVP_CIPHER_fetch
#endif

static void _cffi_d_EVP_CIPHER_free(EVP_CIPHER * x0)
{
  EVP_CIPHER_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_free(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2151), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2151), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_CIPHER_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_CIPHER_free _cffi_d_EVP_CIPHER_free
#endif

static int _cffi_d_EVP_CipherFinal_ex(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2)
{
  return EVP_CipherFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherFinal_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_CipherFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(862), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(862), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherFinal_ex _cffi_d_EVP_CipherFinal_ex
#endif

static int _cffi_d_EVP_CipherInit_ex(EVP_CIPHER_CTX * x0, EVP_CIPHER const * x1, ENGINE * x2, unsigned char const * x3, unsigned char const * x4, int x5)
{
  return EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherInit_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  EVP_CIPHER const * x1;
  ENGINE * x2;
  unsigned char const * x3;
  unsigned char const * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EVP_CipherInit_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherInit_ex _cffi_d_EVP_CipherInit_ex
#endif

static int _cffi_d_EVP_CipherUpdate(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2, unsigned char const * x3, int x4)
{
  return EVP_CipherUpdate(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherUpdate(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  unsigned char const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_CipherUpdate", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1004), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1004), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(862), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(862), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherUpdate(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherUpdate _cffi_d_EVP_CipherUpdate
#endif

static int _cffi_d_EVP_DigestFinalXOF(EVP_MD_CTX * x0, unsigned char * x1, size_t x2)
{
  return EVP_DigestFinalXOF(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinalXOF(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinalXOF", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinalXOF(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinalXOF _cffi_d_EVP_DigestFinalXOF
#endif

static int _cffi_d_EVP_DigestFinal_ex(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return EVP_DigestFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinal_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinal_ex _cffi_d_EVP_DigestFinal_ex
#endif

static int _cffi_d_EVP_DigestInit_ex(EVP_MD_CTX * x0, EVP_MD const * x1, ENGINE * x2)
{
  return EVP_DigestInit_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestInit_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  ENGINE * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestInit_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestInit_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestInit_ex _cffi_d_EVP_DigestInit_ex
#endif

static int _cffi_d_EVP_DigestSign(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestSign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSign(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSign _cffi_d_EVP_DigestSign
#endif

static int _cffi_d_EVP_DigestSignFinal(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_DigestSignFinal(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignFinal", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignFinal(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignFinal _cffi_d_EVP_DigestSignFinal
#endif

static int _cffi_d_EVP_DigestSignInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestSignInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1051), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1051), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignInit _cffi_d_EVP_DigestSignInit
#endif

static int _cffi_d_EVP_DigestSignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestSignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignUpdate _cffi_d_EVP_DigestSignUpdate
#endif

static int _cffi_d_EVP_DigestUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestUpdate _cffi_d_EVP_DigestUpdate
#endif

static int _cffi_d_EVP_DigestVerify(EVP_MD_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestVerify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerify(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerify _cffi_d_EVP_DigestVerify
#endif

static int _cffi_d_EVP_DigestVerifyInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestVerifyInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerifyInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1051), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1051), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerifyInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerifyInit _cffi_d_EVP_DigestVerifyInit
#endif

static int _cffi_d_EVP_MD_CTX_copy_ex(EVP_MD_CTX * x0, EVP_MD_CTX const * x1)
{
  return EVP_MD_CTX_copy_ex(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_copy_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_MD_CTX_copy_ex", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1047), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1047), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_copy_ex(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_copy_ex _cffi_d_EVP_MD_CTX_copy_ex
#endif

static void _cffi_d_EVP_MD_CTX_free(EVP_MD_CTX * x0)
{
  EVP_MD_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_MD_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_MD_CTX_free _cffi_d_EVP_MD_CTX_free
#endif

static EVP_MD_CTX * _cffi_d_EVP_MD_CTX_new(void)
{
  return EVP_MD_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_MD_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1037));
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_new _cffi_d_EVP_MD_CTX_new
#endif

static int _cffi_d_EVP_PBE_scrypt(char const * x0, size_t x1, unsigned char const * x2, size_t x3, uint64_t x4, uint64_t x5, uint64_t x6, uint64_t x7, unsigned char * x8, size_t x9)
{
  return EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PBE_scrypt(PyObject *self, PyObject *args)
{
  char const * x0;
  size_t x1;
  unsigned char const * x2;
  size_t x3;
  uint64_t x4;
  uint64_t x5;
  uint64_t x6;
  uint64_t x7;
  unsigned char * x8;
  size_t x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "EVP_PBE_scrypt", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, uint64_t);
  if (x4 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, uint64_t);
  if (x5 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, uint64_t);
  if (x6 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, uint64_t);
  if (x7 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x9 = _cffi_to_c_int(arg9, size_t);
  if (x9 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PBE_scrypt _cffi_d_EVP_PBE_scrypt
#endif

static void _cffi_d_EVP_PKEY_CTX_free(EVP_PKEY_CTX * x0)
{
  EVP_PKEY_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_free _cffi_d_EVP_PKEY_CTX_free
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new(EVP_PKEY * x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1143));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new _cffi_d_EVP_PKEY_CTX_new
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new_id(int x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new_id(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new_id", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1143));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new_id _cffi_d_EVP_PKEY_CTX_new_id
#endif

static int _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label(EVP_PKEY_CTX * x0, unsigned char * x1, int x2)
{
  return EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set0_rsa_oaep_label", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_mgf1_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1147), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1147), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_oaep_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1147), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1147), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_padding(EVP_PKEY_CTX * x0, int x1)
{
  return EVP_PKEY_CTX_set_rsa_padding(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_padding(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_padding", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_padding(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_padding _cffi_d_EVP_PKEY_CTX_set_rsa_padding
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_pss_saltlen(EVP_PKEY_CTX * x0, int x1)
{
  return EVP_PKEY_CTX_set_rsa_pss_saltlen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_pss_saltlen(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_pss_saltlen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_pss_saltlen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_pss_saltlen _cffi_d_EVP_PKEY_CTX_set_rsa_pss_saltlen
#endif

static int _cffi_d_EVP_PKEY_CTX_set_signature_md(EVP_PKEY_CTX * x0, EVP_MD const * x1)
{
  return EVP_PKEY_CTX_set_signature_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_signature_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_signature_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_signature_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_signature_md _cffi_d_EVP_PKEY_CTX_set_signature_md
#endif

static int _cffi_d_EVP_PKEY_assign_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_assign_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_assign_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_assign_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_assign_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_assign_RSA _cffi_d_EVP_PKEY_assign_RSA
#endif

static int _cffi_d_EVP_PKEY_bits(EVP_PKEY * x0)
{
  return EVP_PKEY_bits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_bits(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_bits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_bits _cffi_d_EVP_PKEY_bits
#endif

static int _cffi_d_EVP_PKEY_cmp(EVP_PKEY const * x0, EVP_PKEY const * x1)
{
  return EVP_PKEY_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_cmp(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  EVP_PKEY const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1131), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1131), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1131), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1131), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_cmp _cffi_d_EVP_PKEY_cmp
#endif

static int _cffi_d_EVP_PKEY_decrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_decrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_decrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt _cffi_d_EVP_PKEY_decrypt
#endif

static int _cffi_d_EVP_PKEY_decrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_decrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt_init _cffi_d_EVP_PKEY_decrypt_init
#endif

static int _cffi_d_EVP_PKEY_derive(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_derive(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive _cffi_d_EVP_PKEY_derive
#endif

static int _cffi_d_EVP_PKEY_derive_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_derive_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_init _cffi_d_EVP_PKEY_derive_init
#endif

static int _cffi_d_EVP_PKEY_derive_set_peer(EVP_PKEY_CTX * x0, EVP_PKEY * x1)
{
  return EVP_PKEY_derive_set_peer(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_set_peer(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive_set_peer", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_set_peer(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_set_peer _cffi_d_EVP_PKEY_derive_set_peer
#endif

static int _cffi_d_EVP_PKEY_encrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_encrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_encrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt _cffi_d_EVP_PKEY_encrypt
#endif

static int _cffi_d_EVP_PKEY_encrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_encrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt_init _cffi_d_EVP_PKEY_encrypt_init
#endif

static void _cffi_d_EVP_PKEY_free(EVP_PKEY * x0)
{
  EVP_PKEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_free _cffi_d_EVP_PKEY_free
#endif

static DH * _cffi_d_EVP_PKEY_get1_DH(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DH(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DH(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DH(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(139));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DH _cffi_d_EVP_PKEY_get1_DH
#endif

static DSA * _cffi_d_EVP_PKEY_get1_DSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(147));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DSA _cffi_d_EVP_PKEY_get1_DSA
#endif

static EC_KEY * _cffi_d_EVP_PKEY_get1_EC_KEY(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_EC_KEY(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_EC_KEY(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_KEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_EC_KEY(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(666));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_EC_KEY _cffi_d_EVP_PKEY_get1_EC_KEY
#endif

static RSA * _cffi_d_EVP_PKEY_get1_RSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_RSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_RSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_RSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(323));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_RSA _cffi_d_EVP_PKEY_get1_RSA
#endif

static size_t _cffi_d_EVP_PKEY_get1_tls_encodedpoint(EVP_PKEY * x0, unsigned char * * x1)
{
  return EVP_PKEY_get1_tls_encodedpoint(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get1_tls_encodedpoint", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1604), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1604), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_tls_encodedpoint(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_tls_encodedpoint _cffi_d_EVP_PKEY_get1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_get_raw_private_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_private_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_private_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_private_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1131), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1131), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_private_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_private_key _cffi_d_EVP_PKEY_get_raw_private_key
#endif

static int _cffi_d_EVP_PKEY_get_raw_public_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_public_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_public_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_public_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1131), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1131), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_public_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_public_key _cffi_d_EVP_PKEY_get_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_id(EVP_PKEY const * x0)
{
  return EVP_PKEY_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_id(PyObject *self, PyObject *arg0)
{
  EVP_PKEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1131), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1131), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_id _cffi_d_EVP_PKEY_id
#endif

static int _cffi_d_EVP_PKEY_keygen(EVP_PKEY_CTX * x0, EVP_PKEY * * x1)
{
  return EVP_PKEY_keygen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_keygen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(210), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(210), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen _cffi_d_EVP_PKEY_keygen
#endif

static int _cffi_d_EVP_PKEY_keygen_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_keygen_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen_init _cffi_d_EVP_PKEY_keygen_init
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new(void)
{
  return EVP_PKEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new(PyObject *self, PyObject *noarg)
{
  EVP_PKEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new _cffi_d_EVP_PKEY_new
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_private_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_private_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_private_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_private_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_private_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_private_key _cffi_d_EVP_PKEY_new_raw_private_key
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_public_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_public_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_public_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_public_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_public_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_public_key _cffi_d_EVP_PKEY_new_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_set1_DH(EVP_PKEY * x0, DH * x1)
{
  return EVP_PKEY_set1_DH(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DH(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DH", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DH(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DH _cffi_d_EVP_PKEY_set1_DH
#endif

static int _cffi_d_EVP_PKEY_set1_DSA(EVP_PKEY * x0, DSA * x1)
{
  return EVP_PKEY_set1_DSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DSA _cffi_d_EVP_PKEY_set1_DSA
#endif

static int _cffi_d_EVP_PKEY_set1_EC_KEY(EVP_PKEY * x0, EC_KEY * x1)
{
  return EVP_PKEY_set1_EC_KEY(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_EC_KEY(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  EC_KEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_EC_KEY", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_EC_KEY(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_EC_KEY _cffi_d_EVP_PKEY_set1_EC_KEY
#endif

static int _cffi_d_EVP_PKEY_set1_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_set1_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_RSA _cffi_d_EVP_PKEY_set1_RSA
#endif

static int _cffi_d_EVP_PKEY_set1_tls_encodedpoint(EVP_PKEY * x0, unsigned char const * x1, size_t x2)
{
  return EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_tls_encodedpoint", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_tls_encodedpoint _cffi_d_EVP_PKEY_set1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_set_type(EVP_PKEY * x0, int x1)
{
  return EVP_PKEY_set_type(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set_type(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set_type", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set_type(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set_type _cffi_d_EVP_PKEY_set_type
#endif

static int _cffi_d_EVP_PKEY_sign(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_sign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_sign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign _cffi_d_EVP_PKEY_sign
#endif

static int _cffi_d_EVP_PKEY_sign_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_sign_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign_init _cffi_d_EVP_PKEY_sign_init
#endif

static int _cffi_d_EVP_PKEY_size(EVP_PKEY * x0)
{
  return EVP_PKEY_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_size(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_size _cffi_d_EVP_PKEY_size
#endif

static int _cffi_d_EVP_PKEY_type(int x0)
{
  return EVP_PKEY_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_type(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_type _cffi_d_EVP_PKEY_type
#endif

static int _cffi_d_EVP_PKEY_verify(EVP_PKEY_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_verify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_verify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify _cffi_d_EVP_PKEY_verify
#endif

static int _cffi_d_EVP_PKEY_verify_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_verify_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_init _cffi_d_EVP_PKEY_verify_init
#endif

static int _cffi_d_EVP_PKEY_verify_recover(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_verify_recover(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_recover(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_verify_recover", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(796), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(796), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_recover(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_recover _cffi_d_EVP_PKEY_verify_recover
#endif

static int _cffi_d_EVP_PKEY_verify_recover_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_verify_recover_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_recover_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1143), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1143), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_recover_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_recover_init _cffi_d_EVP_PKEY_verify_recover_init
#endif

static int _cffi_d_EVP_SignFinal(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2, EVP_PKEY * x3)
{
  return EVP_SignFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_SignFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignFinal _cffi_d_EVP_SignFinal
#endif

static int _cffi_d_EVP_SignInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_SignInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_SignInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignInit _cffi_d_EVP_SignInit
#endif

static int _cffi_d_EVP_SignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_SignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_SignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignUpdate _cffi_d_EVP_SignUpdate
#endif

static int _cffi_d_EVP_VerifyFinal(EVP_MD_CTX * x0, unsigned char const * x1, unsigned int x2, EVP_PKEY * x3)
{
  return EVP_VerifyFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  unsigned int x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyFinal _cffi_d_EVP_VerifyFinal
#endif

static int _cffi_d_EVP_VerifyInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_VerifyInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyInit _cffi_d_EVP_VerifyInit
#endif

static int _cffi_d_EVP_VerifyUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_VerifyUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyUpdate _cffi_d_EVP_VerifyUpdate
#endif

static int _cffi_d_EVP_default_properties_enable_fips(OSSL_LIB_CTX * x0, int x1)
{
  return EVP_default_properties_enable_fips(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_default_properties_enable_fips(PyObject *self, PyObject *args)
{
  OSSL_LIB_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_default_properties_enable_fips", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(194), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OSSL_LIB_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(194), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_default_properties_enable_fips(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_default_properties_enable_fips _cffi_d_EVP_default_properties_enable_fips
#endif

static int _cffi_d_EVP_default_properties_is_fips_enabled(OSSL_LIB_CTX * x0)
{
  return EVP_default_properties_is_fips_enabled(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_default_properties_is_fips_enabled(PyObject *self, PyObject *arg0)
{
  OSSL_LIB_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(194), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OSSL_LIB_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(194), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_default_properties_is_fips_enabled(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_default_properties_is_fips_enabled _cffi_d_EVP_default_properties_is_fips_enabled
#endif

static EVP_CIPHER const * _cffi_d_EVP_get_cipherbyname(char const * x0)
{
  return EVP_get_cipherbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_cipherbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_cipherbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(658));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_cipherbyname _cffi_d_EVP_get_cipherbyname
#endif

static EVP_MD const * _cffi_d_EVP_get_digestbyname(char const * x0)
{
  return EVP_get_digestbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_digestbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_MD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_digestbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(306));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_digestbyname _cffi_d_EVP_get_digestbyname
#endif

static int _cffi_d_FIPS_mode(void)
{
  return FIPS_mode();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_FIPS_mode(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = FIPS_mode(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_FIPS_mode _cffi_d_FIPS_mode
#endif

static int _cffi_d_FIPS_mode_set(int x0)
{
  return FIPS_mode_set(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_FIPS_mode_set(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = FIPS_mode_set(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_FIPS_mode_set _cffi_d_FIPS_mode_set
#endif

static void _cffi_d_GENERAL_NAMES_free(GENERAL_NAMES * x0)
{
  GENERAL_NAMES_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAMES_free(PyObject *self, PyObject *arg0)
{
  GENERAL_NAMES * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(250), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (GENERAL_NAMES *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(250), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { GENERAL_NAMES_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_GENERAL_NAMES_free _cffi_d_GENERAL_NAMES_free
#endif

static int _cffi_d_GENERAL_NAME_print(BIO * x0, GENERAL_NAME * x1)
{
  return GENERAL_NAME_print(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  GENERAL_NAME * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "GENERAL_NAME_print", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(701), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (GENERAL_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(701), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAME_print(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAME_print _cffi_d_GENERAL_NAME_print
#endif

static int _cffi_d_HMAC_CTX_copy(HMAC_CTX * x0, HMAC_CTX * x1)
{
  return HMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_copy(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  HMAC_CTX * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "HMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_CTX_copy _cffi_d_HMAC_CTX_copy
#endif

static void _cffi_d_HMAC_CTX_free(HMAC_CTX * x0)
{
  HMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  HMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { HMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_HMAC_CTX_free _cffi_d_HMAC_CTX_free
#endif

static HMAC_CTX * _cffi_d_HMAC_CTX_new(void)
{
  return HMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  HMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1193));
  return pyresult;
}
#else
#  define _cffi_f_HMAC_CTX_new _cffi_d_HMAC_CTX_new
#endif

static int _cffi_d_HMAC_Final(HMAC_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return HMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Final(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Final _cffi_d_HMAC_Final
#endif

static int _cffi_d_HMAC_Init_ex(HMAC_CTX * x0, void const * x1, int x2, EVP_MD const * x3, ENGINE * x4)
{
  return HMAC_Init_ex(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Init_ex(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  void const * x1;
  int x2;
  EVP_MD const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "HMAC_Init_ex", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Init_ex(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Init_ex _cffi_d_HMAC_Init_ex
#endif

static int _cffi_d_HMAC_Update(HMAC_CTX * x0, unsigned char const * x1, size_t x2)
{
  return HMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Update(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Update _cffi_d_HMAC_Update
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_b64_decode(char const * x0, int x1)
{
  return NETSCAPE_SPKI_b64_decode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_decode(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  NETSCAPE_SPKI * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_b64_decode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_decode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(225));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_decode _cffi_d_NETSCAPE_SPKI_b64_decode
#endif

static char * _cffi_d_NETSCAPE_SPKI_b64_encode(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_b64_encode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_encode(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_encode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(270));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_encode _cffi_d_NETSCAPE_SPKI_b64_encode
#endif

static void _cffi_d_NETSCAPE_SPKI_free(NETSCAPE_SPKI * x0)
{
  NETSCAPE_SPKI_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_free(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NETSCAPE_SPKI_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_free _cffi_d_NETSCAPE_SPKI_free
#endif

static EVP_PKEY * _cffi_d_NETSCAPE_SPKI_get_pubkey(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_get_pubkey(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_get_pubkey(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_get_pubkey(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_get_pubkey _cffi_d_NETSCAPE_SPKI_get_pubkey
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_new(void)
{
  return NETSCAPE_SPKI_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_new(PyObject *self, PyObject *noarg)
{
  NETSCAPE_SPKI * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(225));
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_new _cffi_d_NETSCAPE_SPKI_new
#endif

static int _cffi_d_NETSCAPE_SPKI_set_pubkey(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_set_pubkey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_set_pubkey(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_set_pubkey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_set_pubkey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_set_pubkey _cffi_d_NETSCAPE_SPKI_set_pubkey
#endif

static int _cffi_d_NETSCAPE_SPKI_sign(NETSCAPE_SPKI * x0, EVP_PKEY * x1, EVP_MD const * x2)
{
  return NETSCAPE_SPKI_sign(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_sign(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  EVP_MD const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_sign", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_sign(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_sign _cffi_d_NETSCAPE_SPKI_sign
#endif

static int _cffi_d_NETSCAPE_SPKI_verify(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_verify(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_verify(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_verify", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(225), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(225), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_verify(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_verify _cffi_d_NETSCAPE_SPKI_verify
#endif

static char const * _cffi_d_OBJ_nid2ln(int x0)
{
  return OBJ_nid2ln(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2ln(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2ln(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2ln _cffi_d_OBJ_nid2ln
#endif

static char const * _cffi_d_OBJ_nid2sn(int x0)
{
  return OBJ_nid2sn(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2sn(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2sn(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2sn _cffi_d_OBJ_nid2sn
#endif

static int _cffi_d_OBJ_obj2nid(ASN1_OBJECT const * x0)
{
  return OBJ_obj2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_obj2nid(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(558), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(558), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_obj2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_obj2nid _cffi_d_OBJ_obj2nid
#endif

static int _cffi_d_OBJ_sn2nid(char const * x0)
{
  return OBJ_sn2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_sn2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_sn2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_sn2nid _cffi_d_OBJ_sn2nid
#endif

static int _cffi_d_OBJ_txt2nid(char const * x0)
{
  return OBJ_txt2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2nid _cffi_d_OBJ_txt2nid
#endif

static ASN1_OBJECT * _cffi_d_OBJ_txt2obj(char const * x0, int x1)
{
  return OBJ_txt2obj(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2obj(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OBJECT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OBJ_txt2obj", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2obj(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(419));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2obj _cffi_d_OBJ_txt2obj
#endif

static void _cffi_d_OPENSSL_cleanup(void)
{
  OPENSSL_cleanup();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_cleanup(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_cleanup(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_cleanup _cffi_d_OPENSSL_cleanup
#endif

static void _cffi_d_OPENSSL_config(char const * x0)
{
  OPENSSL_config(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_config(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_config(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_config _cffi_d_OPENSSL_config
#endif

static void _cffi_d_OPENSSL_free(void * x0)
{
  OPENSSL_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_free(PyObject *self, PyObject *arg0)
{
  void * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_free _cffi_d_OPENSSL_free
#endif

static void * _cffi_d_OPENSSL_malloc(size_t x0)
{
  return OPENSSL_malloc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_malloc(PyObject *self, PyObject *arg0)
{
  size_t x0;
  void * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OPENSSL_malloc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(136));
  return pyresult;
}
#else
#  define _cffi_f_OPENSSL_malloc _cffi_d_OPENSSL_malloc
#endif

static OSSL_PROVIDER * _cffi_d_OSSL_PROVIDER_load(OSSL_LIB_CTX * x0, char const * x1)
{
  return OSSL_PROVIDER_load(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OSSL_PROVIDER_load(PyObject *self, PyObject *args)
{
  OSSL_LIB_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OSSL_PROVIDER * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OSSL_PROVIDER_load", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(194), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OSSL_LIB_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(194), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OSSL_PROVIDER_load(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1230));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OSSL_PROVIDER_load _cffi_d_OSSL_PROVIDER_load
#endif

static int _cffi_d_OSSL_PROVIDER_unload(OSSL_PROVIDER * x0)
{
  return OSSL_PROVIDER_unload(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OSSL_PROVIDER_unload(PyObject *self, PyObject *arg0)
{
  OSSL_PROVIDER * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1230), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OSSL_PROVIDER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1230), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OSSL_PROVIDER_unload(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OSSL_PROVIDER_unload _cffi_d_OSSL_PROVIDER_unload
#endif

static void _cffi_d_OpenSSL_add_all_algorithms(void)
{
  OpenSSL_add_all_algorithms();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_add_all_algorithms(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OpenSSL_add_all_algorithms(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OpenSSL_add_all_algorithms _cffi_d_OpenSSL_add_all_algorithms
#endif

static char const * _cffi_d_OpenSSL_version(int x0)
{
  return OpenSSL_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version _cffi_d_OpenSSL_version
#endif

static unsigned long _cffi_d_OpenSSL_version_num(void)
{
  return OpenSSL_version_num();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version_num(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version_num(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version_num _cffi_d_OpenSSL_version_num
#endif

static DH * _cffi_d_PEM_read_bio_DHparams(BIO * x0, DH * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_DHparams(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_DHparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_DHparams", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(130), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(130), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_DHparams(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(139));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_DHparams _cffi_d_PEM_read_bio_DHparams
#endif

static PKCS7 * _cffi_d_PEM_read_bio_PKCS7(BIO * x0, PKCS7 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PKCS7(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PKCS7", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(287), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(287), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PKCS7(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(100));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PKCS7 _cffi_d_PEM_read_bio_PKCS7
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PUBKEY(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PUBKEY(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PUBKEY(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PUBKEY", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(210), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(210), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PUBKEY(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PUBKEY _cffi_d_PEM_read_bio_PUBKEY
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PrivateKey(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PrivateKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PrivateKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(210), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(210), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PrivateKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(142));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PrivateKey _cffi_d_PEM_read_bio_PrivateKey
#endif

static RSA * _cffi_d_PEM_read_bio_RSAPublicKey(BIO * x0, RSA * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_RSAPublicKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_RSAPublicKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_RSAPublicKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(311), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(311), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_RSAPublicKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(323));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_RSAPublicKey _cffi_d_PEM_read_bio_RSAPublicKey
#endif

static X509 * _cffi_d_PEM_read_bio_X509(BIO * x0, X509 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(363), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(363), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(11));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509 _cffi_d_PEM_read_bio_X509
#endif

static X509_CRL * _cffi_d_PEM_read_bio_X509_CRL(BIO * x0, X509_CRL * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509_CRL(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509_CRL(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_CRL * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_CRL * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509_CRL", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(394), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_CRL * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(394), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509_CRL(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(34));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509_CRL _cffi_d_PEM_read_bio_X509_CRL
#endif

static X509_REQ * _cffi_d_PEM_read_bio_X509_REQ(BIO * x0, X509_REQ * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509_REQ(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509_REQ(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_REQ * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_REQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509_REQ", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(474), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_REQ * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(474), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(135));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509_REQ(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(231));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509_REQ _cffi_d_PEM_read_bio_X509_REQ
#endif

static int _cffi_d_PEM_write_bio_DHparams(BIO * x0, DH * x1)
{
  return PEM_write_bio_DHparams(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DHparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DHparams", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DHparams(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DHparams _cffi_d_PEM_write_bio_DHparams
#endif

static int _cffi_d_PEM_write_bio_DHxparams(BIO * x0, DH * x1)
{
  return PEM_write_bio_DHxparams(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DHxparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DHxparams", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DHxparams(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DHxparams _cffi_d_PEM_write_bio_DHxparams
#endif

static int _cffi_d_PEM_write_bio_DSAPrivateKey(BIO * x0, DSA * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_DSAPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DSAPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  DSA * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DSAPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(147), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(147), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(135));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DSAPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DSAPrivateKey _cffi_d_PEM_write_bio_DSAPrivateKey
#endif

static int _cffi_d_PEM_write_bio_ECPrivateKey(BIO * x0, EC_KEY * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_ECPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_ECPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EC_KEY * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_ECPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(135));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_ECPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_ECPrivateKey _cffi_d_PEM_write_bio_ECPrivateKey
#endif

static int _cffi_d_PEM_write_bio_PKCS7(BIO * x0, PKCS7 * x1)
{
  return PEM_write_bio_PKCS7(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS7", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS7(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS7 _cffi_d_PEM_write_bio_PKCS7
#endif

static int _cffi_d_PEM_write_bio_PKCS7_stream(BIO * x0, PKCS7 * x1, BIO * x2, int x3)
{
  return PEM_write_bio_PKCS7_stream(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS7_stream(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * x1;
  BIO * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS7_stream", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS7_stream(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS7_stream _cffi_d_PEM_write_bio_PKCS7_stream
#endif

static int _cffi_d_PEM_write_bio_PKCS8PrivateKey(BIO * x0, EVP_PKEY * x1, EVP_CIPHER const * x2, char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_PKCS8PrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS8PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  EVP_CIPHER const * x2;
  char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS8PrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(135));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS8PrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS8PrivateKey _cffi_d_PEM_write_bio_PKCS8PrivateKey
#endif

static int _cffi_d_PEM_write_bio_PUBKEY(BIO * x0, EVP_PKEY * x1)
{
  return PEM_write_bio_PUBKEY(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PUBKEY(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PUBKEY", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PUBKEY(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PUBKEY _cffi_d_PEM_write_bio_PUBKEY
#endif

static int _cffi_d_PEM_write_bio_PrivateKey(BIO * x0, EVP_PKEY * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_PrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(135));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PrivateKey _cffi_d_PEM_write_bio_PrivateKey
#endif

static int _cffi_d_PEM_write_bio_RSAPrivateKey(BIO * x0, RSA * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_RSAPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_RSAPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_RSAPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(658), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(658), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(135));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_RSAPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_RSAPrivateKey _cffi_d_PEM_write_bio_RSAPrivateKey
#endif

static int _cffi_d_PEM_write_bio_RSAPublicKey(BIO * x0, RSA const * x1)
{
  return PEM_write_bio_RSAPublicKey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_RSAPublicKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_RSAPublicKey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_RSAPublicKey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_RSAPublicKey _cffi_d_PEM_write_bio_RSAPublicKey
#endif

static int _cffi_d_PEM_write_bio_X509(BIO * x0, X509 * x1)
{
  return PEM_write_bio_X509(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509 _cffi_d_PEM_write_bio_X509
#endif

static int _cffi_d_PEM_write_bio_X509_CRL(BIO * x0, X509_CRL * x1)
{
  return PEM_write_bio_X509_CRL(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509_CRL(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_CRL * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509_CRL", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(34), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_CRL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(34), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509_CRL(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509_CRL _cffi_d_PEM_write_bio_X509_CRL
#endif

static int _cffi_d_PEM_write_bio_X509_REQ(BIO * x0, X509_REQ * x1)
{
  return PEM_write_bio_X509_REQ(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509_REQ(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_REQ * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509_REQ", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(231), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_REQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(231), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509_REQ(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509_REQ _cffi_d_PEM_write_bio_X509_REQ
#endif

static PKCS12 * _cffi_d_PKCS12_create(char * x0, char * x1, EVP_PKEY * x2, X509 * x3, Cryptography_STACK_OF_X509 * x4, int x5, int x6, int x7, int x8, int x9)
{
  return PKCS12_create(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_create(PyObject *self, PyObject *args)
{
  char * x0;
  char * x1;
  EVP_PKEY * x2;
  X509 * x3;
  Cryptography_STACK_OF_X509 * x4;
  int x5;
  int x6;
  int x7;
  int x8;
  int x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS12 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "PKCS12_create", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(270), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(270), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(101), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(101), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  x8 = _cffi_to_c_int(arg8, int);
  if (x8 == (int)-1 && PyErr_Occurred())
    return NULL;

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS12_create(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(705));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS12_create _cffi_d_PKCS12_create
#endif

static void _cffi_d_PKCS12_free(PKCS12 * x0)
{
  PKCS12_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_free(PyObject *self, PyObject *arg0)
{
  PKCS12 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(705), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS12 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(705), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { PKCS12_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_PKCS12_free _cffi_d_PKCS12_free
#endif

static int _cffi_d_PKCS12_parse(PKCS12 * x0, char const * x1, EVP_PKEY * * x2, X509 * * x3, Cryptography_STACK_OF_X509 * * x4)
{
  return PKCS12_parse(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_parse(PyObject *self, PyObject *args)
{
  PKCS12 * x0;
  char const * x1;
  EVP_PKEY * * x2;
  X509 * * x3;
  Cryptography_STACK_OF_X509 * * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS12_parse", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(705), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS12 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(705), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(210), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(210), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(363), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(363), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1237), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1237), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS12_parse(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS12_parse _cffi_d_PKCS12_parse
#endif

static int _cffi_d_PKCS5_PBKDF2_HMAC(char const * x0, int x1, unsigned char const * x2, int x3, int x4, EVP_MD const * x5, int x6, unsigned char * x7)
{
  return PKCS5_PBKDF2_HMAC(x0, x1, x2, x3, x4, x5, x6, x7);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS5_PBKDF2_HMAC(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  unsigned char const * x2;
  int x3;
  int x4;
  EVP_MD const * x5;
  int x6;
  unsigned char * x7;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;

  if (!PyArg_UnpackTuple(args, "PKCS5_PBKDF2_HMAC", 8, 8, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS5_PBKDF2_HMAC(x0, x1, x2, x3, x4, x5, x6, x7); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS5_PBKDF2_HMAC _cffi_d_PKCS5_PBKDF2_HMAC
#endif

static int _cffi_d_PKCS7_final(PKCS7 * x0, BIO * x1, int x2)
{
  return PKCS7_final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_final(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  BIO * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "PKCS7_final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_final _cffi_d_PKCS7_final
#endif

static void _cffi_d_PKCS7_free(PKCS7 * x0)
{
  PKCS7_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_free(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { PKCS7_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_PKCS7_free _cffi_d_PKCS7_free
#endif

static Cryptography_STACK_OF_X509 * _cffi_d_PKCS7_get0_signers(PKCS7 * x0, Cryptography_STACK_OF_X509 * x1, int x2)
{
  return PKCS7_get0_signers(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_get0_signers(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  Cryptography_STACK_OF_X509 * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  Cryptography_STACK_OF_X509 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "PKCS7_get0_signers", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(101), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(101), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_get0_signers(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(101));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_get0_signers _cffi_d_PKCS7_get0_signers
#endif

static PKCS7 * _cffi_d_PKCS7_sign(X509 * x0, EVP_PKEY * x1, Cryptography_STACK_OF_X509 * x2, BIO * x3, int x4)
{
  return PKCS7_sign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_sign(PyObject *self, PyObject *args)
{
  X509 * x0;
  EVP_PKEY * x1;
  Cryptography_STACK_OF_X509 * x2;
  BIO * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS7_sign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(101), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(101), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_sign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(100));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_sign _cffi_d_PKCS7_sign
#endif

static PKCS7_SIGNER_INFO * _cffi_d_PKCS7_sign_add_signer(PKCS7 * x0, X509 * x1, EVP_PKEY * x2, EVP_MD const * x3, int x4)
{
  return PKCS7_sign_add_signer(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_sign_add_signer(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  X509 * x1;
  EVP_PKEY * x2;
  EVP_MD const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7_SIGNER_INFO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS7_sign_add_signer", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(306), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(306), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_sign_add_signer(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2498));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_sign_add_signer _cffi_d_PKCS7_sign_add_signer
#endif

static int _cffi_d_PKCS7_type_is_data(PKCS7 * x0)
{
  return PKCS7_type_is_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_data(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_data _cffi_d_PKCS7_type_is_data
#endif

static int _cffi_d_PKCS7_type_is_enveloped(PKCS7 * x0)
{
  return PKCS7_type_is_enveloped(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_enveloped(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_enveloped(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_enveloped _cffi_d_PKCS7_type_is_enveloped
#endif

static int _cffi_d_PKCS7_type_is_signed(PKCS7 * x0)
{
  return PKCS7_type_is_signed(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_signed(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_signed(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_signed _cffi_d_PKCS7_type_is_signed
#endif

static int _cffi_d_PKCS7_type_is_signedAndEnveloped(PKCS7 * x0)
{
  return PKCS7_type_is_signedAndEnveloped(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_signedAndEnveloped(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_signedAndEnveloped(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_signedAndEnveloped _cffi_d_PKCS7_type_is_signedAndEnveloped
#endif

static int _cffi_d_PKCS7_verify(PKCS7 * x0, Cryptography_STACK_OF_X509 * x1, X509_STORE * x2, BIO * x3, BIO * x4, int x5)
{
  return PKCS7_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_verify(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  Cryptography_STACK_OF_X509 * x1;
  X509_STORE * x2;
  BIO * x3;
  BIO * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "PKCS7_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(101), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(101), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(123), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509_STORE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(123), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_verify _cffi_d_PKCS7_verify
#endif

static void _cffi_d_RAND_add(void const * x0, int x1, double x2)
{
  RAND_add(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_add(PyObject *self, PyObject *args)
{
  void const * x0;
  int x1;
  double x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RAND_add", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(86), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(86), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = (double)_cffi_to_c_double(arg2);
  if (x2 == (double)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RAND_add(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RAND_add _cffi_d_RAND_add
#endif

static int _cffi_d_RAND_bytes(unsigned char * x0, int x1)
{
  return RAND_bytes(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_bytes(PyObject *self, PyObject *args)
{
  unsigned char * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "RAND_bytes", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_bytes(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RAND_bytes _cffi_d_RAND_bytes
#endif

static int _cffi_d_RAND_set_rand_method(RAND_METHOD const * x0)
{
  return RAND_set_rand_method(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_set_rand_method(PyObject *self, PyObject *arg0)
{
  RAND_METHOD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1256), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RAND_METHOD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1256), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_set_rand_method(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RAND_set_rand_method _cffi_d_RAND_set_rand_method
#endif

static int _cffi_d_RAND_status(void)
{
  return RAND_status();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_status(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_status(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_RAND_status _cffi_d_RAND_status
#endif

static RSA * _cffi_d_RSAPublicKey_dup(RSA * x0)
{
  return RSAPublicKey_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSAPublicKey_dup(PyObject *self, PyObject *arg0)
{
  RSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSAPublicKey_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(323));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSAPublicKey_dup _cffi_d_RSAPublicKey_dup
#endif

static int _cffi_d_RSA_blinding_on(RSA * x0, BN_CTX * x1)
{
  return RSA_blinding_on(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_blinding_on(PyObject *self, PyObject *args)
{
  RSA * x0;
  BN_CTX * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "RSA_blinding_on", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(58), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(58), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_blinding_on(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_blinding_on _cffi_d_RSA_blinding_on
#endif

static int _cffi_d_RSA_check_key(RSA const * x0)
{
  return RSA_check_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_check_key(PyObject *self, PyObject *arg0)
{
  RSA const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_check_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_check_key _cffi_d_RSA_check_key
#endif

static void _cffi_d_RSA_free(RSA * x0)
{
  RSA_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_free(PyObject *self, PyObject *arg0)
{
  RSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_free _cffi_d_RSA_free
#endif

static int _cffi_d_RSA_generate_key_ex(RSA * x0, int x1, BIGNUM * x2, BN_GENCB * x3)
{
  return RSA_generate_key_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_generate_key_ex(PyObject *self, PyObject *args)
{
  RSA * x0;
  int x1;
  BIGNUM * x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_generate_key_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(621), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(621), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_generate_key_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_generate_key_ex _cffi_d_RSA_generate_key_ex
#endif

static void _cffi_d_RSA_get0_crt_params(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  RSA_get0_crt_params(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_crt_params(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_get0_crt_params", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_crt_params(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_crt_params _cffi_d_RSA_get0_crt_params
#endif

static void _cffi_d_RSA_get0_factors(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  RSA_get0_factors(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_factors(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_get0_factors", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_factors(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_factors _cffi_d_RSA_get0_factors
#endif

static void _cffi_d_RSA_get0_key(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  RSA_get0_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_key(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_get0_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2111), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2111), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_key _cffi_d_RSA_get0_key
#endif

static RSA * _cffi_d_RSA_new(void)
{
  return RSA_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_new(PyObject *self, PyObject *noarg)
{
  RSA * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(323));
  return pyresult;
}
#else
#  define _cffi_f_RSA_new _cffi_d_RSA_new
#endif

static int _cffi_d_RSA_print(BIO * x0, RSA const * x1, int x2)
{
  return RSA_print(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_print", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(732), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(732), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_print(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_print _cffi_d_RSA_print
#endif

static int _cffi_d_RSA_set0_crt_params(RSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return RSA_set0_crt_params(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_crt_params(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_set0_crt_params", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_crt_params(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_crt_params _cffi_d_RSA_set0_crt_params
#endif

static int _cffi_d_RSA_set0_factors(RSA * x0, BIGNUM * x1, BIGNUM * x2)
{
  return RSA_set0_factors(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_factors(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_set0_factors", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_factors(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_factors _cffi_d_RSA_set0_factors
#endif

static int _cffi_d_RSA_set0_key(RSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return RSA_set0_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_key(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_set0_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(323), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(323), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(7), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(7), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_key _cffi_d_RSA_set0_key
#endif

static PKCS7 * _cffi_d_SMIME_read_PKCS7(BIO * x0, BIO * * x1)
{
  return SMIME_read_PKCS7(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SMIME_read_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  BIO * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SMIME_read_PKCS7", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(283), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIO * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(283), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SMIME_read_PKCS7(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(100));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SMIME_read_PKCS7 _cffi_d_SMIME_read_PKCS7
#endif

static int _cffi_d_SMIME_write_PKCS7(BIO * x0, PKCS7 * x1, BIO * x2, int x3)
{
  return SMIME_write_PKCS7(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SMIME_write_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * x1;
  BIO * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "SMIME_write_PKCS7", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(100), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(100), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(129), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(129), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SMIME_write_PKCS7(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SMIME_write_PKCS7 _cffi_d_SMIME_write_PKCS7
#endif

static SSL_CIPHER const * _cffi_d_SSL_CIPHER_find(SSL * x0, unsigned char const * x1)
{
  return SSL_CIPHER_find(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_find(PyObject *self, PyObject *args)
{
  SSL * x0;
  unsigned char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  SSL_CIPHER const * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CIPHER_find", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(328), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(328), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_find(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(536));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_find _cffi_d_SSL_CIPHER_find
#endif

static int _cffi_d_SSL_CIPHER_get_auth_nid(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_auth_nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_auth_nid(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_auth_nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_auth_nid _cffi_d_SSL_CIPHER_get_auth_nid
#endif

static int _cffi_d_SSL_CIPHER_get_bits(SSL_CIPHER const * x0, int * x1)
{
  return SSL_CIPHER_get_bits(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_bits(PyObject *self, PyObject *args)
{
  SSL_CIPHER const * x0;
  int * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CIPHER_get_bits", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(862), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(862), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_bits(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_bits _cffi_d_SSL_CIPHER_get_bits
#endif

static int _cffi_d_SSL_CIPHER_get_cipher_nid(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_cipher_nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_cipher_nid(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_cipher_nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_cipher_nid _cffi_d_SSL_CIPHER_get_cipher_nid
#endif

static int _cffi_d_SSL_CIPHER_get_digest_nid(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_digest_nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_digest_nid(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_digest_nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_digest_nid _cffi_d_SSL_CIPHER_get_digest_nid
#endif

static uint64_t _cffi_d_SSL_CIPHER_get_id(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_id(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  uint64_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, uint64_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_id _cffi_d_SSL_CIPHER_get_id
#endif

static int _cffi_d_SSL_CIPHER_get_kx_nid(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_kx_nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_kx_nid(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_kx_nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_kx_nid _cffi_d_SSL_CIPHER_get_kx_nid
#endif

static char const * _cffi_d_SSL_CIPHER_get_name(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_name(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_name _cffi_d_SSL_CIPHER_get_name
#endif

static char const * _cffi_d_SSL_CIPHER_get_version(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_get_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_get_version(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_get_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(23));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_get_version _cffi_d_SSL_CIPHER_get_version
#endif

static int _cffi_d_SSL_CIPHER_is_aead(SSL_CIPHER const * x0)
{
  return SSL_CIPHER_is_aead(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CIPHER_is_aead(PyObject *self, PyObject *arg0)
{
  SSL_CIPHER const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(536), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(536), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CIPHER_is_aead(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CIPHER_is_aead _cffi_d_SSL_CIPHER_is_aead
#endif

static int _cffi_d_SSL_CTX_add_client_CA(SSL_CTX * x0, X509 * x1)
{
  return SSL_CTX_add_client_CA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_add_client_CA(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_add_client_CA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_add_client_CA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_add_client_CA _cffi_d_SSL_CTX_add_client_CA
#endif

static int _cffi_d_SSL_CTX_add_client_custom_ext(SSL_CTX * x0, unsigned int x1, int(* x2)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *), void(* x3)(SSL *, unsigned int, unsigned char const *, void *), void * x4, int(* x5)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *), void * x6)
{
  return SSL_CTX_add_client_custom_ext(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_add_client_custom_ext(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned int x1;
  int(* x2)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *);
  void(* x3)(SSL *, unsigned int, unsigned char const *, void *);
  void * x4;
  int(* x5)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_add_client_custom_ext", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned int);
  if (x1 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x2 = (int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *))_cffi_to_c_pointer(arg2, _cffi_type(1444));
  if (x2 == (int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *))NULL && PyErr_Occurred())
    return NULL;

  x3 = (void(*)(SSL *, unsigned int, unsigned char const *, void *))_cffi_to_c_pointer(arg3, _cffi_type(1445));
  if (x3 == (void(*)(SSL *, unsigned int, unsigned char const *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = (int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *))_cffi_to_c_pointer(arg5, _cffi_type(1447));
  if (x5 == (int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_add_client_custom_ext(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_add_client_custom_ext _cffi_d_SSL_CTX_add_client_custom_ext
#endif

static unsigned long _cffi_d_SSL_CTX_add_extra_chain_cert(SSL_CTX * x0, X509 * x1)
{
  return SSL_CTX_add_extra_chain_cert(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_add_extra_chain_cert(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_add_extra_chain_cert", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_add_extra_chain_cert(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_add_extra_chain_cert _cffi_d_SSL_CTX_add_extra_chain_cert
#endif

static int _cffi_d_SSL_CTX_add_server_custom_ext(SSL_CTX * x0, unsigned int x1, int(* x2)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *), void(* x3)(SSL *, unsigned int, unsigned char const *, void *), void * x4, int(* x5)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *), void * x6)
{
  return SSL_CTX_add_server_custom_ext(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_add_server_custom_ext(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned int x1;
  int(* x2)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *);
  void(* x3)(SSL *, unsigned int, unsigned char const *, void *);
  void * x4;
  int(* x5)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_add_server_custom_ext", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned int);
  if (x1 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x2 = (int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *))_cffi_to_c_pointer(arg2, _cffi_type(1444));
  if (x2 == (int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *))NULL && PyErr_Occurred())
    return NULL;

  x3 = (void(*)(SSL *, unsigned int, unsigned char const *, void *))_cffi_to_c_pointer(arg3, _cffi_type(1445));
  if (x3 == (void(*)(SSL *, unsigned int, unsigned char const *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = (int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *))_cffi_to_c_pointer(arg5, _cffi_type(1447));
  if (x5 == (int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_add_server_custom_ext(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_add_server_custom_ext _cffi_d_SSL_CTX_add_server_custom_ext
#endif

static int _cffi_d_SSL_CTX_check_private_key(SSL_CTX const * x0)
{
  return SSL_CTX_check_private_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_check_private_key(PyObject *self, PyObject *arg0)
{
  SSL_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_check_private_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_check_private_key _cffi_d_SSL_CTX_check_private_key
#endif

static unsigned long _cffi_d_SSL_CTX_clear_mode(SSL_CTX * x0, unsigned long x1)
{
  return SSL_CTX_clear_mode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_clear_mode(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_clear_mode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned long);
  if (x1 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_clear_mode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_clear_mode _cffi_d_SSL_CTX_clear_mode
#endif

static unsigned long _cffi_d_SSL_CTX_clear_options(SSL_CTX * x0, unsigned long x1)
{
  return SSL_CTX_clear_options(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_clear_options(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_clear_options", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned long);
  if (x1 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_clear_options(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_clear_options _cffi_d_SSL_CTX_clear_options
#endif

static void _cffi_d_SSL_CTX_free(SSL_CTX * x0)
{
  SSL_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_free(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_free _cffi_d_SSL_CTX_free
#endif

static X509_VERIFY_PARAM * _cffi_d_SSL_CTX_get0_param(SSL_CTX * x0)
{
  return SSL_CTX_get0_param(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get0_param(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_VERIFY_PARAM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get0_param(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1674));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get0_param _cffi_d_SSL_CTX_get0_param
#endif

static X509_STORE * _cffi_d_SSL_CTX_get_cert_store(SSL_CTX const * x0)
{
  return SSL_CTX_get_cert_store(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_cert_store(PyObject *self, PyObject *arg0)
{
  SSL_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_STORE * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_cert_store(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(123));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_cert_store _cffi_d_SSL_CTX_get_cert_store
#endif

static void * _cffi_d_SSL_CTX_get_ex_data(SSL_CTX const * x0, int x1)
{
  return SSL_CTX_get_ex_data(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_ex_data(PyObject *self, PyObject *args)
{
  SSL_CTX const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_get_ex_data", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_ex_data(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(136));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_ex_data _cffi_d_SSL_CTX_get_ex_data
#endif

static void(* _cffi_d_SSL_CTX_get_info_callback(SSL_CTX * x0))(SSL const *, int, int)
{
  return SSL_CTX_get_info_callback(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_info_callback(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void(* result)(SSL const *, int, int);
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_info_callback(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2317));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_info_callback _cffi_d_SSL_CTX_get_info_callback
#endif

static void(* _cffi_d_SSL_CTX_get_keylog_callback(SSL_CTX * x0))(SSL const *, char const *)
{
  return SSL_CTX_get_keylog_callback(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_keylog_callback(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void(* result)(SSL const *, char const *);
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_keylog_callback(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2313));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_keylog_callback _cffi_d_SSL_CTX_get_keylog_callback
#endif

static long _cffi_d_SSL_CTX_get_max_proto_version(SSL_CTX * x0)
{
  return SSL_CTX_get_max_proto_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_max_proto_version(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_max_proto_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_max_proto_version _cffi_d_SSL_CTX_get_max_proto_version
#endif

static long _cffi_d_SSL_CTX_get_min_proto_version(SSL_CTX * x0)
{
  return SSL_CTX_get_min_proto_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_min_proto_version(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_min_proto_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_min_proto_version _cffi_d_SSL_CTX_get_min_proto_version
#endif

static unsigned long _cffi_d_SSL_CTX_get_mode(SSL_CTX * x0)
{
  return SSL_CTX_get_mode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_mode(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_mode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_mode _cffi_d_SSL_CTX_get_mode
#endif

static unsigned long _cffi_d_SSL_CTX_get_options(SSL_CTX * x0)
{
  return SSL_CTX_get_options(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_options(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_options(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_options _cffi_d_SSL_CTX_get_options
#endif

static long _cffi_d_SSL_CTX_get_read_ahead(SSL_CTX * x0)
{
  return SSL_CTX_get_read_ahead(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_read_ahead(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_read_ahead(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_read_ahead _cffi_d_SSL_CTX_get_read_ahead
#endif

static unsigned long _cffi_d_SSL_CTX_get_session_cache_mode(SSL_CTX * x0)
{
  return SSL_CTX_get_session_cache_mode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_session_cache_mode(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_session_cache_mode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_session_cache_mode _cffi_d_SSL_CTX_get_session_cache_mode
#endif

static long _cffi_d_SSL_CTX_get_timeout(SSL_CTX const * x0)
{
  return SSL_CTX_get_timeout(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_timeout(PyObject *self, PyObject *arg0)
{
  SSL_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_timeout(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_timeout _cffi_d_SSL_CTX_get_timeout
#endif

static int _cffi_d_SSL_CTX_get_verify_depth(SSL_CTX const * x0)
{
  return SSL_CTX_get_verify_depth(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_verify_depth(PyObject *self, PyObject *arg0)
{
  SSL_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_verify_depth(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_verify_depth _cffi_d_SSL_CTX_get_verify_depth
#endif

static int _cffi_d_SSL_CTX_get_verify_mode(SSL_CTX const * x0)
{
  return SSL_CTX_get_verify_mode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_get_verify_mode(PyObject *self, PyObject *arg0)
{
  SSL_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(494), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(494), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_get_verify_mode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_get_verify_mode _cffi_d_SSL_CTX_get_verify_mode
#endif

static int _cffi_d_SSL_CTX_load_verify_locations(SSL_CTX * x0, char const * x1, char const * x2)
{
  return SSL_CTX_load_verify_locations(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_load_verify_locations(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  char const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_load_verify_locations", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_load_verify_locations(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_load_verify_locations _cffi_d_SSL_CTX_load_verify_locations
#endif

static SSL_CTX * _cffi_d_SSL_CTX_new(SSL_METHOD * x0)
{
  return SSL_CTX_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_new(PyObject *self, PyObject *arg0)
{
  SSL_METHOD * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  SSL_CTX * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(349), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(349), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(331));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_new _cffi_d_SSL_CTX_new
#endif

static long _cffi_d_SSL_CTX_sess_accept(SSL_CTX * x0)
{
  return SSL_CTX_sess_accept(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_accept(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_accept(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_accept _cffi_d_SSL_CTX_sess_accept
#endif

static long _cffi_d_SSL_CTX_sess_accept_good(SSL_CTX * x0)
{
  return SSL_CTX_sess_accept_good(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_accept_good(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_accept_good(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_accept_good _cffi_d_SSL_CTX_sess_accept_good
#endif

static long _cffi_d_SSL_CTX_sess_accept_renegotiate(SSL_CTX * x0)
{
  return SSL_CTX_sess_accept_renegotiate(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_accept_renegotiate(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_accept_renegotiate(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_accept_renegotiate _cffi_d_SSL_CTX_sess_accept_renegotiate
#endif

static long _cffi_d_SSL_CTX_sess_cache_full(SSL_CTX * x0)
{
  return SSL_CTX_sess_cache_full(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_cache_full(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_cache_full(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_cache_full _cffi_d_SSL_CTX_sess_cache_full
#endif

static long _cffi_d_SSL_CTX_sess_cb_hits(SSL_CTX * x0)
{
  return SSL_CTX_sess_cb_hits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_cb_hits(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_cb_hits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_cb_hits _cffi_d_SSL_CTX_sess_cb_hits
#endif

static long _cffi_d_SSL_CTX_sess_connect(SSL_CTX * x0)
{
  return SSL_CTX_sess_connect(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_connect(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_connect(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_connect _cffi_d_SSL_CTX_sess_connect
#endif

static long _cffi_d_SSL_CTX_sess_connect_good(SSL_CTX * x0)
{
  return SSL_CTX_sess_connect_good(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_connect_good(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_connect_good(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_connect_good _cffi_d_SSL_CTX_sess_connect_good
#endif

static long _cffi_d_SSL_CTX_sess_connect_renegotiate(SSL_CTX * x0)
{
  return SSL_CTX_sess_connect_renegotiate(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_connect_renegotiate(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_connect_renegotiate(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_connect_renegotiate _cffi_d_SSL_CTX_sess_connect_renegotiate
#endif

static long _cffi_d_SSL_CTX_sess_hits(SSL_CTX * x0)
{
  return SSL_CTX_sess_hits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_hits(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_hits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_hits _cffi_d_SSL_CTX_sess_hits
#endif

static long _cffi_d_SSL_CTX_sess_misses(SSL_CTX * x0)
{
  return SSL_CTX_sess_misses(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_misses(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_misses(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_misses _cffi_d_SSL_CTX_sess_misses
#endif

static long _cffi_d_SSL_CTX_sess_number(SSL_CTX * x0)
{
  return SSL_CTX_sess_number(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_number(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_number(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_number _cffi_d_SSL_CTX_sess_number
#endif

static long _cffi_d_SSL_CTX_sess_timeouts(SSL_CTX * x0)
{
  return SSL_CTX_sess_timeouts(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_sess_timeouts(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_sess_timeouts(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_sess_timeouts _cffi_d_SSL_CTX_sess_timeouts
#endif

static int _cffi_d_SSL_CTX_set_alpn_protos(SSL_CTX * x0, unsigned char const * x1, unsigned int x2)
{
  return SSL_CTX_set_alpn_protos(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_alpn_protos(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned char const * x1;
  unsigned int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_alpn_protos", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_alpn_protos(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_alpn_protos _cffi_d_SSL_CTX_set_alpn_protos
#endif

static void _cffi_d_SSL_CTX_set_alpn_select_cb(SSL_CTX * x0, int(* x1)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *), void * x2)
{
  SSL_CTX_set_alpn_select_cb(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_alpn_select_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *);
  void * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_alpn_select_cb", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *))_cffi_to_c_pointer(arg1, _cffi_type(2265));
  if (x1 == (int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_alpn_select_cb(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_alpn_select_cb _cffi_d_SSL_CTX_set_alpn_select_cb
#endif

static void _cffi_d_SSL_CTX_set_cert_cb(SSL_CTX * x0, int(* x1)(SSL *, void *), void * x2)
{
  SSL_CTX_set_cert_cb(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cert_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, void *);
  void * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cert_cb", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, void *))_cffi_to_c_pointer(arg1, _cffi_type(1894));
  if (x1 == (int(*)(SSL *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_cert_cb(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_cert_cb _cffi_d_SSL_CTX_set_cert_cb
#endif

static void _cffi_d_SSL_CTX_set_cert_store(SSL_CTX * x0, X509_STORE * x1)
{
  SSL_CTX_set_cert_store(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cert_store(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  X509_STORE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cert_store", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(123), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_STORE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(123), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_cert_store(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_cert_store _cffi_d_SSL_CTX_set_cert_store
#endif

static void _cffi_d_SSL_CTX_set_cert_verify_callback(SSL_CTX * x0, int(* x1)(X509_STORE_CTX *, void *), void * x2)
{
  SSL_CTX_set_cert_verify_callback(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cert_verify_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(X509_STORE_CTX *, void *);
  void * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cert_verify_callback", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(X509_STORE_CTX *, void *))_cffi_to_c_pointer(arg1, _cffi_type(2283));
  if (x1 == (int(*)(X509_STORE_CTX *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_cert_verify_callback(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_cert_verify_callback _cffi_d_SSL_CTX_set_cert_verify_callback
#endif

static int _cffi_d_SSL_CTX_set_cipher_list(SSL_CTX * x0, char const * x1)
{
  return SSL_CTX_set_cipher_list(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cipher_list(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cipher_list", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_cipher_list(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_cipher_list _cffi_d_SSL_CTX_set_cipher_list
#endif

static int _cffi_d_SSL_CTX_set_ciphersuites(SSL_CTX * x0, char const * x1)
{
  return SSL_CTX_set_ciphersuites(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_ciphersuites(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_ciphersuites", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_ciphersuites(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_ciphersuites _cffi_d_SSL_CTX_set_ciphersuites
#endif

static void _cffi_d_SSL_CTX_set_client_CA_list(SSL_CTX * x0, Cryptography_STACK_OF_X509_NAME * x1)
{
  SSL_CTX_set_client_CA_list(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_client_CA_list(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  Cryptography_STACK_OF_X509_NAME * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_client_CA_list", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(440), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(440), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_client_CA_list(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_client_CA_list _cffi_d_SSL_CTX_set_client_CA_list
#endif

static int _cffi_d_SSL_CTX_set_client_cert_engine(SSL_CTX * x0, ENGINE * x1)
{
  return SSL_CTX_set_client_cert_engine(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_client_cert_engine(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_client_cert_engine", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(219), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(219), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_client_cert_engine(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_client_cert_engine _cffi_d_SSL_CTX_set_client_cert_engine
#endif

static void _cffi_d_SSL_CTX_set_cookie_generate_cb(SSL_CTX * x0, int(* x1)(SSL *, unsigned char *, unsigned int *))
{
  SSL_CTX_set_cookie_generate_cb(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cookie_generate_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, unsigned char *, unsigned int *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cookie_generate_cb", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, unsigned char *, unsigned int *))_cffi_to_c_pointer(arg1, _cffi_type(2261));
  if (x1 == (int(*)(SSL *, unsigned char *, unsigned int *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_cookie_generate_cb(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_cookie_generate_cb _cffi_d_SSL_CTX_set_cookie_generate_cb
#endif

static void _cffi_d_SSL_CTX_set_cookie_verify_cb(SSL_CTX * x0, int(* x1)(SSL *, unsigned char const *, unsigned int))
{
  SSL_CTX_set_cookie_verify_cb(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_cookie_verify_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, unsigned char const *, unsigned int);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_cookie_verify_cb", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, unsigned char const *, unsigned int))_cffi_to_c_pointer(arg1, _cffi_type(2274));
  if (x1 == (int(*)(SSL *, unsigned char const *, unsigned int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_cookie_verify_cb(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_cookie_verify_cb _cffi_d_SSL_CTX_set_cookie_verify_cb
#endif

static void _cffi_d_SSL_CTX_set_default_passwd_cb(SSL_CTX * x0, int(* x1)(char *, int, int, void *))
{
  SSL_CTX_set_default_passwd_cb(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_default_passwd_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(char *, int, int, void *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_default_passwd_cb", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg1, _cffi_type(135));
  if (x1 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_default_passwd_cb(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_default_passwd_cb _cffi_d_SSL_CTX_set_default_passwd_cb
#endif

static void _cffi_d_SSL_CTX_set_default_passwd_cb_userdata(SSL_CTX * x0, void * x1)
{
  SSL_CTX_set_default_passwd_cb_userdata(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_default_passwd_cb_userdata(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  void * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_default_passwd_cb_userdata", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_default_passwd_cb_userdata(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_default_passwd_cb_userdata _cffi_d_SSL_CTX_set_default_passwd_cb_userdata
#endif

static int _cffi_d_SSL_CTX_set_default_verify_paths(SSL_CTX * x0)
{
  return SSL_CTX_set_default_verify_paths(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_default_verify_paths(PyObject *self, PyObject *arg0)
{
  SSL_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_default_verify_paths(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_default_verify_paths _cffi_d_SSL_CTX_set_default_verify_paths
#endif

static long _cffi_d_SSL_CTX_set_ecdh_auto(SSL_CTX * x0, int x1)
{
  return SSL_CTX_set_ecdh_auto(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_ecdh_auto(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_ecdh_auto", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_ecdh_auto(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_ecdh_auto _cffi_d_SSL_CTX_set_ecdh_auto
#endif

static void _cffi_d_SSL_CTX_set_info_callback(SSL_CTX * x0, void(* x1)(SSL const *, int, int))
{
  SSL_CTX_set_info_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_info_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  void(* x1)(SSL const *, int, int);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_info_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (void(*)(SSL const *, int, int))_cffi_to_c_pointer(arg1, _cffi_type(2317));
  if (x1 == (void(*)(SSL const *, int, int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_info_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_info_callback _cffi_d_SSL_CTX_set_info_callback
#endif

static void _cffi_d_SSL_CTX_set_keylog_callback(SSL_CTX * x0, void(* x1)(SSL const *, char const *))
{
  SSL_CTX_set_keylog_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_keylog_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  void(* x1)(SSL const *, char const *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_keylog_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (void(*)(SSL const *, char const *))_cffi_to_c_pointer(arg1, _cffi_type(2313));
  if (x1 == (void(*)(SSL const *, char const *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_keylog_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_keylog_callback _cffi_d_SSL_CTX_set_keylog_callback
#endif

static int _cffi_d_SSL_CTX_set_max_early_data(SSL_CTX * x0, uint32_t x1)
{
  return SSL_CTX_set_max_early_data(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_max_early_data(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  uint32_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_max_early_data", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, uint32_t);
  if (x1 == (uint32_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_max_early_data(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_max_early_data _cffi_d_SSL_CTX_set_max_early_data
#endif

static long _cffi_d_SSL_CTX_set_max_proto_version(SSL_CTX * x0, int x1)
{
  return SSL_CTX_set_max_proto_version(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_max_proto_version(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_max_proto_version", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_max_proto_version(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_max_proto_version _cffi_d_SSL_CTX_set_max_proto_version
#endif

static long _cffi_d_SSL_CTX_set_min_proto_version(SSL_CTX * x0, int x1)
{
  return SSL_CTX_set_min_proto_version(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_min_proto_version(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_min_proto_version", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_min_proto_version(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_min_proto_version _cffi_d_SSL_CTX_set_min_proto_version
#endif

static unsigned long _cffi_d_SSL_CTX_set_mode(SSL_CTX * x0, unsigned long x1)
{
  return SSL_CTX_set_mode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_mode(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_mode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned long);
  if (x1 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_mode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_mode _cffi_d_SSL_CTX_set_mode
#endif

static unsigned long _cffi_d_SSL_CTX_set_options(SSL_CTX * x0, unsigned long x1)
{
  return SSL_CTX_set_options(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_options(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_options", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned long);
  if (x1 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_options(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_options _cffi_d_SSL_CTX_set_options
#endif

static void _cffi_d_SSL_CTX_set_post_handshake_auth(SSL_CTX * x0, int x1)
{
  SSL_CTX_set_post_handshake_auth(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_post_handshake_auth(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_post_handshake_auth", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_post_handshake_auth(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_post_handshake_auth _cffi_d_SSL_CTX_set_post_handshake_auth
#endif

static void _cffi_d_SSL_CTX_set_psk_client_callback(SSL_CTX * x0, unsigned int(* x1)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))
{
  SSL_CTX_set_psk_client_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_psk_client_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned int(* x1)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_psk_client_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))_cffi_to_c_pointer(arg1, _cffi_type(2301));
  if (x1 == (unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_psk_client_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_psk_client_callback _cffi_d_SSL_CTX_set_psk_client_callback
#endif

static void _cffi_d_SSL_CTX_set_psk_find_session_callback(SSL_CTX * x0, int(* x1)(SSL *, unsigned char const *, size_t, SSL_SESSION * *))
{
  SSL_CTX_set_psk_find_session_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_psk_find_session_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, unsigned char const *, size_t, SSL_SESSION * *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_psk_find_session_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, unsigned char const *, size_t, SSL_SESSION * *))_cffi_to_c_pointer(arg1, _cffi_type(2270));
  if (x1 == (int(*)(SSL *, unsigned char const *, size_t, SSL_SESSION * *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_psk_find_session_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_psk_find_session_callback _cffi_d_SSL_CTX_set_psk_find_session_callback
#endif

static void _cffi_d_SSL_CTX_set_psk_server_callback(SSL_CTX * x0, unsigned int(* x1)(SSL *, char const *, unsigned char *, unsigned int))
{
  SSL_CTX_set_psk_server_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_psk_server_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned int(* x1)(SSL *, char const *, unsigned char *, unsigned int);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_psk_server_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int))_cffi_to_c_pointer(arg1, _cffi_type(2305));
  if (x1 == (unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_psk_server_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_psk_server_callback _cffi_d_SSL_CTX_set_psk_server_callback
#endif

static void _cffi_d_SSL_CTX_set_psk_use_session_callback(SSL_CTX * x0, int(* x1)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *))
{
  SSL_CTX_set_psk_use_session_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_psk_use_session_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_psk_use_session_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *))_cffi_to_c_pointer(arg1, _cffi_type(2253));
  if (x1 == (int(*)(SSL *, EVP_MD const *, unsigned char const * *, size_t *, SSL_SESSION * *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_psk_use_session_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_psk_use_session_callback _cffi_d_SSL_CTX_set_psk_use_session_callback
#endif

static long _cffi_d_SSL_CTX_set_read_ahead(SSL_CTX * x0, long x1)
{
  return SSL_CTX_set_read_ahead(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_read_ahead(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_read_ahead", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_read_ahead(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_read_ahead _cffi_d_SSL_CTX_set_read_ahead
#endif

static unsigned long _cffi_d_SSL_CTX_set_session_cache_mode(SSL_CTX * x0, unsigned long x1)
{
  return SSL_CTX_set_session_cache_mode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_session_cache_mode(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_session_cache_mode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned long);
  if (x1 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_session_cache_mode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_session_cache_mode _cffi_d_SSL_CTX_set_session_cache_mode
#endif

static int _cffi_d_SSL_CTX_set_session_id_context(SSL_CTX * x0, unsigned char const * x1, unsigned int x2)
{
  return SSL_CTX_set_session_id_context(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_session_id_context(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  unsigned char const * x1;
  unsigned int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_session_id_context", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_session_id_context(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_session_id_context _cffi_d_SSL_CTX_set_session_id_context
#endif

static long _cffi_d_SSL_CTX_set_timeout(SSL_CTX * x0, long x1)
{
  return SSL_CTX_set_timeout(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_timeout(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_timeout", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_timeout(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_timeout _cffi_d_SSL_CTX_set_timeout
#endif

static void _cffi_d_SSL_CTX_set_tlsext_servername_arg(SSL_CTX * x0, void * x1)
{
  SSL_CTX_set_tlsext_servername_arg(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tlsext_servername_arg(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  void * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tlsext_servername_arg", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_tlsext_servername_arg(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_tlsext_servername_arg _cffi_d_SSL_CTX_set_tlsext_servername_arg
#endif

static void _cffi_d_SSL_CTX_set_tlsext_servername_callback(SSL_CTX * x0, int(* x1)(SSL *, int *, void *))
{
  SSL_CTX_set_tlsext_servername_callback(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tlsext_servername_callback(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, int *, void *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tlsext_servername_callback", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, int *, void *))_cffi_to_c_pointer(arg1, _cffi_type(2257));
  if (x1 == (int(*)(SSL *, int *, void *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_tlsext_servername_callback(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_tlsext_servername_callback _cffi_d_SSL_CTX_set_tlsext_servername_callback
#endif

static long _cffi_d_SSL_CTX_set_tlsext_status_arg(SSL_CTX * x0, void * x1)
{
  return SSL_CTX_set_tlsext_status_arg(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tlsext_status_arg(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  void * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tlsext_status_arg", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(136), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(136), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_tlsext_status_arg(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_tlsext_status_arg _cffi_d_SSL_CTX_set_tlsext_status_arg
#endif

static long _cffi_d_SSL_CTX_set_tlsext_status_cb(SSL_CTX * x0, int(* x1)(SSL *, void *))
{
  return SSL_CTX_set_tlsext_status_cb(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tlsext_status_cb(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int(* x1)(SSL *, void *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tlsext_status_cb", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = (int(*)(SSL *, void *))_cffi_to_c_pointer(arg1, _cffi_type(1894));
  if (x1 == (int(*)(SSL *, void *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_tlsext_status_cb(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_tlsext_status_cb _cffi_d_SSL_CTX_set_tlsext_status_cb
#endif

static int _cffi_d_SSL_CTX_set_tlsext_use_srtp(SSL_CTX * x0, char const * x1)
{
  return SSL_CTX_set_tlsext_use_srtp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tlsext_use_srtp(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tlsext_use_srtp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_tlsext_use_srtp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_tlsext_use_srtp _cffi_d_SSL_CTX_set_tlsext_use_srtp
#endif

static unsigned long _cffi_d_SSL_CTX_set_tmp_dh(SSL_CTX * x0, DH * x1)
{
  return SSL_CTX_set_tmp_dh(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tmp_dh(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tmp_dh", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(139), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(139), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_tmp_dh(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_tmp_dh _cffi_d_SSL_CTX_set_tmp_dh
#endif

static unsigned long _cffi_d_SSL_CTX_set_tmp_ecdh(SSL_CTX * x0, EC_KEY * x1)
{
  return SSL_CTX_set_tmp_ecdh(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_tmp_ecdh(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  EC_KEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_tmp_ecdh", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(666), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(666), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_set_tmp_ecdh(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_set_tmp_ecdh _cffi_d_SSL_CTX_set_tmp_ecdh
#endif

static void _cffi_d_SSL_CTX_set_verify(SSL_CTX * x0, int x1, int(* x2)(int, X509_STORE_CTX *))
{
  SSL_CTX_set_verify(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_verify(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  int(* x2)(int, X509_STORE_CTX *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_verify", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = (int(*)(int, X509_STORE_CTX *))_cffi_to_c_pointer(arg2, _cffi_type(2218));
  if (x2 == (int(*)(int, X509_STORE_CTX *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_verify(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_verify _cffi_d_SSL_CTX_set_verify
#endif

static void _cffi_d_SSL_CTX_set_verify_depth(SSL_CTX * x0, int x1)
{
  SSL_CTX_set_verify_depth(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_set_verify_depth(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_set_verify_depth", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_CTX_set_verify_depth(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_CTX_set_verify_depth _cffi_d_SSL_CTX_set_verify_depth
#endif

static int _cffi_d_SSL_CTX_use_PrivateKey(SSL_CTX * x0, EVP_PKEY * x1)
{
  return SSL_CTX_use_PrivateKey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_PrivateKey(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_PrivateKey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(142), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(142), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_PrivateKey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_PrivateKey _cffi_d_SSL_CTX_use_PrivateKey
#endif

static int _cffi_d_SSL_CTX_use_PrivateKey_file(SSL_CTX * x0, char const * x1, int x2)
{
  return SSL_CTX_use_PrivateKey_file(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_PrivateKey_file(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_PrivateKey_file", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_PrivateKey_file(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_PrivateKey_file _cffi_d_SSL_CTX_use_PrivateKey_file
#endif

static int _cffi_d_SSL_CTX_use_certificate(SSL_CTX * x0, X509 * x1)
{
  return SSL_CTX_use_certificate(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_certificate(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_certificate", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(11), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(11), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_certificate(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_certificate _cffi_d_SSL_CTX_use_certificate
#endif

static int _cffi_d_SSL_CTX_use_certificate_chain_file(SSL_CTX * x0, char const * x1)
{
  return SSL_CTX_use_certificate_chain_file(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_certificate_chain_file(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_certificate_chain_file", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_certificate_chain_file(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_certificate_chain_file _cffi_d_SSL_CTX_use_certificate_chain_file
#endif

static int _cffi_d_SSL_CTX_use_certificate_file(SSL_CTX * x0, char const * x1, int x2)
{
  return SSL_CTX_use_certificate_file(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_certificate_file(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_certificate_file", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_certificate_file(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_certificate_file _cffi_d_SSL_CTX_use_certificate_file
#endif

static int _cffi_d_SSL_CTX_use_psk_identity_hint(SSL_CTX * x0, char const * x1)
{
  return SSL_CTX_use_psk_identity_hint(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_CTX_use_psk_identity_hint(PyObject *self, PyObject *args)
{
  SSL_CTX * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_CTX_use_psk_identity_hint", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(331), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(331), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(23), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(23), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_CTX_use_psk_identity_hint(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_CTX_use_psk_identity_hint _cffi_d_SSL_CTX_use_psk_identity_hint
#endif

static void _cffi_d_SSL_SESSION_free(SSL_SESSION * x0)
{
  SSL_SESSION_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_free(PyObject *self, PyObject *arg0)
{
  SSL_SESSION * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1298), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1298), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SSL_SESSION_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SSL_SESSION_free _cffi_d_SSL_SESSION_free
#endif

static unsigned char const * _cffi_d_SSL_SESSION_get_id(SSL_SESSION const * x0, unsigned int * x1)
{
  return SSL_SESSION_get_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_id(PyObject *self, PyObject *args)
{
  SSL_SESSION const * x0;
  unsigned int * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned char const * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SSL_SESSION_get_id", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1076), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1076), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(67));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_id _cffi_d_SSL_SESSION_get_id
#endif

static size_t _cffi_d_SSL_SESSION_get_master_key(SSL_SESSION const * x0, unsigned char * x1, size_t x2)
{
  return SSL_SESSION_get_master_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_master_key(PyObject *self, PyObject *args)
{
  SSL_SESSION const * x0;
  unsigned char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_SESSION_get_master_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(638), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(638), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_master_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_master_key _cffi_d_SSL_SESSION_get_master_key
#endif

static uint32_t _cffi_d_SSL_SESSION_get_max_early_data(SSL_SESSION const * x0)
{
  return SSL_SESSION_get_max_early_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_max_early_data(PyObject *self, PyObject *arg0)
{
  SSL_SESSION const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  uint32_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_max_early_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, uint32_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_max_early_data _cffi_d_SSL_SESSION_get_max_early_data
#endif

static long _cffi_d_SSL_SESSION_get_ticket_lifetime_hint(SSL_SESSION const * x0)
{
  return SSL_SESSION_get_ticket_lifetime_hint(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_ticket_lifetime_hint(PyObject *self, PyObject *arg0)
{
  SSL_SESSION const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_ticket_lifetime_hint(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_ticket_lifetime_hint _cffi_d_SSL_SESSION_get_ticket_lifetime_hint
#endif

static long _cffi_d_SSL_SESSION_get_time(SSL_SESSION const * x0)
{
  return SSL_SESSION_get_time(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_time(PyObject *self, PyObject *arg0)
{
  SSL_SESSION const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_time(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_time _cffi_d_SSL_SESSION_get_time
#endif

static long _cffi_d_SSL_SESSION_get_timeout(SSL_SESSION const * x0)
{
  return SSL_SESSION_get_timeout(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_get_timeout(PyObject *self, PyObject *arg0)
{
  SSL_SESSION const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_get_timeout(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_get_timeout _cffi_d_SSL_SESSION_get_timeout
#endif

static int _cffi_d_SSL_SESSION_has_ticket(SSL_SESSION const * x0)
{
  return SSL_SESSION_has_ticket(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_has_ticket(PyObject *self, PyObject *arg0)
{
  SSL_SESSION const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1472), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1472), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_has_ticket(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_has_ticket _cffi_d_SSL_SESSION_has_ticket
#endif

static int _cffi_d_SSL_SESSION_set1_id_context(SSL_SESSION * x0, unsigned char const * x1, unsigned int x2)
{
  return SSL_SESSION_set1_id_context(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_set1_id_context(PyObject *self, PyObject *args)
{
  SSL_SESSION * x0;
  unsigned char const * x1;
  unsigned int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_SESSION_set1_id_context", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1298), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL_SESSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1298), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(67), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(67), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SSL_SESSION_set1_id_context(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SSL_SESSION_set1_id_context _cffi_d_SSL_SESSION_set1_id_context
#endif

static int _cffi_d_SSL_SESSION_set1_master_key(SSL_SESSION * x0, unsigned char const * x1, size_t x2)
{
  return SSL_SESSION_set1_master_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SSL_SESSION_set1_master_key(PyObject *self, PyObject *args)
{
  SSL_SESSION * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SSL_SESSION_set1_master_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  data